/**
 *	Copyright (C) 2011 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.file;
/**
 * 
 */



import java.io.*;

import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.string.DateFormatter;

/**
 * This abstract class offers convenient file system operations.
 * 
 * @author denis
 *
 */
public abstract class FileUtil
{
	static private final String			TempFolder = "temp/FileUtil";
	
	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	--------		Copy				-------------------------------------------------------
	//	--------		(recursive)			-------------------------------------------------------

	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Replacing means that the destination folder will contain EXACTLY the same files like the source, no matter if the
	 * destination folder already existed and contained files or folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException 
	 * 
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 */
	public static File copyToFolderOverwriting(String fileName, String toFolderName) throws IOException
	{
		return copyToFolder(fileName, toFolderName, true);
	}

	
	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Replacing means that the destination folder will contain EXACTLY the same files like the source, no matter if the
	 * destination folder already existed and contained files or folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException 
	 * 
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 */
	public static File copyToFolderOverwriting(File sourceFile, File toFolder) throws IOException
	{
		return copyToFolderOverwriting(sourceFile.getPath(), toFolder.getPath());
	}
	

	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException 
	 * 
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 */
	public static File copyToFolderMerging(String fileName, String toFolderName) throws IOException
	{
		return copyToFolder(fileName, toFolderName, false);
	}

	
	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException 
	 * 
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 */
	public static File copyToFolderMerging(File sourceFile, File toFolder) throws IOException
	{
		return copyToFolderMerging(sourceFile.getPath(), toFolder.getPath());
	}
	

	//	--------		Copy				-------------------------------------------------------
	//	--------		(recursive)			-------------------------------------------------------

	/**
	 * This method copies single files and whole directory trees.
	 * Non-existing destination folders will be created. Any existing destination files or folders will be deleted.
	 * @param sourceFile file or folder to copy
	 * @param destFile file to copy to
	 * @throws IOException 
	 */
	public static File copyToOverwriting(File sourceFile, File destFile) throws IOException
	{
		return copyTo(sourceFile, destFile, true);
	}
	

	/**
	 * This method copies single files and whole directory trees.
	 * Non-existing destination folders will be created. Any existing destination files or folders will be deleted.
	 * @param sourceFileName Name of file or folder to copy
	 * @param destFileName Name of folder to copy to
	 * @throws IOException 
	 * 
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 */
	public static File copyToOverwriting(String sourceFileName, String destFileName) throws IOException
	{
		return copyToOverwriting(new File(sourceFileName), new File(destFileName));
	}


	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException 
	 * 
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 */
	public static File copyToMerging(File sourceFile, File toFolder) throws IOException
	{
		return copyTo(sourceFile, toFolder, false);
	}
	

	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * Merging means that the destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @throws IOException 
	 * 
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 */
	public static File copyToMerging(String fileName, String toFolderName) throws IOException
	{
		return copyToMerging(new File(fileName), new File(toFolderName));
	}

	
	//	--------		Delete				-------------------------------------------------------
	//	--------		(recursive)			-------------------------------------------------------

	/**
	 * This method deletes single files and whole directory trees.
	 * If the file or folder doesn't exist, ignore it silently.
	 * @param fileName Name of File or Directory to delete
	 */
	public static File delete(String fileName) throws IOException
	{
		return delete(new File(fileName));
	}
	

	/**
	 * This method deletes single files and whole directory trees:
	 * If the file or folder doesn't exist, ignore it silently.
	 * @param file File or Directory to delete
	 */
	public static File delete(File file) throws IOException
	{
		Logger.info("Deleting file: '" + file.getPath() + "'");

		if (!file.exists())			return null;

		if (file.isDirectory())		for (File f: file.listFiles())	delete(f);		//	Recursion!

		//	Delete the file, retry 20 times, wait 1/2 sec between each try. Give up after 20 tries:
		boolean success = false;
		for (int i = 20; i > 0; i--)
		{
			if (file.delete())
			{
				Logger.debug("Deleted file: '" + file.getPath() + "'");

				success = true;
				break;
			}

			Logger.debug("Retrying: " + i);

			//	Wait 1/2 sec and then retry:
			try { Thread.sleep(500); } catch (InterruptedException x){}
		}

		if (!success)		throw new IOException("Could not delete File '" + file.getPath() + "'");

		return file;
	}
	
	
	//	--------		Rename				-------------------------------------------------------
	
	/**
	 * This method moves or renames files and folders. Non-existing destination folders will be created.
	 * @param oldName Name of File or Directory to move or rename
	 * @param newName Name of File or Directory to move or rename to
	 * @throws IOException
	 */
	public static File renameTo(String oldName, String newName) throws IOException
	{
		return renameTo(new File(oldName), new File(newName));
	}

	
	/**
	 * This method moves or renames files and folders. Non-existing destination folders will be created.
	 * @param oldFile File or Directory to move or rename
	 * @param newFile File or Directory to move or rename to
	 * @throws IOException 
	 */
	public static File renameTo(File oldFile, File newFile) throws IOException
	{
		Logger.info("Renaming file: '" + oldFile.getPath() + "' to: '" + newFile.getPath() + "'");

		if (!oldFile.exists())				throw new IOException("Source File '" + oldFile.getPath() + "' doesn't exist.");

		newFile.getParentFile().mkdirs();

		//	Rename the file, retry 20 times, wait 1/2 sec between each try. Give up after 20 tries:
		boolean success = false;
		for (int i = 20; i > 0; i--)
		{
			if (oldFile.renameTo(newFile))
			{
				Logger.debug("Renamed file: '" + oldFile.getPath() + "' to: '" + newFile.getPath() + "'");

				success = true;
				break;
			}
			
			Logger.debug("Retrying: " + i);

			//	Wait 1/2 sec and then retry:
			try { Thread.sleep(500); } catch (InterruptedException x){}
		}

		if (!success)		throw new IOException("Could not rename file '" + oldFile.getPath() + "' to: '" + newFile.getPath() + "'");

		return newFile;
	}

	
	//	--------		Move				-------------------------------------------------------
	
	/**
	 * This method moves files and folders to another folder. Non-existing destination folders will be created.
	 * @param fileName Name of File or Directory to move or rename
	 * @param toFolderName Name of Directory to move to
	 * @throws IOException
	 */
	public static File moveToFolder(String fileName, String toFolderName) throws IOException
	{
		return moveToFolder(new File(fileName), new File(toFolderName));
	}

	
	/**
	 * This method moves files and folders to another folder. Non-existing destination folders will be created.
	 * @param sourceFile File or Directory to move
	 * @param toFolder Directory to move to
	 * @throws IOException 
	 */
	public static File moveToFolder(File sourceFile, File toFolder) throws IOException
	{
		return renameTo(sourceFile, new File(toFolder.getPath() + "/" + sourceFile.getName()));
	}

	
	//	--------		Create Folder		-------------------------------------------------------
	
	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be deleted.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	public static File createFolderOverwriting(String folderName, String inFolderName) throws IOException
	{
		return createFolderOverwriting(folderName, new File(inFolderName));
	}

	
	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be deleted.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	public static File createFolderOverwriting(String folderName, File inFolder) throws IOException
	{
		return createFolderOverwriting(inFolder.getPath() + "/" + folderName);
	}

	
	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be deleted.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	public static File createFolderOverwriting(String newFolderName) throws IOException
	{
		return createFolderOverwriting(new File(newFolderName));
	}

	
	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be deleted.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	public static File createFolderOverwriting(File newFolder) throws IOException
	{
		Logger.info("Creating folder (overwriting): '" + newFolder.getPath() + "'");

		if (newFolder.exists())		delete(newFolder);

		if (!newFolder.mkdirs())	throw new IOException("Could not create folder '" + newFolder.getPath() + "'");
		
		return newFolder;
	}
	

	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be left as it is.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	public static File createFolderMerging(String folderName, String inFolderName) throws IOException
	{
		return createFolderMerging(folderName, new File(inFolderName));
	}

	
	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be left as it is.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	public static File createFolderMerging(String folderName, File inFolder) throws IOException
	{
		return createFolderMerging(inFolder.getPath() + "/" + folderName);
	}

	
	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be left as it is.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	public static File createFolderMerging(String newFolderName) throws IOException
	{
		return createFolderMerging(new File(newFolderName));
	}

	
	/**
	 * This method creates a new folder in the target folder. If this folder already exists, it will be left as it is.
	 * @param folderName Name of Folder to create
	 * @param inFolderName Name of target folder
	 * @throws IOException
	 */
	public static File createFolderMerging(File newFolder) throws IOException
	{
		Logger.info("Creating folder (merging): '" + newFolder.getPath() + "'");

		if (!newFolder.exists())
		{
			if (!newFolder.mkdirs())		throw new IOException("Could not create folder '" + newFolder.getPath() + "'");
		}

		return newFolder;
	}

	
	//	--------		File Content		-------------------------------------------------------
		
	static public String getFileContentAsString(String filePath)
	{
		StringBuilder text = new StringBuilder();
		BufferedReader reader = null;
		try
		{
			reader = new BufferedReader(new FileReader(filePath));
			String line;
			do
			{
				line = reader.readLine();
				if (line == null)		break;		//	EOF reached

				text.append(line).append("\n");
			}
			while (line != null);
		}
		catch (FileNotFoundException e)
		{
			e.printStackTrace();
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		finally
		{
			try
			{
				if (reader != null)		reader.close();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
		}
		
		return text.toString();
	}

	
	//	========	Static Private			=======================================================

	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * @param fileName Name of file or folder to copy
	 * @param toFolderName Name of folder to copy to
	 * @param overwriting If true, the destination will contain EXACTLY the same files like the source. If false, the 
	 * destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @throws IOException 
	 * 
	 * NOTE: A temporary folder "temp" and a temporary zipfile with a unique name is generated in the application's home folder.
	 * The zipfile is deleted afterwards.
	 */
	private static File copyToFolder(String fileName, String toFolderName, boolean overwriting) throws IOException
	{
		Logger.info("Copying file: '" + fileName + "' to folder: '" + toFolderName + "'");

		String tempZipFileName = TempFolder + "/" + "CopyToFolder_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + ".zip";

		Zipper.zip(fileName, tempZipFileName);
		if (overwriting)		delete(toFolderName + "/" + new File(fileName).getName());
		Zipper.unzip(tempZipFileName, toFolderName);
		delete(tempZipFileName);
		return new File(toFolderName + "/" + new File(fileName).getName());
	}
	

	/**
	 * This method copies single files and whole directory trees. Non-existing destination folders will be created.
	 * @param sourceFile File or folder to copy
	 * @param destFile File or folder to copy to
	 * @param overwriting If true, the destination will contain EXACTLY the same files like the source. If false, the 
	 * destination will contain all files from the source PLUS some possibly already existing files and folders.
	 * @throws IOException 
	 */
	private static File copyTo(File sourceFile, File destFile, boolean overwriting) throws IOException
	{
		Logger.info("Copying file: '" + sourceFile.getPath() + "' to file: '" + destFile.getPath() + "'");
		
		String tempZipFileName = TempFolder + "/" + "CopyTo_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs) + ".zip";

		Zipper.zip(sourceFile.getPath(), tempZipFileName, destFile.getName());
		if (overwriting)		delete(destFile);		//	In case of a directory: don't merge but overwrite!
		Zipper.unzip(tempZipFileName, destFile.getParent());
		delete(tempZipFileName);
		return destFile;
	}
	
}
