/**
 *	Copyright (C) 2011 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 *	Copyright (C) 2011 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.util;

import java.awt.Desktop;
import java.io.*;
import java.util.Arrays;

import ch.docuteam.docutools.exceptions.FileIsNotADirectoryException;
import ch.docuteam.docutools.out.Logger;

/**
 * @author denis
 *
 */
public abstract class SystemProcess
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	static public final int		OK = 0;

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

//	/**
//	 * Just for testing
//	 */
//	static public void main(String[] args) throws java.lang.Exception
//	{
//		//	Mac:
//		openExternally("/Volumes/daten/Ablage/B  Docu/2_Entw/21_Produktentwicklung/507 Bewertungsdatenbank/Versionen/Current/temp/Freaky Wireless Technology.tiff");
//
//		//	Windows:
//		//	openExternally("//Skylla/daten/Ablage/B  Docu/2_Entw/21_Produktentwicklung/507 Bewertungsdatenbank/Versionen/Current/temp/Freaky Wireless Technology.tiff");
//	}
	
	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	static public int execute(String... args) throws FileNotFoundException, IOException, InterruptedException, FileIsNotADirectoryException
	{
		return execute((File)null, args);
	}


	static public int execute(String workingDirectoryPath, String[] args) throws FileNotFoundException, IOException, InterruptedException, FileIsNotADirectoryException
	{
		return execute(new File(workingDirectoryPath), args);
	}


	static public int execute(File workingDirectory, String... args) throws FileNotFoundException, IOException, InterruptedException, FileIsNotADirectoryException
	{
		int error = OK;

		if (args.length == 0)			throw new IndexOutOfBoundsException("No command to execute");

		Logger.debug("Executing command line: " + Arrays.toString(args));

		//	Create process:
		ProcessBuilder processBuilder = new ProcessBuilder(args);
		processBuilder.redirectErrorStream(true);

		if (workingDirectory != null)
		{
			if (!workingDirectory.exists())			throw new FileNotFoundException(workingDirectory.getPath());
			if (!workingDirectory.isDirectory())	throw new FileIsNotADirectoryException(workingDirectory.getPath());

			processBuilder.directory(workingDirectory);
		}
		Logger.debug("Working directory: " + processBuilder.directory());

		Process process = null;
		try
		{
			// Start process:
			process = processBuilder.start();
			StreamGobbler outputGobbler = new StreamGobbler(process.getInputStream());
			outputGobbler.start();

			// Wait for the process to finish and check the result:
			error = process.waitFor();
			if (error == 5)		error = OK;			//	Ignore error #5

			// wait for the outputGobbler:
			outputGobbler.join();
		}
		finally
		{
			// close all IO streams:
			try { if (process != null)		process.getInputStream().close(); }
			catch (IOException x){}

			try { if (process != null)		process.getOutputStream().close(); }
			catch (IOException x){}

			try { if (process != null)		process.getErrorStream().close(); }
			catch (IOException x){}
			
			//	Cleanup all objects thoroughly:
			if (process != null)			process.destroy();
			process = null;
		}

		return error;
	}


	/**
	 * Open files, directories, or URLs with an external program (text editor, file browser, internet browser etc.).
	 * @throws FileIsNotADirectoryException 
	 */
	static public void openExternally(String filePath) throws IOException, InterruptedException, SystemProcessException, FileIsNotADirectoryException
	{
		Logger.debug("Opening externally: " + filePath);

		if (OperatingSystem.isWindows())
		{
			//	OS = Windows:

			execute(new String[]{"cmd", "/c", "rundll32 url.dll,FileProtocolHandler \"" + filePath + "\""});
		}
		else
		{
			//	OS = anything else:

			File file = new File(filePath);

			if (Desktop.isDesktopSupported() && file.exists())
			{
				//	Only files can be opened using Desktop:
				Logger.debug("Opening using Desktop: " + filePath);
				
				Desktop.getDesktop().open(file);
			}
			else
			{
				//	Open the file or URL using an OS call:
				String[] cmd = new String[]{"open", filePath};
	
				Logger.debug("Executing command line: " + Arrays.toString(cmd));
	
				ProcessBuilder processBuilder = new ProcessBuilder(cmd);
				processBuilder.redirectErrorStream(true);
				Process process = processBuilder.start();
	
				// Wait for the process to finish and check the result:
				int error = process.waitFor();
				Logger.debug("" + error);
				if (error == 5)		error = OK;			//	Ignore error #5
				
				if (error != OK)		throw new SystemProcessException(error);
			}
		}
	}
	
	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	/**
	 * @author andreas
	 * 
	 */
	static public class StreamGobbler extends Thread
	{
		InputStream is;

		StreamGobbler(InputStream is)
		{
			this.is = is;
		}

		@Override
		public void run()
		{
			Logger.debug("Run");

			BufferedReader br = null;

			try
			{
				br = new BufferedReader(new InputStreamReader(is));

				String line = null;
				while ((line = br.readLine()) != null)		Logger.info(line);
			}
			catch (IOException x)
			{
				Logger.error(x.getMessage());
			}
			finally
			{
				try { if (br != null)	br.close(); }
				catch (IOException ex){}
			}
		}
	}

}
