package fedora.services.sipcreator.tasks;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.GridLayout;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.SwingWorker;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.translations.I18N;

import de.schlichtherle.truezip.zip.ZipEntry;
import de.schlichtherle.truezip.zip.ZipOutputStream;

import uk.gov.nationalarchives.droid.FileFormatHit;

import beowulf.gui.Utility;
import beowulf.util.StreamUtility;
import fedora.services.sipcreator.Constants;
import fedora.services.sipcreator.FileSystemEntry;
import fedora.services.sipcreator.SIPCreator;
import fedora.services.sipcreator.SelectableEntry;
import fedora.services.sipcreator.SelectableEntryNode;
import fedora.services.sipcreator.ZipFileEntry;
import fedora.services.sipcreator.acceptor.NoHiddenFileAcceptor;
import fedora.services.sipcreator.acceptor.SelectionAcceptor;
import fedora.services.sipcreator.metadata.Metadata;
import fedora.services.sipcreator.utility.CheckRenderer;

public class FileSelectTask extends JPanel implements Constants {

	private static final long serialVersionUID = 4051332249108427830L;

	public static final String METS_NS = "http://www.loc.gov/METS/";

	private String[] pid;

	private EventHandler eventHandler = new EventHandler();
	private OpenFolderAction openFolderAction;
	private File sourceFile;

	private SaveSIPAction saveSIPAction;
	private File targetFile;
	private String identificationErrors = "";

	private NoHiddenFileAcceptor acceptor = new NoHiddenFileAcceptor();

	// Data structures and UI components involved with the file browsing task
	private CheckRenderer fileSelectTreeRenderer;
	private DefaultTreeModel fileSelectTreeModel = new DefaultTreeModel(null);
	private JTree fileSelectTreeDisplay = new JTree(fileSelectTreeModel);

	private SIPCreator creator;
	private SmallPeskyMessageWindow waitWindow;

	public FileSelectTask(SIPCreator newCreator) {
		creator = newCreator;
		I18N.initialize(creator.getProperties().getProperty("sipcreator.language"), "ch.docuteam.sipcreator.translations.Translations");

		fileSelectTreeRenderer = new CheckRenderer(creator);

		openFolderAction = new OpenFolderAction();
		saveSIPAction = new SaveSIPAction();

		fileSelectTreeDisplay.setCellRenderer(fileSelectTreeRenderer);
		fileSelectTreeDisplay.addMouseListener(eventHandler);
		fileSelectTreeDisplay.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

		JPanel tempP1 = new JPanel(new BorderLayout());
		tempP1.add(new JPanel(), BorderLayout.CENTER);
		JPanel tempP2 = new JPanel(new GridLayout(1, 0));
		tempP2.add(new JButton(openFolderAction));
		tempP1.add(tempP2, BorderLayout.WEST);
		JPanel tempP3 = new JPanel(new GridLayout(1, 0));
		tempP3.add(new JButton(saveSIPAction));
		tempP1.add(tempP3, BorderLayout.EAST);

		setLayout(new BorderLayout(5, 5));
		setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
		add(tempP1, BorderLayout.NORTH);
		add(new JScrollPane(fileSelectTreeDisplay), BorderLayout.CENTER);
	}

	public void updateTree(SelectableEntry newRoot) {
		if (newRoot == null) {
			fileSelectTreeModel.setRoot(null);
		} else {
			fileSelectTreeModel.setRoot(new SelectableEntryNode(newRoot, null, acceptor, fileSelectTreeModel));
		}
	}

	public void refreshTree() {
		fileSelectTreeModel.nodeStructureChanged((TreeNode) fileSelectTreeModel.getRoot());
	}

	public SelectableEntry getRootEntry() {
		SelectableEntryNode rootNode = (SelectableEntryNode) fileSelectTreeModel.getRoot();
		return (rootNode == null ? null : rootNode.getEntry());
	}

	public OpenFolderAction getOpenFolderAction() {
		return openFolderAction;
	}

	public SaveSIPAction getSaveSIPAction() {
		return saveSIPAction;
	}

	/**
	 * @return the pid
	 */
	public String[] getPid() {
		return pid;
	}

	private class EventHandler extends MouseAdapter {

		@Override
		public void mouseClicked(MouseEvent me) {
			try {
				if (me.getClickCount() > 1)
					return;
				if (!isEnabled())
					return;

				int x = me.getX();
				int y = me.getY();

				TreePath path = fileSelectTreeDisplay.getPathForLocation(x, y);
				if (path == null)
					return;

				// The next two lines ensure that the user clicked on the
				// checkbox, not the
				// icon or the text. The "-2" is a hack, but seems to work
				Rectangle bounds = fileSelectTreeDisplay.getPathBounds(path);
				if (x > bounds.x + fileSelectTreeRenderer.getCheckBoxWidth() - 2)
					return;

				SelectableEntryNode node = (SelectableEntryNode) path.getLastPathComponent();
				int previousBaseSelection = node.getEntry().getSelectionLevel();
				boolean fullySelected = previousBaseSelection == SelectableEntry.FULLY_SELECTED;
				int selectionLevel = fullySelected ? SelectableEntry.UNSELECTED : SelectableEntry.FULLY_SELECTED;
				node.getEntry().setSelectionLevel(selectionLevel, acceptor);

				SelectableEntryNode parent = (SelectableEntryNode) node.getParent();
				if (parent != null) {
					parent.getEntry().setSelectionLevelFromChildren(acceptor);
				}
				fileSelectTreeModel.nodeChanged(node);
				creator.getMetadataEntryTask().refreshTree();
			} catch (Exception e) {
				Utility.showExceptionDialog(creator, e);
			}
		}

	}

	public class OpenFolderAction extends AbstractAction {

		private static final long serialVersionUID = 3763096349595678519L;

		private OpenFolderAction() {
			putValue(Action.NAME, I18N.translate("OpenFolder"));
			URL imgURL = creator.getURL(FOLDER_IMAGE_NAME);
			putValue(Action.SMALL_ICON, new ImageIcon(creator.getImage(imgURL)));
			putValue(Action.SHORT_DESCRIPTION, I18N.translate("OpenFolderTT"));
		}

		public void actionPerformed(ActionEvent ae) {
			JFileChooser fileChooser = creator.getFileChooser();
			fileChooser.setDialogType(JFileChooser.OPEN_DIALOG);
			fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
			fileChooser.setMultiSelectionEnabled(false);
			fileChooser.addChoosableFileFilter(new FolderFileFilter());

			int choice = fileChooser.showOpenDialog(creator);
			if (choice != JFileChooser.APPROVE_OPTION)
				return;

			sourceFile = fileChooser.getSelectedFile();

			if (fileSelectTreeModel.getRoot() != null) {
				choice = JOptionPane.showConfirmDialog(creator, I18N.translate("ConfirmOverwritingMetadata"), I18N.translate("Warning"), JOptionPane.YES_NO_OPTION);
				if (choice != JOptionPane.YES_OPTION)
					return;
			}

			if (!sourceFile.isDirectory()) {
				JOptionPane.showMessageDialog(creator, I18N.translate("MessageFolderRequired"));
				return;
			}

			try {
				creator.setFileLabelText(sourceFile.getCanonicalPath());
			} catch (IOException e) {
				e.printStackTrace();
				JOptionPane.showMessageDialog(creator, e.toString(), I18N.translate("ExCouldNotGetFolderLabel"), JOptionPane.ERROR_MESSAGE);
				return;
			}

			new SwingWorker<Integer, Object>() {
				@Override
				public Integer doInBackground() {
					waitWindow = SmallPeskyMessageWindow.open(creator, I18N.translate("MessageReadFolderStructure") + ", " + I18N.translate("PleaseWait"));
					creator.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
					openDirectory(sourceFile);
					waitWindow.close();
					creator.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));

					return 0;
				}
			}.execute();
		}

		public void openDirectory(File file) {
			SelectableEntry rootEntry;

			if (file.getName().equalsIgnoreCase("METS.xml")) {
				JOptionPane.showMessageDialog(creator, "MessageNameNotAllowedAsRoot");
				return;
			}

			rootEntry = new FileSystemEntry(file, null, creator);
			rootEntry.setSelectionLevel(SelectableEntry.UNSELECTED, acceptor);

			creator.getMetadataEntryTask().closeAllTabs();
			creator.getMetadataEntryTask().updateTree(rootEntry);
			updateTree(rootEntry);
			System.gc();
		}

		public class FolderFileFilter extends javax.swing.filechooser.FileFilter {
			/*
			 * (non-Javadoc)
			 * 
			 * @see javax.swing.filechooser.FileFilter#accept(java.io.File)
			 */
			@Override
			public boolean accept(File f) {
				return (f.isDirectory());
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see javax.swing.filechooser.FileFilter#getDescription()
			 */
			@Override
			public String getDescription() {
				return I18N.translate("Folders");
			}
		}

	}

	public class SaveSIPAction extends AbstractAction {

		private static final int BUFFER_SIZE = 4096;

		private static final String HEADER = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" + "<METS:mets xmlns:METS=\"http://www.loc.gov/METS/\" xmlns:DC=\"http://purl.org/dc/elements/1.1/\" "
				+ "xmlns:EAD=\"urn:isbn:1-931666-22-9\" xmlns:PREMIS=\"info:lc/xmlns/premis-v2\" "
				+ "xmlns:xlink=\"http://www.w3.org/1999/xlink\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" "
				+ "xsi:schemaLocation=\"http://www.loc.gov/METS/ http://www.loc.gov/standards/mets/mets.xsd " + "info:lc/xmlns/premis-v2 http://www.loc.gov/premis/premis.xsd "
				+ "http://purl.org/dc/elements/1.1/ http://dublincore.org/schemas/xmls/qdc/dc.xsd " + "urn:isbn:1-931666-22-9 http://www.loc.gov/ead/ead.xsd\">\n";

		private static final String FOOTER = "</METS:mets>";

		private SelectionAcceptor acceptor = new SelectionAcceptor(FileSystemEntry.FULLY_SELECTED | FileSystemEntry.PARTIALLY_SELECTED);

		private StringBuffer amdMapBuffer;

		private SaveSIPAction() {
			I18N.initialize(creator.getProperties().getProperty("sipcreator.language"), "ch.docuteam.sipcreator.translations.Translations");
			putValue(Action.NAME, I18N.translate("SaveSIP"));
			URL imgURL = creator.getURL(SAVE_IMAGE_NAME);
			putValue(Action.SMALL_ICON, new ImageIcon(creator.getImage(imgURL)));
			putValue(Action.SHORT_DESCRIPTION, I18N.translate("SaveSIPTT"));
		}

		public void actionPerformed(ActionEvent ae) {
			// Make sure a submission agreement is selected. This also prevents a user without permission for any submission agreement from submitting stuff.
			if (creator.getSubmissionAgreementTask().getSaType()==null) {
				JOptionPane.showMessageDialog(creator, I18N.translate("NoSASelected"), "SIP-Creator", JOptionPane.INFORMATION_MESSAGE);
				return;
			}

			// Make sure at least one folder/file is selected
			if (creator.getFileSelectTask().getRootEntry().getSelectionLevel()==fedora.services.sipcreator.SelectableEntry.UNSELECTED) {
				JOptionPane.showMessageDialog(creator, I18N.translate("NoItemSelected"), "SIP-Creator", JOptionPane.INFORMATION_MESSAGE);
				return;
			}

			targetFile = null;
			switch (Integer.valueOf(creator.getProperties().getProperty("sipcreator.default.storage.type"))) {
			case 1:
				// Speicherung an vorgegebenem Ort
				targetFile = new File(creator.getProperties().getProperty("sipcreator.default.storage.location") + System.getProperty("file.separator")
						+ creator.getSubmissionAgreementTask().getSaType() + "_" + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + ".zip");
				try {
					if (targetFile.createNewFile()) {
						break;
					}
				} catch (IOException e) {
					JOptionPane.showMessageDialog(creator, I18N.translate("ExCouldNotSaveSIP"), "SIP-Creator", JOptionPane.ERROR_MESSAGE);
				}
			case 2:
			default:
				// Speicherung an frei wählbarem Ort im Dateisystem
				JFileChooser fileChooser = creator.getFileChooser();
				File storageLocation = new File(creator.getProperties().getProperty("sipcreator.default.storage.location"));
				if (!storageLocation.exists()) {
					storageLocation = new File(System.getProperty("user.home"));
				}
				fileChooser.setCurrentDirectory(storageLocation);
				fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
				int choice = fileChooser.showOpenDialog(creator);
				if (choice != JFileChooser.APPROVE_OPTION)
					return;

				targetFile = new File(fileChooser.getSelectedFile().getAbsoluteFile() + File.separator + creator.getSubmissionAgreementTask().getSaType() + "_"
						+ new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + ".zip");
				Logger.info(I18N.translate("Saving") + ": " + targetFile.getAbsolutePath());
				if (targetFile.exists()) {
					choice = JOptionPane.showConfirmDialog(creator, I18N.translate("ConfirmOverwriteFile"));
					if (choice != JOptionPane.YES_OPTION)
						return;
				}
				break;
			case 3:
				// Direkte Speicherung in ein Fedora-System
				// TODO Objekterstellung in Fedora
				JOptionPane.showMessageDialog(creator, I18N.translate("MessageDirectSaveNotYetImplemented"), "SIP-Creator", JOptionPane.INFORMATION_MESSAGE);
				break;
			}

			saveFile();
		}

		public void saveFile() {

			new SwingWorker<Integer, Object>() {
				@Override
				public Integer doInBackground() {
					waitWindow = SmallPeskyMessageWindow.open(creator, I18N.translate("Saving") + " 'METS.xml', " + I18N.translate("PleaseWait"));
					try {
						creator.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

						boolean savingSameFile = false;
						SelectableEntry root = creator.getFileSelectTask().getRootEntry();
						if (root instanceof ZipFileEntry && targetFile.exists()) {
							// Die Klasse ZipFile des Pakets
							// org.apache.tools.zip.* hat im
							// Gegensatz zu java.util.zip.* keine Methode
							// getName()
							/*
							 * String sourceName = ((ZipFileEntry)
							 * root).getSourceFile().getName(); if
							 * (sourceName.equals(file.getAbsolutePath())) {
							 * savingSameFile = true; }
							 */
							savingSameFile = true;
						}

						if (savingSameFile) {
							targetFile = File.createTempFile("zip", ".tmp");
						}

						Logger.info("Initializing ZIP...");
						ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(targetFile));
//						zos.setEncoding("UTF-8");
						
						creator.getMetadataEntryTask().updateMetadata();

						Logger.info("Creating METS header...");
						StringBuffer headerBuffer = getMetsHeader();
						amdMapBuffer = new StringBuffer("<METS:amdSec ID=\"" + Metadata.getNextID() + "\">");
						StringBuffer fileMapBuffer = new StringBuffer("<METS:fileSec><METS:fileGrp>");
						StringBuffer structMapBuffer = new StringBuffer("<METS:structMap>");
						walkTree(zos, amdMapBuffer, fileMapBuffer, structMapBuffer, "", root);
						amdMapBuffer.append("</METS:amdSec>");
						structMapBuffer.append("</METS:structMap>");
						fileMapBuffer.append("</METS:fileGrp></METS:fileSec>");

						StringBuffer xmlBuffer = new StringBuffer(HEADER);
						xmlBuffer.insert(xmlBuffer.length() - 2, " LABEL=\"" + creator.getSubmissionAgreementTask().getSaLabel() + "\"");
						xmlBuffer.insert(xmlBuffer.length() - 2, " TYPE=\"" + creator.getSubmissionAgreementTask().getSaType() + "\"");
						xmlBuffer.insert(xmlBuffer.length() - 2, " PROFILE=\"http://www.docuteam.ch/xmlns/sip-profile.xml\"");
						xmlBuffer.append(headerBuffer);
						xmlBuffer.append(amdMapBuffer);
						xmlBuffer.append(fileMapBuffer);
						xmlBuffer.append(structMapBuffer);
						xmlBuffer.append(FOOTER);

						File mets = new File(System.getProperty("user.home") + System.getProperty("file.separator") + "mets.xml");
						OutputStreamWriter osw = new OutputStreamWriter(new FileOutputStream(mets), "UTF-8");
						osw.write(xmlBuffer.toString());
						osw.close();
						handleFile(zos, "mets.xml", new FileInputStream(mets));
						mets.delete();

						// Keine Ausgabe der crules.xml
						/*
						 * StringReader xmlReader = new
						 * StringReader(xmlBuffer.toString()); int byteRead;
						 * ZipEntry entry; entry = new ZipEntry("crules.xml");
						 * entry.setTime(System.currentTimeMillis());
						 * zos.putNextEntry(entry); xmlReader = new
						 * StringReader(creator.getConversionRulesTask()
						 * .getRules().toXML()); while ((byteRead =
						 * xmlReader.read()) != -1) { zos.write(byteRead); }
						 * zos.closeEntry();
						 */

						zos.close();

						if (savingSameFile) {
							((ZipFileEntry) root).getSourceFile().close();
							try {
								JFileChooser fileChooser = creator.getFileChooser();
								fileChooser.getSelectedFile().delete();
								targetFile.renameTo(fileChooser.getSelectedFile());
							} catch (Exception e) {
								Utility.showExceptionDialog(creator, e);
							}
						}
					} catch (IOException e) {
						e.printStackTrace();
						JOptionPane.showMessageDialog(creator, e.toString(), I18N.translate("ExCouldNotSaveFile"), JOptionPane.ERROR_MESSAGE);
						return 1;
					} finally {
						waitWindow.close();
						creator.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
					}

					JOptionPane.showMessageDialog(creator, I18N.translate("MessageSaveSuccess"));

					// send mail
					if (Boolean.parseBoolean(creator.getProperties().getProperty("mail.send"))) {
						String subject = creator.getSubmissionAgreementTask().getSaType();
						String body = I18N.translate("NewSubmission") + " \"" + targetFile.getName() + "\" " + I18N.translate("by") + creator.getSubmissionAgreementTask().getUserFullname();
						if (!creator.getSubmissionAgreementTask().getUserNote().equals("")) {
							body += "\n\n" + I18N.translate("Note") + ": " + creator.getSubmissionAgreementTask().getUserNote();
						}
						if (identificationErrors.length() > 0)
							body += "\n\n" + I18N.translate("MessageFormatDetectionNoSuccess") + ":\n" + identificationErrors;

						ch.docuteam.docutools.out.MailSender.sendMessageSMTP(subject, body, creator.getProperties());
					}

					return 0;
				}
			}.execute();
		}

		/**
		 * @author Andreas Nef, Docuteam GmbH
		 * @return A String containing the METS-Header with dynamically set
		 *         attributes
		 */
		private StringBuffer getMetsHeader() {
			StringBuffer header = new StringBuffer("");
			header.append("<METS:metsHdr CREATEDATE=\"" + new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss").format(Calendar.getInstance().getTime()) + "\" RECORDSTATUS=\"Submitted\">\n");
			header.append("<METS:agent ROLE=\"CREATOR\" TYPE=\"INDIVIDUAL\">\n");
			header.append("<METS:name>" + creator.getSubmissionAgreementTask().getUserFullname() + "</METS:name>\n");
			header.append("<METS:note>" + creator.getSubmissionAgreementTask().getUserNote() + "</METS:note>\n");
			header.append("</METS:agent>\n</METS:metsHdr>\n");
			return header;
		}

		/**
		 * @param zos
		 * @param amdMap
		 * @param fileMap
		 * @param structMap
		 * @param name
		 * @param entry
		 * @throws IOException
		 */
		private void walkTree(ZipOutputStream zos, StringBuffer amdMap, StringBuffer fileMap, StringBuffer structMap, String name, SelectableEntry entry) throws IOException {
			waitWindow.setText(I18N.translate("Saving") + " '" + entry.getSafeShortName() + "', " + I18N.translate("PleaseWait"));

			name += entry.getSafeShortName();
			handleFile(zos, name, entry.isDirectory() ? null : entry.getStream());
			if (!entry.isDirectory()) {
				FileSystemEntry fse = (FileSystemEntry) entry;
				entry.setChecksum(creator.calculateChecksum(fse.getFile().getCanonicalPath()));
				// Instead of calculating the MimeType already when listing the
				// directory, we do it only if the file has been selected, A.
				// Nef, 28.1.2010
				entry.setMimeType(creator.getMimeType(fse.getFile()));
				handleFileData(fileMap, name, entry);
				handleFileStructure(structMap, entry);
				return;
			}

			handleDirectoryData(fileMap, entry);
			startDirectoryStructure(structMap, entry);

			name += File.separator;
			int childCount = entry.getChildCount(acceptor);
			for (int ctr = 0; ctr < childCount; ctr++) {
				walkTree(zos, amdMap, fileMap, structMap, name, entry.getChildAt(ctr, acceptor));
			}

			endDirectoryStructure(structMap);
		}

		private void handleFileData(StringBuffer buffer, String name, SelectableEntry entry) {
			buffer.append("<METS:file ID=\"");
			buffer.append(StreamUtility.enc(entry.getID()));
			buffer.append("\" MIMETYPE=\"");
			buffer.append(StreamUtility.enc(entry.getMimeType()));
			buffer.append("\" CHECKSUMTYPE=\"MD5\" CHECKSUM=\"");
			buffer.append(StreamUtility.enc(entry.getChecksum()));
			buffer.append("\">");
			buffer.append("<METS:FLocat LOCTYPE=\"URL\" xlink:href=\"file:///");
			buffer.append(StreamUtility.enc(name.replaceAll("\\\\", "/")));
			buffer.append("\"/>");
			buffer.append("</METS:file>");

			for (int ctr = 0; ctr < entry.getMetadataCount(); ctr++) {
				Metadata metadata = entry.getMetadata(ctr);
				buffer.append("<METS:file ID=\"");
				buffer.append(StreamUtility.enc(metadata.getID()));
				buffer.append("\" MIMETYPE=\"text/xml\">");
				buffer.append("<METS:FContent USE=\"");
				buffer.append(StreamUtility.enc(metadata.getClass().getName()));
				buffer.append("\"><METS:xmlData>");
				buffer.append(metadata.getAsXML());
				buffer.append("</METS:xmlData></METS:FContent>");
				buffer.append("</METS:file>");
			}
		}

		private void handleFileStructure(StringBuffer buffer, SelectableEntry entry) {
			buffer.append("<METS:div LABEL=\"");
			buffer.append(StreamUtility.enc(entry.getSafeShortName()));
			buffer.append("\" TYPE=\"file");
			buffer.append("\" ADMID=\"");
			buffer.append(createPremis(entry));
			buffer.append("\">");

			buffer.append("<METS:div LABEL=\"Content\" TYPE=\"content\">");
			buffer.append("<METS:fptr FILEID=\"");
			buffer.append(StreamUtility.enc(entry.getID()));
			buffer.append("\"/>");
			buffer.append("</METS:div>");

			for (int ctr = 0; ctr < entry.getMetadataCount(); ctr++) {
				Metadata metadata = entry.getMetadata(ctr);

				buffer.append("<METS:div LABEL=\"");
				buffer.append(StreamUtility.enc(metadata.getLabel()));
				buffer.append("\" TYPE=\"");
				buffer.append(StreamUtility.enc(metadata.getType()));
				buffer.append("\">");

				buffer.append("<METS:fptr FILEID=\"");
				buffer.append(StreamUtility.enc(metadata.getID()));
				buffer.append("\"/>");

				buffer.append("</METS:div>");
			}

			buffer.append("</METS:div>");
		}

		private void handleDirectoryData(StringBuffer buffer, SelectableEntry entry) {
			if (entry.getMetadataCount() == 0)
				return;

			for (int ctr = 0; ctr < entry.getMetadataCount(); ctr++) {
				Metadata metadata = entry.getMetadata(ctr);

				buffer.append("<METS:file ID=\"");
				buffer.append(StreamUtility.enc(metadata.getID()));
				buffer.append("\" MIMETYPE=\"text/xml\">");
				buffer.append("<METS:FContent USE=\"");
				buffer.append(StreamUtility.enc(metadata.getClass().getName()));
				buffer.append("\"><METS:xmlData>");
				buffer.append(metadata.getAsXML());
				buffer.append("</METS:xmlData></METS:FContent>");
				buffer.append("</METS:file>");
			}
		}

		private void startDirectoryStructure(StringBuffer buffer, SelectableEntry entry) {
			buffer.append("<METS:div LABEL=\"");
			buffer.append(StreamUtility.enc(entry.getSafeShortName()));
			buffer.append("\" TYPE=\"");
			if (entry.getParent() == null) {
				buffer.append("rootfolder");
			} else {
				buffer.append("folder");
			}
			buffer.append("\" ADMID=\"");
			buffer.append(createPremis(entry));
			buffer.append("\">");

			for (int ctr = 0; ctr < entry.getMetadataCount(); ctr++) {
				Metadata metadata = entry.getMetadata(ctr);

				buffer.append("<METS:div LABEL=\"");
				buffer.append(StreamUtility.enc(metadata.getLabel()));
				buffer.append("\" TYPE=\"");
				buffer.append(StreamUtility.enc(metadata.getType()));
				buffer.append("\">");

				buffer.append("<METS:fptr FILEID=\"");
				buffer.append(StreamUtility.enc(metadata.getID()));
				buffer.append("\"/>");

				buffer.append("</METS:div>");
			}
		}

		private void endDirectoryStructure(StringBuffer buffer) {
			buffer.append("</METS:div>");
		}

		private void handleFile(ZipOutputStream zos, String name, InputStream stream) throws IOException {
			Logger.info("Processing " + name);
			if (stream == null) {
				ZipEntry entry = new ZipEntry(name.replace("\\", "/") + "/");
				// entry.setTime(file.lastModified());
				// no need to setCRC, or setSize as they are computed
				// automatically.
				zos.putNextEntry((ZipEntry) entry);
				zos.closeEntry();
				return;
			}

			ZipEntry entry = new ZipEntry(name.replace("\\", "/"));
			// entry.setTime(file.lastModified());
			// no need to setCRC, or setSize as they are computed automatically.

			zos.putNextEntry((ZipEntry) entry);
			byte[] buffer = new byte[BUFFER_SIZE];
			while (stream.available() > 0) {
				int bytesRead = stream.read(buffer, 0, BUFFER_SIZE);
				if (bytesRead == -1) {
					break;
				}
				zos.write(buffer, 0, bytesRead);
			}

			stream.close();
			zos.closeEntry();
		}

		private String createPremis(SelectableEntry entry) {
			String id = Metadata.getNextID();
			amdMapBuffer.append("<METS:digiprovMD ID=\"" + id + "\">\n");
			amdMapBuffer.append("<METS:mdWrap MDTYPE=\"PREMIS\">\n<METS:xmlData>\n<PREMIS:premis version=\"2.0\">");
			String objectID = "";
			if (entry.getChildCount(new NoHiddenFileAcceptor()) > 0) {
				objectID = createPremisFolderObject(entry);
			} else {
				objectID = createPremisFileObject((FileSystemEntry) entry);
			}
			if (!entry.getSafeShortName().equals(entry.getShortName())) {
				amdMapBuffer.append(createPremisEventRename(entry, objectID));
			}
			amdMapBuffer.append("</PREMIS:premis>\n</METS:xmlData>\n</METS:mdWrap>\n</METS:digiprovMD>");
			return id;
		}

		private String createPremisFileObject(FileSystemEntry fsentry) {
			String id = Metadata.getNextID();
			String formatName = null;
			String formatVersion = null;
			String puid = null;

			try {
				FileFormatHit hit = fsentry.getPRONOM();
				formatName = hit.getFileFormatName();
				formatVersion = hit.getFileFormatVersion();
				puid = hit.getFileFormatPUID();
				Logger.info("Identified as " + puid);
			} catch (Exception e) {
				identificationErrors += "\n" + fsentry.getShortName();
			}

			amdMapBuffer.append("<PREMIS:object xsi:type=\"PREMIS:file\">\n<PREMIS:objectIdentifier>\n<PREMIS:objectIdentifierType>Docuteam</PREMIS:objectIdentifierType>");
			amdMapBuffer.append("<PREMIS:objectIdentifierValue>" + id + "</PREMIS:objectIdentifierValue>\n</PREMIS:objectIdentifier>");
			amdMapBuffer.append("<PREMIS:objectCharacteristics>\n<PREMIS:compositionLevel>0</PREMIS:compositionLevel>\n");
			amdMapBuffer.append("<PREMIS:fixity><PREMIS:messageDigestAlgorithm>MD5</PREMIS:messageDigestAlgorithm>\n<PREMIS:messageDigest>" + fsentry.getChecksum()
					+ "</PREMIS:messageDigest>\n</PREMIS:fixity>\n");
			amdMapBuffer.append("<PREMIS:size>" + fsentry.getFile().length() + "</PREMIS:size>\n");
			amdMapBuffer.append("<PREMIS:format><PREMIS:formatDesignation><PREMIS:formatName>" + formatName + "</PREMIS:formatName>" + "<PREMIS:formatVersion>" + formatVersion
					+ "</PREMIS:formatVersion></PREMIS:formatDesignation>" + "<PREMIS:formatRegistry><PREMIS:formatRegistryName>PRONOM</PREMIS:formatRegistryName><PREMIS:formatRegistryKey>" + puid
					+ "</PREMIS:formatRegistryKey></PREMIS:formatRegistry></PREMIS:format>\n</PREMIS:objectCharacteristics>\n");
			amdMapBuffer.append("<PREMIS:originalName xlink:type=\"simple\">" + protectSpecialCharacters(fsentry.getShortName()) + "</PREMIS:originalName>\n");
			amdMapBuffer.append("</PREMIS:object>\n");
			return id;
		}

		private String createPremisFolderObject(SelectableEntry entry) {
			String id = Metadata.getNextID();
			amdMapBuffer.append("<PREMIS:object xsi:type=\"PREMIS:representation\">\n<PREMIS:objectIdentifier>\n<PREMIS:objectIdentifierType>Docuteam</PREMIS:objectIdentifierType>");
			amdMapBuffer.append("<PREMIS:objectIdentifierValue>" + id + "</PREMIS:objectIdentifierValue>\n</PREMIS:objectIdentifier>");
			if (!entry.getSafeShortName().equals(entry.getShortName())) {
				amdMapBuffer.append("<PREMIS:originalName xlink:type=\"simple\">" + protectSpecialCharacters(entry.getShortName()) + "</PREMIS:originalName>\n");
			}
			amdMapBuffer.append("</PREMIS:object>\n");
			return id;
		}

		private String createPremisEventRename(SelectableEntry entry, String objectID) {
			StringBuffer buffer = new StringBuffer();
			buffer.append("<PREMIS:event>\n<PREMIS:eventIdentifier>\n<PREMIS:eventIdentifierType>Docuteam</PREMIS:eventIdentifierType>");
			buffer.append("<PREMIS:eventIdentifierValue>" + Metadata.getNextID() + "</PREMIS:eventIdentifierValue>\n</PREMIS:eventIdentifier>");
			buffer.append("<PREMIS:eventType>Renaming</PREMIS:eventType>");
			buffer.append("<PREMIS:eventDateTime>" + new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss").format(Calendar.getInstance().getTime()) + "</PREMIS:eventDateTime>\n");
			buffer.append("<PREMIS:eventDetail>Replace special characters in file and folder names</PREMIS:eventDetail>\n");
			buffer.append("<PREMIS:eventOutcomeInformation><PREMIS:eventOutcome>Success</PREMIS:eventOutcome></PREMIS:eventOutcomeInformation>\n");
			buffer.append("<PREMIS:linkingObjectIdentifier xlink:type=\"simple\">" + "<PREMIS:linkingObjectIdentifierType>Docuteam</PREMIS:linkingObjectIdentifierType>"
					+ "<PREMIS:linkingObjectIdentifierValue>" + objectID + "</PREMIS:linkingObjectIdentifierValue></PREMIS:linkingObjectIdentifier>");
			buffer.append("</PREMIS:event>\n");
			return buffer.toString();
		}

	}

	public static String protectSpecialCharacters(String originalUnprotectedString) {
		if (originalUnprotectedString == null) {
			return null;
		}
		boolean anyCharactersProtected = false;

		StringBuffer stringBuffer = new StringBuffer();
		for (int i = 0; i < originalUnprotectedString.length(); i++) {
			char ch = originalUnprotectedString.charAt(i);

			boolean controlCharacter = ch < 32;
			boolean unicodeButNotAscii = ch > 126;
			boolean characterWithSpecialMeaningInXML = ch == '<' || ch == '&' || ch == '>';

			if (characterWithSpecialMeaningInXML || unicodeButNotAscii || controlCharacter) {
				stringBuffer.append("&#" + (int) ch + ";");
				anyCharactersProtected = true;
			} else {
				stringBuffer.append(ch);
			}
		}
		if (anyCharactersProtected == false) {
			return originalUnprotectedString;
		}

		return stringBuffer.toString();
	}

	@SuppressWarnings("serial")
	private static class SmallPeskyMessageWindow extends JFrame {
		JLabel label = new JLabel("", JLabel.CENTER);

		private SmallPeskyMessageWindow(Component parent, String message) {
			this.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
			this.setAlwaysOnTop(true);
			this.setUndecorated(true);
			this.label.setText(message);
			this.add(label);
			this.setSize(500, 100);
			this.setLocationRelativeTo(parent);
		}

		static private SmallPeskyMessageWindow open(Component parent, String message) {
			SmallPeskyMessageWindow w = new SmallPeskyMessageWindow(parent, message);
			w.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			w.setVisible(true);
			return w;
		}

		private void setText(String message) {
			this.label.setText(message);
		}

		private void close() {
			this.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			this.setVisible(false);
			this.dispose();
		}
	}

}
