/**
 *	Copyright (C) 2011 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.file;
/**
 * 
 */



import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import ch.docuteam.docutools.out.Logger;
import de.schlichtherle.truezip.zip.ZipEntry;
import de.schlichtherle.truezip.zip.ZipFile;
import de.schlichtherle.truezip.zip.ZipOutputStream;

/**
 * This abstract class zips and unzips files and directories.
 * 
 * @author denis
 */
public abstract class Zipper
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Private			=======================================================

	static private final int		BUFFER_SIZE = 2048;

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	--------		Zipping				-------------------------------------------------------

	/**
	 * This single-argument method zips whatever is in the source path (a single file or directory name) and creates a .zip-file with the same name as the source file (with '.zip' appended).
	 * In case of a directory, all contained files and directories are zipped too.
	 * @return The created zip-file
	 */
	static public File zip(String sourcePath) throws IOException
	{
		return zip(sourcePath, sourcePath + ".zip");
	}


	/**
	 * This two-argument method zips whatever is in the source path (a single file or directory name) and creates the .zip-file as specified in the second parameter (with '.zip' appended if necessary).
	 * In case of a directory, all contained files and directories are zipped too.
	 * In case the boolean parameter 'skipTopFolder' is true AND the top element is a directory, zipping excludes the top folder and zips only the content of it.
	 * @return The created zip-file
	 */
	static public File zip(String sourcePath, String zipFilePath, boolean skipTopFolder) throws IOException
	{
		if (!skipTopFolder)			return zip(sourcePath, zipFilePath);
		
		//	Does source file exist?:
		File sourceFile = new File(sourcePath);
		if (!sourceFile.exists())
		{
			Logger.error("Source file for zipping doesn't exist:" + sourcePath);
			throw new FileNotFoundException(sourcePath);
		}

		if (sourceFile.isFile())	return zip(sourcePath, zipFilePath);
		
		//	Construct the list of file names within the sourcePath (which is a folder):
		List<String> subFileNames = new ArrayList<String>();
		for (String fileName: sourceFile.list())		subFileNames.add(sourcePath + "/" + fileName);
		
		return zip(subFileNames.toArray(new String[]{}), zipFilePath);
	}
	

	/**
	 * This two-argument method zips whatever is in the source path (a single file or directory name) and creates the .zip-file as specified in the second parameter (with '.zip' appended if necessary).
	 * In case of a directory, all contained files and directories are zipped too.
	 * @return The created zip-file
	 */
	static public File zip(String sourcePath, String zipFilePath) throws IOException
	{
		Logger.info("Zipping: " + sourcePath + " Into:" + zipFilePath);

		//	Does source file exist?:
		File sourceFile = new File(sourcePath);
		if (!sourceFile.exists())
		{
			Logger.error("Source file for zipping doesn't exist:" + sourcePath);
			throw new FileNotFoundException(sourcePath);
		}

		String zipRootDirectory = sourceFile.getParent();
		
		//	Add ".zip" to destination file name if necessary:
		if (!(zipFilePath.endsWith(".zip") || zipFilePath.endsWith(".ZIP")))		zipFilePath += ".zip";

		//	Create destination directories if necessary:
		new File(zipFilePath).getParentFile().mkdirs();

		//	Zip it:
		ZipOutputStream zipOutStream = new ZipOutputStream(new FileOutputStream(zipFilePath));
		try
		{
			zip(zipRootDirectory, sourceFile, zipOutStream);
		}
		finally
		{
			zipOutStream.close();
		}

		return new File(zipFilePath);
	}
	

	/**
	 * This three-argument method zips whatever is in the source path (a single file or a directory) and creates the .zip-file as specified in the second argument (with '.zip' appended if necessary),
	 * but INSIDE the zip-file, the root file or directory is renamed as specified in the 3rd parameter.
	 * When extracting this zip-file, the resulting file or directory then has this name.
	 * In case of a directory, all contained files and directories are zipped too.
	 * @return The created zip-file
	 */
	static public File zip(String sourcePath, String zipFilePath, String newName) throws IOException
	{
		Logger.info("Zipping: " + sourcePath + " Into:" + zipFilePath + " Renaming to:" + newName);

		//	Does source file exist?:
		File sourceFile = new File(sourcePath);
		if (!sourceFile.exists())
		{
			Logger.error("Source file for zipping doesn't exist:" + sourcePath);
			throw new FileNotFoundException(sourcePath);
		}

		//	Rename source File, zip it, then rename it back:
		String newSourcePath = sourceFile.getParent() + "/" + newName;
		File newSourceFile = new File(newSourcePath);

		try
		{
			if (!sourceFile.renameTo(newSourceFile))		throw new IOException("Could not rename File '" + sourcePath + "' to: '" + newSourcePath + "'");
			return zip(newSourcePath, zipFilePath);
		}
		finally
		{
			if (!newSourceFile.renameTo(sourceFile))		throw new IOException("Could not rename File '" + newSourcePath + "' back to: '" + sourcePath + "'");
		}
	}


	/**
	 * This multi-argument method zips whatever is in the source paths (an array of file or directory names) and creates the .zip-file as specified in the second parameter (with '.zip' appended if necessary).
	 * IMPORTANT NOTE: All source paths must be located within the same common directory!
	 * In case of a directory, all contained files and directories are zipped too.
	 * @return The created zip-file
	 */
	static public File zip(String[] sourcePaths, String zipFilePath) throws IOException
	{
		Logger.info("Zipping: " + Arrays.toString(sourcePaths) + " Into:" + zipFilePath);

		if (sourcePaths.length == 0)
		{
			Logger.info("Nothing to zip found");
			return null;
		}

		String zipRootDirectory = new File(sourcePaths[0]).getParent();
		
		//	Add ".zip" to destination file name if necessary:
		if (!(zipFilePath.endsWith(".zip") || zipFilePath.endsWith(".ZIP")))		zipFilePath += ".zip";

		//	Create destination directories if necessary:
		new File(zipFilePath).getParentFile().mkdirs();

		//	Zip it:
		ZipOutputStream zipOutStream = new ZipOutputStream(new FileOutputStream(zipFilePath));
		try
		{
			for (String sourcePath: sourcePaths)
			{
				File sourceFile = new File(sourcePath);
				if (sourceFile.exists())
				{
					zip(zipRootDirectory, new File(sourcePath), zipOutStream);
				}
				else
				{
					Logger.error("Source file for zipping doesn't exist:" + sourcePath);
					throw new FileNotFoundException(sourcePath);
				}
			}
		}
		finally
		{
			zipOutStream.close();
		}

		return new File(zipFilePath);
	}
	

	//	--------		Unzipping			-------------------------------------------------------

	/**
	 * Unzip the zipFile to the same folder where the zipFile is.
	 */
	static public void unzip(String zipFilePath) throws IOException
	{
		String destPath = ".";

		Integer lastIndexOfFileSeparator = zipFilePath.lastIndexOf("/");
		if (lastIndexOfFileSeparator != -1)		destPath = zipFilePath.substring(0, lastIndexOfFileSeparator);

		unzip(zipFilePath, destPath);
	}


	/**
	 * Unzip the zipFile to the destination folder. Non-existing folders are generated if necessary.
	 */
	static public void unzip(String zipFilePath, String destPath) throws IOException
	{
		Logger.info("Unzipping: " + zipFilePath + " Into:" + destPath);

		//	Does source file exist?:
		if (!new File(zipFilePath).exists())
		{
			Logger.error("Source file for unzipping doesn't exist:" + zipFilePath);
			throw new FileNotFoundException(zipFilePath);
		}

		ZipFile zipFile = new ZipFile(zipFilePath);
		try
		{
			for (ZipEntry entry: Collections.list(zipFile.entries()))
			{
				unzipEntry(zipFile, entry, destPath);
			}
		}
		finally
		{
			zipFile.close();
		}
	}
	

	//	========	Static Private			=======================================================

	//	--------		Zipping				-------------------------------------------------------

	static private void zip(String zipRootDir, File sourceFile, ZipOutputStream zipOutStream) throws IOException
	{
		if (sourceFile.isDirectory())
		{
			zipDirectory(zipRootDir, sourceFile, zipOutStream);
		}
		else
		{
			zipFile(zipRootDir, sourceFile, zipOutStream);
		}
	}
	
	
	static private void zipDirectory(String zipRootDir, File sourceDir, ZipOutputStream zipOutStream) throws IOException
	{
		Logger.debug("Zipping Folder: " + sourceDir.getPath());

		//	Empty folder have to be created explicitly (those others are generated automatically):
		if (sourceDir.listFiles().length == 0)
		{
			//	... but remove the zipRootDir from the filePath beforehand:
			//	NOTE: I have to use "File.separator" here because both file names are in the platform-dependent form
			if (zipRootDir != null)
				zipOutStream.putNextEntry(new ZipEntry(sourceDir.getPath().replace(zipRootDir + File.separator, "") + "/"));
			else
				zipOutStream.putNextEntry(new ZipEntry(sourceDir.getPath() + "/"));
		}

		//	Now recurse into the directory:
		for (File file: sourceDir.listFiles())		zip(zipRootDir, file, zipOutStream);
	}


	static private void zipFile(String zipRootDir, File sourceFile, ZipOutputStream zipOutputStream) throws IOException
	{
		Logger.debug("Zipping File:   " + sourceFile.getPath());

		//	First create the file entry in the zipOutputStream...
		//	... but remove the zipRootDir beforehand:
		//	NOTE: I have to use "File.separator" here because both file names are in the platform-dependent form
		if (zipRootDir != null)
			zipOutputStream.putNextEntry(new ZipEntry(sourceFile.getPath().replace(zipRootDir + File.separator, "")));
		else
			zipOutputStream.putNextEntry(new ZipEntry(sourceFile.getPath()));

		//	... then fill the entry with the file content:
		BufferedInputStream fileInputStream = new BufferedInputStream(new FileInputStream(sourceFile), BUFFER_SIZE);

		try
		{
			byte data[] = new byte[BUFFER_SIZE];
			for (int i; (i = fileInputStream.read(data, 0, BUFFER_SIZE)) > 0;)		zipOutputStream.write(data, 0, i);
		}
		finally
		{
			zipOutputStream.closeEntry();
			fileInputStream.close();
		}
	}

	
	//	--------		Unzipping			-------------------------------------------------------
	
	static private void unzipEntry(ZipFile zipFile, ZipEntry zipEntry, String destDir) throws IOException
	{
		File file = new File(destDir, zipEntry.getName());

		if (zipEntry.isDirectory())
		{
			Logger.debug("Unzipping Folder: " + zipEntry.getName());
			file.mkdirs();
		}
		else
		{
			Logger.debug("Unzipping File:   " + zipEntry.getName());

			new File(file.getParent()).mkdirs();

			InputStream zipEntryInputStream = zipFile.getInputStream(zipEntry);
			OutputStream fileOutputStream = new FileOutputStream(file);

			try
			{
				byte data[] = new byte[BUFFER_SIZE];
				for (int i; (i = zipEntryInputStream.read(data)) != -1;)		fileOutputStream.write(data, 0, i);
			}
			finally
			{
				fileOutputStream.close();
				zipEntryInputStream.close();
			}
		}
	}

}
