/**
 *	Copyright (C) 2011 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.out;



import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.xml.DOMConfigurator;

/**
 * This abstract class offers a convenient interface to Log4J.
 * It handles
 * <ol>
 * <li>The initialization of the Log4J system</li>
 * <li>Class-individual loggers</li>
 * </ol>
 * 
 * <h3>Initialization</h3>
 * When this class is used for the 1st time, it initializes the Log4J System. Every subsequent usage does NOT initialize the Log4J System again.
 * 
 * <h3>Class-Individual Loggers</h3>
 * When a class uses this class for logging, the appropriate logger is stored internally and reused on every subsequent log from that class.
 * <p>
 * NOTE: If a config file other than ./config/log4j.xml is used, set the config file with Logger.setConfigFile(String);
 * 
 * @author denis
 */
public abstract class Logger
{
	//	=======================================================================
	//	==========	Structure			=======================================
	//	=======================================================================
	
	//	==========	Public Static		=======================================

	//	==========	Private Static		=======================================

	/**
	 * The default log4j configuration file location:
	 */
	private static final String								DefaultConfigFile = "config/log4j.xml";

	private static URL										ConfigFile;

	private static Map<String, org.apache.log4j.Logger>		Loggers = new HashMap<String, org.apache.log4j.Logger>();
	private static Boolean									IsInitialized = false;
	private static Boolean									CouldNotInitialize = false;


	//	=======================================================================
	//	==========	Methods				=======================================
	//	=======================================================================

	//	==========	Public Static		=======================================
	
	public static void setConfigFile(String newConfigFile)
	{
		try {
			ConfigFile = new File(newConfigFile).toURI().toURL();
		} catch (MalformedURLException e) {
			e.printStackTrace();
			Tracer.error("Logger could not be initialized properly.");
		}

		//	(Re-)initialize with new ConfigFile:
		IsInitialized = false;
		initializeIfNecessary();
	}
	
	
	public static void setConfigFile(URL newConfigFile)
	{
		ConfigFile = newConfigFile;

		//	(Re-)initialize with new ConfigFile:
		IsInitialized = false;
		initializeIfNecessary();
	}
	
	
	public static void trace()
	{
		if (CouldNotInitialize)		return;

		getLogger().trace("");
	}


	public static void trace(Object message)
	{
		if (CouldNotInitialize)		return;

		getLogger().trace(message);
	}


	public static void trace(Object message, Throwable t)
	{
		if (CouldNotInitialize)		return;

		getLogger().trace(message, t);
	}


	public static void debug()
	{
		if (CouldNotInitialize)		return;

		getLogger().debug("");
	}


	public static void debug(Object message)
	{
		if (CouldNotInitialize)		return;

		getLogger().debug(message);
	}


	public static void debug(Object message, Throwable t)
	{
		if (CouldNotInitialize)		return;

		getLogger().debug(message, t);
	}


	public static void info()
	{
		if (CouldNotInitialize)		return;

		getLogger().info("");
	}


	public static void info(Object message)
	{
		if (CouldNotInitialize)		return;

		getLogger().info(message);
	}

	
	public static void info(Object message, Throwable t)
	{
		if (CouldNotInitialize)		return;

		getLogger().info(message, t);
	}


	public static void warn(Object message)
	{
		if (CouldNotInitialize)		return;

		getLogger().warn(message);
	}


	public static void warn()
	{
		if (CouldNotInitialize)		return;

		getLogger().warn("");
	}


	public static void warn(Object message, Throwable t)
	{
		if (CouldNotInitialize)		return;

		getLogger().warn(message, t);
	}


	public static void error(Object message)
	{
		if (CouldNotInitialize)		return;

		getLogger().error(message);
	}


	public static void error()
	{
		if (CouldNotInitialize)		return;

		getLogger().error("");
	}


	public static void error(Object message, Throwable t)
	{
		if (CouldNotInitialize)		return;

		getLogger().error(message, t);
	}


	public static void fatal()
	{
		if (CouldNotInitialize)		return;

		getLogger().fatal("");
	}


	public static void fatal(Object message)
	{
		if (CouldNotInitialize)		return;

		getLogger().fatal(message);
	}


	public static void fatal(Object message, Throwable t)
	{
		if (CouldNotInitialize)		return;

		getLogger().fatal(message, t);
	}


	//	==========	Private Static		=======================================

	private static org.apache.log4j.Logger getLogger()
	{
		initializeIfNecessary();

		String callerClassName = Thread.currentThread().getStackTrace()[3].getClassName();
		if (!Loggers.containsKey(callerClassName))		Loggers.put(callerClassName, org.apache.log4j.Logger.getLogger(callerClassName));

		return Loggers.get(callerClassName);
	}


	private static void initializeIfNecessary()
	{
		if (IsInitialized)		return;

		try
		{
			//	Tell commons-logging to use log4j:
			System.setProperty("org.apache.commons.logging.LogFactory", "org.apache.commons.logging.impl.Log4jFactory");
			System.setProperty("org.apache.commons.logging.Log", "org.apache.commons.logging.impl.Log4JLogger");

			//	Initialize log4j:
			if (ConfigFile==null)		ConfigFile = new File(DefaultConfigFile).toURI().toURL();
			DOMConfigurator.configure(ConfigFile);

			IsInitialized = true;
		}
		catch (java.lang.Exception ex)
		{
			ex.printStackTrace();
			Tracer.error("Logger could not be initialized properly.");

			CouldNotInitialize = true;
		}
	}

}

