/**
 *	Copyright (C) 2011 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.out;
import ch.docuteam.docutools.string.DateFormatter;




/**
 * This abstract class replaces that tedious "System.out.println(...)" by the simpler and better "Tracer.trace(...)".
 * In addition, the tracer writes to the console the exact time, class and method of the issuer. And the tracer can be switched on and off.
 * @author denis
 */
public abstract class Tracer
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	The dateFormat used to format the timestamp.
	static private final String				DateFormatString = "HH:mm:ss.SSS";

	//	String prefix to be placed before the output string.
	static private final String				Prefix = "";

	//	The trace switch.
	static private Boolean					DoTrace = true;


	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	/**
	 * Switch the tracer on. Any "Tracer.trace(...)" will be ignored silently, but the Tracer.error(...) will be executed.
	 */
	static public void switchOn() { DoTrace = true; }

	/**
	 * Switch the tracer on.
	 */
	static public void switchOff() { DoTrace = false; }

	/**
	 * 
	 * @return if the tracer is switched on or off.
	 */
	static public Boolean isOn() { return DoTrace; }


	/**
	 * Trace the empty string. This is still useful because you see the time, class and method.
	 */
	static public void trace() { doTrace(null); }

	/**
	 * Trace the toString of the object passed.
	 */
	static public void trace(Object o) { doTrace(o); };

	/**
	 * Trace the empty string to "System.err".
	 * Note: This gets traced even when the tracer is switched off.
	 */
	static public void error() { doError(null); }

	/**
	 * Trace the toString of the object passed to "System.err".
	 * Note: This gets traced even when the tracer is switched off.
	 */
	static public void error(Object o) { doError(o); };


	//	========	Static Private			=======================================================

	/**
	 * Actually do the tracing.
	 */
	static private void doTrace(Object o)
	{
		if (DoTrace)	System.out.println(constructMessage(o));
	}


	/**
	 * Actually do the tracing on error level.
	 */
	static private void doError(Object o)
	{
		System.err.println(constructMessage(o));
	}


	/**
	 * Create the trace string out of the object to be traced. That means adding a prefix, the time, the class name, and the method name.
	 * @param o The object to be written out
	 * @return The string to be written out
	 */
	static private String constructMessage(Object o)
	{
		//	The "4" is a "Magic Number". I have to go 4 steps back (or: down) on the call stack to get to the original issuer.
		StackTraceElement caller = Thread.currentThread().getStackTrace()[4];
		String className = caller.getClassName();

		return(
			new StringBuilder()
				.append(Prefix)
				.append(DateFormatter.getCurrentDateTimeString(DateFormatString))
				.append(":")
				.append(className.substring(className.lastIndexOf(".") + 1))
				.append(".")
				.append(caller.getMethodName())
				.append("()[")
				.append(caller.getLineNumber())
				.append((o == null) || (o.equals(""))? "]": "]:" + o)
				.toString());
	}

}
