/**
 *	Copyright (C) 2011 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.docutools.util;
/**
 * 
 */


import java.util.Date;

import ch.docuteam.docutools.string.DateFormatter;

/**
 * An instance of this class represents an exception. These exceptions are not thrown but are collected in the class <a href="./ExceptionCollector.html">ExceptionCollector</a>.<p>
 * The public interface is kept as tiny as possible. When an instance of Exception is created, it is automatically added to the ExceptionCollector.
 * @author denis
 *
 */
public class Exception
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	/**
	 * The Date format used in toString()
	 */
	static final private String			DateFormatString = "HH:mm:ss.SSS";

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	/**
	 * The timestamp when this exception occurred.
	 */
	private Date			timeStamp = new Date();
	
	/**
	 * The class name where this exception occurred.
	 */
	private String			className;

	/**
	 * The method name where this exception occurred.
	 */
	private String			methodName;

	/**
	 * The message of this exception. When an exception is created out of a Throwable, this message is set to this Throwable's message.
	 */
	private String			message;

	/**
	 * Optional: the original exception.
	 */
	private Throwable		exception;				//	Optional, might be null
	
	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	//	========	Constructors Private	=======================================================

	/**
	 * Constructor without an original exception but only with a message string.
	 * In this case, the class name and method name are taken from the place where this instance was created.
	 */
	protected Exception(String message)
	{
		this.message = message;

		StackTraceElement caller = Thread.currentThread().getStackTrace()[3];
		this.className = caller.getClassName();
		this.methodName = caller.getMethodName();
	}


	/**
	 * Constructor with an original exception.
	 * In this case, the class name and method name are taken from the place where this original exception was thrown.
	 * The message is taken from the original exception.
	 */
	protected Exception(Throwable x)
	{
		this.exception = x;
		this.message = x.toString();

		StackTraceElement caller = x.getStackTrace()[0];
		this.className = caller.getClassName();
		this.methodName = caller.getMethodName();
	}


	//	========	Static Public			=======================================================
	//				Factories:
	
	/**
	 * Create a new instance of Exception and add it to the ExceptionCollector.
	 */
	static public void remember(String message)
	{
		Exception x = new Exception(message);
//		System.err.println(x.toStringLong());

		ExceptionCollector.add(x);
	}
	

	/**
	 * Create a new instance of Exception and add it to the ExceptionCollector.
	 */
	static public void remember(Throwable x)
	{
//		x.printStackTrace();

		ExceptionCollector.add(new Exception(x));
	}
	
	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public Date getTimeStamp()
	{
		return timeStamp;
	}


	public String getClassName()
	{
		return className;
	}


	public String getMethodName()
	{
		return methodName;
	}


	public String getMessage()
	{
		return message;
	}


	public Throwable getException()
	{
		return exception;
	}


	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------

	@Override
	public String toString()
	{
		return message;
	}


	public String toStringLong()
	{
		return new StringBuilder("[Exception:")
			.append(DateFormatter.getDateTimeString(this.timeStamp, DateFormatString))
			.append(":")
			.append(this.className)
			.append(".")
			.append(this.methodName)
			.append("():")
			.append(this.message)
			.append((this.exception == null)? "": "(" + this.exception.getClass().getSimpleName() + ")")
			.append("]")
			.toString();
	}

	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------
	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

}
