/**
 *  � The National Archives 2005-2008.  All rights reserved.
 * See Licence.txt for full licence details.
 * <p/>
 *
 *  DROID DCS Profile Tool
 * <p/>
 * <p/>
 * Developed By:
 * Tessella Support Services
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * <p/>
 * email:  info@tessella.com
 * web:    www.tessella.com
 * <p/>
 * <p/>
 * Created Date:   1-Nov-2008
 */
package uk.gov.nationalarchives.droid.profile;

import uk.gov.nationalarchives.droid.profile.domain.FileObject;
import uk.gov.nationalarchives.droid.profile.domain.Format;
import uk.gov.nationalarchives.droid.profile.domain.ProfileVolume;
import uk.gov.nationalarchives.droid.profile.domain.VolumeSummary;
import uk.gov.nationalarchives.droid.profile.service.ProfilingManager;
import org.apache.commons.io.DirectoryWalker;
import uk.gov.nationalarchives.droid.AnalysisController;
import uk.gov.nationalarchives.droid.Droid;
import uk.gov.nationalarchives.droid.FileFormatHit;
import uk.gov.nationalarchives.droid.IdentificationFile;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.Callable;
import org.apache.commons.logging.*;
import uk.gov.nationalarchives.droid.MessageDisplay;



/**
 * This is the main walker that traverses a directory, and saves profiling information to the database as it goes.
 */
public class FileFormatWalker extends DirectoryWalker implements Callable<Boolean> {

    

    private File startDirectory;
    private ProfileVolume profileVolume;
    private volatile boolean cancelled = false;
    private volatile boolean paused = false;
    private Droid droid;
    private AnalysisController analysisControl;
    private ProfilingManager manager;
    private long wait = 0L;
    private int waitFreq = 0;
    private int counter;

    private VolumeSummary summary;
    private Log log = LogFactory.getLog(this.getClass());
    /**
     * Creates a walker - used by command line implementation
     *
     * @param manager       a profiling manager
     * @param profileVolume the volume
     * @param signatureFile a droid signature file location
     */
    public FileFormatWalker(ProfilingManager manager, ProfileVolume profileVolume, String signatureFile) {
        super();
        this.profileVolume = profileVolume;
        this.startDirectory = new File(profileVolume.getVolume());
        try {
            droid = new Droid();
            droid.readSignatureFile(signatureFile);
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());

        }
        this.manager = manager;
        this.wait = 0L;
        this.waitFreq = 0;
        this.counter = 0;
        this.summary = new VolumeSummary();
    }

    /**
     * Creates a walker - used by GUI implementation
     *
     * @param profileVolume   the profile volume
     * @param analysisControl the analysis controller
     * @param manager         the profiling manager
     */
    public FileFormatWalker(ProfileVolume profileVolume, AnalysisController analysisControl, ProfilingManager manager) {
        super();
        this.profileVolume = profileVolume;
        this.startDirectory = new File(profileVolume.getVolume());
        this.analysisControl = analysisControl;
        this.manager = manager;
        this.wait = analysisControl.getPauseLength();
        this.waitFreq = analysisControl.getPauseFrequency();
        this.summary = new VolumeSummary();
    }

    /**
     * Starts the walker and waits until it finishes
     *
     * @return true if completed successfully, false if error or manually cancelled
     */
    public Boolean call() {
        try {
            walk(this.startDirectory, new ArrayList());
            return Boolean.TRUE;
        } catch (IOException e) {
            //
        }
        return Boolean.FALSE;
    }

    @Override
    protected boolean handleIsCancelled(File file, int depth, Collection results) throws java.io.IOException {
        return cancelled;
    }

    @Override
    protected void handleFile(File diskfile, int depth, Collection results) throws java.io.IOException {
        //Check if operation has been cancelled
        if (cancelled) {
            throw new CancelException("Operation Cancelled on file: " + diskfile.getAbsolutePath(), diskfile, depth);
        }
        //Check if operation has been paused
        if (paused) {
            waitWhilePaused(diskfile, depth);
        }
        //Extract the details about the file
        FileObject fileObject = new FileObject(diskfile, profileVolume);
        IdentificationFile identificationFile;
        if (droid != null) {
            identificationFile = droid.identify(diskfile.getAbsolutePath());
        } else {
            identificationFile = analysisControl.performAnalysis(diskfile);
        }
        fileObject.setClassificationText(identificationFile.getClassificationText());
        Set<Format> formats = new HashSet<Format>();
        for (int i = 0; i < identificationFile.getNumHits(); i++) {
            FileFormatHit hit = identificationFile.getHit(i);
            Format format = new Format(fileObject);
            format.setPuid(hit.getFileFormatPUID());
            format.setName(hit.getFileFormatName());
            format.setMimeType(hit.getMimeType());
            format.setVersion(hit.getFileFormatVersion());
            format.setWarning(hit.getHitWarning());
            formats.add(format);
        }
        
        //check to make sure there is enough space to grow the database
        //Threshold is set to 1MB
        double freeSpace = (manager.getDatabaseLocation().getUsableSpace()/1024d)/1024d;

        if (freeSpace < 1) {
            String error = "There is not enough free disk space on location: "+manager.getDatabaseLocation().getAbsolutePath()+".\nDROID needs a minimum of 1.0 MB to perform profiling.\nPlease free space on this drive and try again.";
            try {
                MessageDisplay.generalErrorjDialog(error);
            } catch (Exception ex) {
                
            }
            log.error(error);
            System.out.println(error);
            cancel();
        } else {
           
            if(!summary.addFileAndFormats(fileObject, formats)){
                cancel();
            }else{
                //Save the data only if summary save was successful
                manager.saveFileAndFormats(fileObject, formats);
            }
            //For throttling
            counter++;
            if (waitFreq > 0 && counter >= waitFreq && wait > 0L) {
                try {
                    counter = 0;
                    Thread.sleep(wait);
                } catch (InterruptedException e) {
                    //Do nothing.
                }
            }
        }
    }

    /**
     * Waits while the walker is paused - sets date completed on profile
     *
     * @param diskfile the current file
     * @param depth    the depth
     * @throws CancelException if cancelled
     */
    private void waitWhilePaused(File diskfile, int depth) throws CancelException {
        //We mark the date completed in case they cancel to avoid getting null data
        //profileVolume.setDateCompleted(new Date());
        manager.saveVolume(profileVolume);
        System.out.println("Entering pause loop");
        log.info("Entering pause loop");
        pauseWait(diskfile, depth);
    }

    /**
     * Waits while paused
     *
     * @param diskfile the disk file
     * @param depth    the depth
     * @throws CancelException if cancelled
     */
    private void pauseWait(File diskfile, int depth) throws CancelException {
        //First check if it's been cancelled
        if (cancelled) {
            throw new CancelException("Operation Cancelled on file: " + diskfile.getAbsolutePath(), diskfile, depth);
        }
        //Sleep then pause again
        if (paused) {
            try {
                Thread.sleep(3000L);
            } catch (InterruptedException e) {
                e.printStackTrace();
                log.error(e.toString());
            } finally {
                pauseWait(diskfile, depth);
            }
        } else {
            System.out.println("Exiting pause loop");
            log.info("Exiting pause loop");
        }
    }

    /**
     * Marks the walker as cancelled
     */
    public void cancel() {
        analysisControl.setCancelAnalysis(true);
        this.cancelled = true;
    }

    /**
     * Marks the walker as paused
     */
    public void pause() {
        this.paused = true;
    }

    /**
     * Removes the paused marker
     */
    public void restart() {
        this.paused = false;
    }

    /**
     * Checks to see if this walker is paused
     *
     * @return boolean
     */
    public boolean isPaused() {
        return paused;
    }

    /**
     * Get the summary data for this volume
     *
     * @return the summary
     */
    public VolumeSummary getSummary() {
        return summary;
    }
}
