/**
 * (c) Copyright  2008 Tessella Support Services plc.  
 * All rights reserved. 
 * <p/>
 * Project: 5890
 * DROID DCS Profile Tool
 * <p/>
 * <p/>
 * Developed By:
 * Tessella Support Services
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * <p/>
 * email:  info@tessella.com
 * web:    www.tessella.com
 * <p/>
 * <p/>
 * Created By:     Lawrence Owusu (owul)
 * Created Date:   2-Nov-2008
 */
package uk.gov.nationalarchives.droid.profile.GUI;

import com.jgoodies.plaf.plastic.PlasticLookAndFeel;
import uk.gov.nationalarchives.droid.profile.service.ProfilingManager;
import uk.gov.nationalarchives.droid.profile.service.ReportManager;

import java.awt.*;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import javax.swing.JOptionPane;
import net.sf.jasperreports.view.JasperViewer;
import org.apache.commons.logging.*;
import uk.gov.nationalarchives.droid.AnalysisController;
import uk.gov.nationalarchives.droid.GUI.CustomFileFilter;

/**
 * This is the main dialog from which the user selects and generates reports
 *
 * @author owul
 */
public class ReportForm extends javax.swing.JDialog {

    private ReportManager rptManager = new ReportManager();
    private ProfilingManager manager;
    
    private Image formIcon = null;
    private Log log = LogFactory.getLog(this.getClass());
    private AnalysisController controller = null;
   
    public static enum ReportType {

        PROFILE, DATABASE, ALL_REPORTS
    };    //The default report type is set to all reports
    public ReportType reportType = null;

    /**
     * Creates new form ReportForm
     *
     * @param parent  the parent object
     * @param modal   is it modal
     * @param manager the profiling manager
     */
    public ReportForm(java.awt.Frame parent, boolean modal, ProfilingManager manager,AnalysisController controller) {
        super(parent, modal);
        setCustomLookAndFeel();
        this.manager = manager;
        initComponents();
        this.controller = controller;
        if(!controller.isFileFormatPopulated){
            log.error("Filter options could not be populated, please check the Droid Configuration and Signature file");
        }
        
        this.lblError.setVisible(false);
        this.lblInfo.setVisible(false);
        this.lblProfileError.setVisible(false);
       // this.txtSearchCriteria.setEnabled(false);
        this.cboxSearchCriteria.setEnabled(false);
        //Set the icon for the window
        
        try {
            formIcon = javax.imageio.ImageIO.read(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/DROID16.gif"));
            setIconImage(formIcon);
        } catch (java.io.IOException e) {
            //Silently ignore exception
        }
        setFocusable(true);
        requestFocus();

        //Centre on screen
        this.setLocationRelativeTo(null);

        //keep form size constant
        this.setResizable(false);
        this.setSize(this.getWidth(), 540);
        this.cboxFilterOperator.setEnabled(false);
        jStatus.setText("Database path:   "+controller.getProfileDatabasePath());
    }

    /**
     * This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jPanel1 = new javax.swing.JPanel();
        cboxReportNames = new javax.swing.JComboBox();
        jLabel1 = new javax.swing.JLabel();
        btnGenerateReport = new javax.swing.JButton();
        lblProfileName1 = new javax.swing.JLabel();
        lblProfileName2 = new javax.swing.JLabel();
        cboxFilterParam = new javax.swing.JComboBox();
        cboxSearchCriteria = new javax.swing.JComboBox();
        lblProfileName3 = new javax.swing.JLabel();
        cboxFilterOperator = new javax.swing.JComboBox();
        pnlProfiles = new javax.swing.JPanel();
        lblProfileName = new javax.swing.JLabel();
        cboxProfileNamesAvailable = new javax.swing.JComboBox();
        lblSelectedProfileName = new javax.swing.JLabel();
        cboxProfileNamesSelected = new javax.swing.JComboBox();
        btnRemoveProfile = new javax.swing.JButton();
        btnAddProfile = new javax.swing.JButton();
        jPanel2 = new javax.swing.JPanel();
        lblProfileError = new javax.swing.JLabel();
        jPanel3 = new javax.swing.JPanel();
        lblInfo = new javax.swing.JLabel();
        lblError = new javax.swing.JLabel();
        btnGenerateCSVReport = new javax.swing.JButton();
        btnGenerateReport1 = new javax.swing.JButton();
        jStatus = new javax.swing.JLabel();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        setTitle("DROID Reporting ");
        setResizable(false);

        jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder("Select Options"));

        cboxReportNames.setBackground(new java.awt.Color(241, 242, 242));
        cboxReportNames.setFont(new java.awt.Font("Tahoma", 0, 14));
        cboxReportNames.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                cboxReportNamesItemStateChanged(evt);
            }
        });
        cboxReportNames.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cboxReportNamesActionPerformed(evt);
            }
        });

        jLabel1.setText("Select A Report");

        btnGenerateReport.setIcon(new javax.swing.ImageIcon(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/column-chart-24.png"))); // NOI18N
        btnGenerateReport.setMnemonic('V');
        btnGenerateReport.setText("View Report ...");
        btnGenerateReport.setHorizontalTextPosition(javax.swing.SwingConstants.RIGHT);
        btnGenerateReport.setIconTextGap(10);
        btnGenerateReport.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnGenerateReportActionPerformed(evt);
            }
        });

        lblProfileName1.setText("Select Filter Parameter");

        lblProfileName2.setText("Enter Search Criteria");

        cboxFilterParam.setBackground(new java.awt.Color(241, 242, 242));
        cboxFilterParam.setFont(new java.awt.Font("Tahoma", 0, 14));
        cboxFilterParam.setMaximumRowCount(9);
        cboxFilterParam.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "File Name", "File Path", "Format", "Identification Status", "Mime Type", "PUID", "Version", "Warning", "Year Modified" }));
        cboxFilterParam.setAutoscrolls(true);
        cboxFilterParam.setEnabled(false);
        cboxFilterParam.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cboxFilterParamActionPerformed(evt);
            }
        });

        cboxSearchCriteria.setBackground(new java.awt.Color(241, 242, 242));
        cboxSearchCriteria.setEditable(true);
        cboxSearchCriteria.setFont(new java.awt.Font("Tahoma", 0, 14));
        cboxSearchCriteria.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "*" }));

        lblProfileName3.setText("Select Filter Operator");

        cboxFilterOperator.setBackground(new java.awt.Color(241, 242, 242));
        cboxFilterOperator.setFont(new java.awt.Font("Tahoma", 0, 14));
        cboxFilterOperator.setMaximumRowCount(9);
        cboxFilterOperator.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "= (equal to)", "> (greater than)", ">= (greater than or equal to)", "< (less than)", "<= (less than or equal to)", "!= (not equal to)", "LIKE" }));
        cboxFilterOperator.setAutoscrolls(true);
        cboxFilterOperator.setEnabled(false);
        cboxFilterOperator.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cboxFilterOperatorActionPerformed(evt);
            }
        });

        pnlProfiles.setBorder(javax.swing.BorderFactory.createTitledBorder("Select Profiles"));
        pnlProfiles.setEnabled(false);
        pnlProfiles.setMaximumSize(new java.awt.Dimension(672, 166));

        lblProfileName.setText("Available Profiles ");

        cboxProfileNamesAvailable.setBackground(new java.awt.Color(241, 242, 242));
        cboxProfileNamesAvailable.setFont(new java.awt.Font("Tahoma", 0, 14));
        cboxProfileNamesAvailable.setEnabled(false);

        lblSelectedProfileName.setText("Selected Profiles");

        cboxProfileNamesSelected.setBackground(new java.awt.Color(241, 242, 242));
        cboxProfileNamesSelected.setFont(new java.awt.Font("Tahoma", 0, 14));
        cboxProfileNamesSelected.setEnabled(false);

        btnRemoveProfile.setIcon(new javax.swing.ImageIcon(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/Delete Document 2 16 n g.gif"))); // NOI18N
        btnRemoveProfile.setText("Remove Profile");
        btnRemoveProfile.setDisabledIcon(new javax.swing.ImageIcon(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/Delete Document 2 16 d g.gif"))); // NOI18N
        btnRemoveProfile.setEnabled(false);
        btnRemoveProfile.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                btnRemoveProfileMousePressed(evt);
            }
        });
        btnRemoveProfile.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnRemoveProfileActionPerformed(evt);
            }
        });

        btnAddProfile.setIcon(new javax.swing.ImageIcon(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/Add Document 2 16 h g.gif"))); // NOI18N
        btnAddProfile.setText("Add Profile");
        btnAddProfile.setDisabledIcon(new javax.swing.ImageIcon(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/Add Document 2 16 d g.gif"))); // NOI18N
        btnAddProfile.setEnabled(false);
        btnAddProfile.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mousePressed(java.awt.event.MouseEvent evt) {
                btnAddProfileMousePressed(evt);
            }
        });
        btnAddProfile.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnAddProfileActionPerformed(evt);
            }
        });

        lblProfileError.setFont(new java.awt.Font("Tahoma", 0, 12));
        lblProfileError.setForeground(new java.awt.Color(255, 0, 51));
        lblProfileError.setText("No selected profile. Please add a profile");

        javax.swing.GroupLayout jPanel2Layout = new javax.swing.GroupLayout(jPanel2);
        jPanel2.setLayout(jPanel2Layout);
        jPanel2Layout.setHorizontalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel2Layout.createSequentialGroup()
                .addGap(115, 115, 115)
                .addComponent(lblProfileError)
                .addContainerGap(114, Short.MAX_VALUE))
        );
        jPanel2Layout.setVerticalGroup(
            jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addComponent(lblProfileError, javax.swing.GroupLayout.PREFERRED_SIZE, 16, javax.swing.GroupLayout.PREFERRED_SIZE)
        );

        javax.swing.GroupLayout pnlProfilesLayout = new javax.swing.GroupLayout(pnlProfiles);
        pnlProfiles.setLayout(pnlProfilesLayout);
        pnlProfilesLayout.setHorizontalGroup(
            pnlProfilesLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(pnlProfilesLayout.createSequentialGroup()
                .addGroup(pnlProfilesLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(pnlProfilesLayout.createSequentialGroup()
                        .addGap(252, 252, 252)
                        .addComponent(btnAddProfile, javax.swing.GroupLayout.PREFERRED_SIZE, 114, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(69, 69, 69)
                        .addComponent(btnRemoveProfile))
                    .addGroup(pnlProfilesLayout.createSequentialGroup()
                        .addGap(43, 43, 43)
                        .addGroup(pnlProfilesLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(pnlProfilesLayout.createSequentialGroup()
                                .addComponent(lblProfileName)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                .addComponent(cboxProfileNamesAvailable, javax.swing.GroupLayout.PREFERRED_SIZE, 504, javax.swing.GroupLayout.PREFERRED_SIZE))
                            .addGroup(pnlProfilesLayout.createSequentialGroup()
                                .addComponent(lblSelectedProfileName, javax.swing.GroupLayout.PREFERRED_SIZE, 106, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addGroup(pnlProfilesLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                    .addComponent(cboxProfileNamesSelected, javax.swing.GroupLayout.PREFERRED_SIZE, 506, javax.swing.GroupLayout.PREFERRED_SIZE)
                                    .addComponent(jPanel2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))))))
                .addGap(12, 12, 12))
        );
        pnlProfilesLayout.setVerticalGroup(
            pnlProfilesLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(pnlProfilesLayout.createSequentialGroup()
                .addGap(8, 8, 8)
                .addGroup(pnlProfilesLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(lblProfileName)
                    .addComponent(cboxProfileNamesAvailable, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(pnlProfilesLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(btnAddProfile)
                    .addComponent(btnRemoveProfile))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(pnlProfilesLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(lblSelectedProfileName)
                    .addComponent(cboxProfileNamesSelected, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jPanel2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        lblInfo.setFont(new java.awt.Font("Tahoma", 0, 12));
        lblInfo.setForeground(new java.awt.Color(0, 70, 213));
        lblInfo.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        lblInfo.setText("Use \"*\" as a wildcard character for string filtering e.g. (enter \"*\" to select all records)                  ");
        lblInfo.setHorizontalTextPosition(javax.swing.SwingConstants.RIGHT);

        lblError.setFont(new java.awt.Font("Tahoma", 0, 12));
        lblError.setForeground(new java.awt.Color(255, 0, 51));
        lblError.setHorizontalAlignment(javax.swing.SwingConstants.CENTER);
        lblError.setText("Invalid search criteria. Please enter a valid number                       ");

        javax.swing.GroupLayout jPanel3Layout = new javax.swing.GroupLayout(jPanel3);
        jPanel3.setLayout(jPanel3Layout);
        jPanel3Layout.setHorizontalGroup(
            jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(lblInfo, javax.swing.GroupLayout.DEFAULT_SIZE, 696, Short.MAX_VALUE)
                    .addGroup(jPanel3Layout.createSequentialGroup()
                        .addComponent(lblError, javax.swing.GroupLayout.DEFAULT_SIZE, 684, Short.MAX_VALUE)
                        .addContainerGap())))
        );
        jPanel3Layout.setVerticalGroup(
            jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel3Layout.createSequentialGroup()
                .addComponent(lblInfo, javax.swing.GroupLayout.PREFERRED_SIZE, 16, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addComponent(lblError, javax.swing.GroupLayout.PREFERRED_SIZE, 16, javax.swing.GroupLayout.PREFERRED_SIZE))
        );

        btnGenerateCSVReport.setIcon(new javax.swing.ImageIcon(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/export1-32.png"))); // NOI18N
        btnGenerateCSVReport.setMnemonic('E');
        btnGenerateCSVReport.setHorizontalTextPosition(javax.swing.SwingConstants.RIGHT);
        btnGenerateCSVReport.setIconTextGap(10);
        btnGenerateCSVReport.setLabel("Export to CSV ...");
        btnGenerateCSVReport.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnGenerateCSVReportActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING, false)
                    .addComponent(jPanel3, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addGroup(javax.swing.GroupLayout.Alignment.LEADING, jPanel1Layout.createSequentialGroup()
                        .addComponent(jLabel1, javax.swing.GroupLayout.PREFERRED_SIZE, 111, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addGap(53, 53, 53)
                        .addComponent(cboxReportNames, javax.swing.GroupLayout.PREFERRED_SIZE, 503, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(javax.swing.GroupLayout.Alignment.LEADING, jPanel1Layout.createSequentialGroup()
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(lblProfileName2, javax.swing.GroupLayout.PREFERRED_SIZE, 123, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(lblProfileName1)
                            .addComponent(lblProfileName3, javax.swing.GroupLayout.PREFERRED_SIZE, 127, javax.swing.GroupLayout.PREFERRED_SIZE))
                        .addGap(38, 38, 38)
                        .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(cboxFilterOperator, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(cboxFilterParam, javax.swing.GroupLayout.PREFERRED_SIZE, 157, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addComponent(cboxSearchCriteria, javax.swing.GroupLayout.PREFERRED_SIZE, 500, javax.swing.GroupLayout.PREFERRED_SIZE)))
                    .addComponent(pnlProfiles, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
            .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel1Layout.createSequentialGroup()
                .addContainerGap(366, Short.MAX_VALUE)
                .addComponent(btnGenerateCSVReport, javax.swing.GroupLayout.PREFERRED_SIZE, 169, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(18, 18, 18)
                .addComponent(btnGenerateReport)
                .addGap(26, 26, 26))
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel1)
                    .addComponent(cboxReportNames, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(18, 18, 18)
                .addComponent(pnlProfiles, javax.swing.GroupLayout.PREFERRED_SIZE, 145, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(lblProfileName1)
                    .addComponent(cboxFilterParam, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(26, 26, 26)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(lblProfileName3)
                    .addComponent(cboxFilterOperator, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(19, 19, 19)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(lblProfileName2)
                    .addComponent(cboxSearchCriteria, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jPanel3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(btnGenerateReport)
                    .addComponent(btnGenerateCSVReport, javax.swing.GroupLayout.PREFERRED_SIZE, 31, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        btnGenerateReport1.setIcon(new javax.swing.ImageIcon(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/error-24.png"))); // NOI18N
        btnGenerateReport1.setMnemonic('C');
        btnGenerateReport1.setText("Close");
        btnGenerateReport1.setHorizontalTextPosition(javax.swing.SwingConstants.RIGHT);
        btnGenerateReport1.setIconTextGap(10);
        btnGenerateReport1.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                btnGenerateReport1ActionPerformed(evt);
            }
        });

        jStatus.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
        jStatus.setMaximumSize(new java.awt.Dimension(100, 20));
        jStatus.setMinimumSize(new java.awt.Dimension(0, 10));
        jStatus.setPreferredSize(new java.awt.Dimension(100, 20));

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(layout.createSequentialGroup()
                            .addComponent(jPanel1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addContainerGap())
                        .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                            .addComponent(btnGenerateReport1, javax.swing.GroupLayout.PREFERRED_SIZE, 101, javax.swing.GroupLayout.PREFERRED_SIZE)
                            .addGap(50, 50, 50)))
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(jStatus, javax.swing.GroupLayout.DEFAULT_SIZE, 748, Short.MAX_VALUE)
                        .addContainerGap())))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addComponent(jPanel1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addComponent(btnGenerateReport1)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jStatus, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    /**
     * Controls the way the Profile Combo Box behaves. It is enableed only when the report name selected expects a profile name
     *
     * @param evt the event
     */
    private void cboxReportNamesActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cboxReportNamesActionPerformed
        //if report name contains the word "Profile" then it requires a profile name as filter
        if (cboxReportNames.getItemCount() > 0) {

            
            resetAvailableProfileBox();
            if (cboxReportNames.getSelectedItem().toString().contains("Profile")) {
                setProfileComponentsEnabled(true);
            } else {
                setProfileComponentsEnabled(false);
            }
            if (cboxReportNames.getSelectedItem().toString().contains("File Detail")) {
                btnGenerateCSVReport.setEnabled(true);
            } else {
                btnGenerateCSVReport.setEnabled(false);
            }
        }
        cboxReportNamesItemStateChanged(null);

    }//GEN-LAST:event_cboxReportNamesActionPerformed

    /**
     *
     * @param setValue A boolean value to which all the components related to Profiles enabled property are set
     */
    private void setProfileComponentsEnabled(boolean setValue){
        //reset values to default
        
        if(cboxProfileNamesSelected.getItemCount() > 0){
            
        }
        
        pnlProfiles.setEnabled(setValue);
        cboxProfileNamesAvailable.setEnabled(setValue);
        cboxProfileNamesSelected.setEnabled(setValue);
        btnAddProfile.setEnabled(setValue);
        if(!setValue){
            btnRemoveProfile.setEnabled(setValue);
        }
        if(!setValue){
            //remove error message if Profile components are disabled
            lblProfileError.setVisible(false);
        }

    }

    /**
     * Generate the report
     */
    private void generateReport(){
        //set cursor to wait cursor as the report view is about to be displayed
        this.setCursor(new Cursor(Cursor.WAIT_CURSOR));

        //String filter = txtSearchCriteria.getText().trim();
        String filter = ((String) cboxSearchCriteria.getEditor().getItem()).trim();

        //if filter string is an empty string show error label and exit the method
        if (cboxSearchCriteria.isEnabled() &&  filter.trim().equals("")) {
            this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
            lblError.setVisible(true);
            return;
        } else {
            //Jasper reports uses % as its wildcard character, so we replace all * with %
            filter = filter.replace('*', '%').trim();
        }


        //retrieve the report name
        String reportName = "";
        if (cboxReportNames.getItemCount() > 0) {
            if (cboxFilterParam.isEnabled()) {
                reportName = cboxReportNames.getSelectedItem().toString().trim() + "_" + getReportNameSuffix();
            } else {
                reportName = cboxReportNames.getSelectedItem().toString().trim();
            }

        }

        //Its only when the profile name combo box is enabled that we select profile name as a filter
        java.util.List<String> profileNames = new ArrayList<String>();
        boolean errorOccured = false;
        if (cboxProfileNamesSelected.isEnabled()) {
            if (cboxProfileNamesSelected.getItemCount() > 0) {
                for (int i = 0; i < cboxProfileNamesSelected.getItemCount(); i++) {
                    profileNames.add(cboxProfileNamesSelected.getItemAt(i).toString().toUpperCase().trim());
                }
                lblProfileError.setVisible(false);

            } else {
                lblProfileError.setVisible(true);
                errorOccured = true;
            }
        }

        //retrieve selected operator for numeric fields in this case the "Date" field
        lblError.setVisible(false);
        if (reportName.trim().contains("Date")) {
            int index = cboxFilterOperator.getSelectedIndex();
            String operator = "";
            switch (index) {
                case 0:
                    operator = "=";
                    break;
                case 1:
                    operator = ">";
                    break;
                case 2:
                    operator = ">=";
                    break;
                case 3:
                    operator = "<";
                    break;
                case 4:
                    operator = "<=";
                    break;
                case 5:
                    operator = "!=";
                    break;
                case 6:
                    operator = "like";
                    break;
            }

            if (index == 6 && filter.trim().equals("%")) {
                filter = "";
            } else if (index == 6) { //filter for a LIKE search (search type using the "LIKE" keyword)
                if (profileNames.size() == 0) {
                    filter = "WHERE UCASE(cast(YEAR(fileobject.lastmodified)as char(4))) LIKE UCASE('" + filter.trim() + "')";
                } else {
                    filter = " (UCASE(cast(YEAR(fileobject.lastmodified)as char(4))) LIKE UCASE('" + filter.trim() + "'))";
                }

            } else { //numeric search type
                int year = 0;
                try {
                    //test whether or not the filter is a valid interger number
                    year = Integer.parseInt(filter);
                } catch (NumberFormatException ex) {
                    lblError.setVisible(true);
                    this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
                    errorOccured = true;
                }
                filter = Integer.toString(year);
                if (profileNames.size() == 0) {
                    filter = "WHERE YEAR(fileobject.lastmodified) " + operator.trim() + " " + filter.trim();
                } else {
                    filter = " ( YEAR(fileobject.lastmodified) " + operator.trim() + " " + filter.trim() + ")";
                }

            }
        }

        //if no error occured, show the report
        if (!errorOccured) {
            //show the report viewer
            JasperViewer viewer = rptManager.showReport(reportName, profileNames, manager, filter.trim());

            if (viewer == null) {
                System.out.println("The report could not be generated. An error occured");
                log.error("The report could not be generated. An error occured");
            } else {
                viewer.setVisible(true);
                viewer.setLocationRelativeTo(this);
            }
        //revert cursor to default cursor

        }
        this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    }

    
/**
 * Populate the search criteria combo box with options
 * @param evt
 */    
private void cboxFilterParamActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cboxFilterParamActionPerformed
    if (controller.isFileFormatPopulated && cboxFilterParam.getItemCount() > 0) {
        int index = cboxFilterParam.getSelectedIndex();
        if (cboxSearchCriteria.getItemCount() > 0) {
            cboxSearchCriteria.removeAllItems();
        }
            switch (index) {
                case 2:
                    for (String str : controller.formatName) {
                        cboxSearchCriteria.addItem(str.trim());
                    }
                    break;
                case 3:
                     cboxSearchCriteria.addItem("Not identified");
                     cboxSearchCriteria.addItem("Positive");
                     cboxSearchCriteria.addItem("Tentative");
                    break;
                case 4:
                    for (String str : controller.mimeType) {
                        cboxSearchCriteria.addItem(str.trim());
                    }
                    break;
                case 5:
                    for (String str : controller.PUID) {
                        cboxSearchCriteria.addItem(str.trim());
                    }
                   break;
                case 6:
                    for (String str : controller.version) {
                        cboxSearchCriteria.addItem(str.trim());
                    }
                    
                    break;  
                case 8:
                    cboxFilterOperator.setEnabled(true);
                    lblInfo.setVisible(false);
                    //lblError.setVisible(true);
                    cboxSearchCriteria.addItem("0");
                    break;
                default:
                    cboxSearchCriteria.addItem("*");
                    break;
            }
            
            if(index != 8){
                cboxFilterOperator.setSelectedIndex(0);
                cboxFilterOperator.setEnabled(false);
                lblInfo.setVisible(true);
                lblError.setVisible(false);
            }
            
        
    }
}//GEN-LAST:event_cboxFilterParamActionPerformed

private void cboxReportNamesItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_cboxReportNamesItemStateChanged
   if(cboxReportNames.getItemCount() > 0 && this.isVisible())          {
       this.lblError.setVisible(false);
       this.lblInfo.setVisible(false);
       setFilterComponents(); 
   }
}//GEN-LAST:event_cboxReportNamesItemStateChanged

private void cboxFilterOperatorActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cboxFilterOperatorActionPerformed
// TODO add your handling code here:
}//GEN-LAST:event_cboxFilterOperatorActionPerformed

private void btnRemoveProfileMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_btnRemoveProfileMousePressed
    // TODO add your handling code here:
}//GEN-LAST:event_btnRemoveProfileMousePressed


/**
 * Fill available profile box with a list of all profiles and empty the selected profile box
 */
private void resetAvailableProfileBox(){
    if(cboxProfileNamesSelected.getItemCount()> 0){
        java.util.List<String> items = new ArrayList<String>();
        for (int i = 0; i < cboxProfileNamesAvailable.getItemCount(); i++) {
            items.add(cboxProfileNamesAvailable.getItemAt(i).toString().trim());
        }
        for (int i = 0; i < cboxProfileNamesSelected.getItemCount(); i++) {
            items.add(cboxProfileNamesSelected.getItemAt(i).toString().trim());
        }
        Collections.sort(items);
        cboxProfileNamesSelected.removeAllItems();
        cboxProfileNamesAvailable.removeAllItems();
        for (String str : items) {
            cboxProfileNamesAvailable.addItem(str);
        }
        btnRemoveProfile.setEnabled(false);
        btnAddProfile.setEnabled(true);
        
    }
    
}


/**
 * Remove selected item from the "Selected Profiles" combo box into the "Available Profiles" combo box
 * @param evt
 */
private void btnRemoveProfileActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnRemoveProfileActionPerformed
     if(cboxProfileNamesSelected.getItemCount() > 0){
        String profile = cboxProfileNamesSelected.getSelectedItem().toString().trim();
        int index = cboxProfileNamesSelected.getSelectedIndex();
        if(profile != null & !profile.trim().equals("")){
            java.util.List<String> items = new ArrayList<String>();
            for(int i=0;i<cboxProfileNamesAvailable.getItemCount();i++){
                items.add(cboxProfileNamesAvailable.getItemAt(i).toString().trim());
            }
            items.add(profile);
            Collections.sort(items);
            cboxProfileNamesAvailable.removeAllItems();

            for(String str: items){
                cboxProfileNamesAvailable.addItem(str);
            }
            cboxProfileNamesSelected.removeItemAt(index);
            if(cboxProfileNamesSelected.getItemCount() == 0){
                btnRemoveProfile.setEnabled(false);
            }
            btnAddProfile.setEnabled(true);
        }
    }
}//GEN-LAST:event_btnRemoveProfileActionPerformed

private void btnAddProfileMousePressed(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_btnAddProfileMousePressed
    // TODO add your handling code here:
}//GEN-LAST:event_btnAddProfileMousePressed

/**
 *  Remove selected item from the "Available Profiles" combo box into the "Selected Profiles" combo box
 * @param evt
 */
private void btnAddProfileActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnAddProfileActionPerformed
    if(cboxProfileNamesAvailable.getItemCount() > 0){
        String profile = cboxProfileNamesAvailable.getSelectedItem().toString().trim();
        int index = cboxProfileNamesAvailable.getSelectedIndex();
        if(profile != null & !profile.trim().equals("")){
            java.util.List<String> items = new ArrayList<String>();
            for(int i=0;i<cboxProfileNamesSelected.getItemCount();i++){
                items.add(cboxProfileNamesSelected.getItemAt(i).toString().trim());
            }
            items.add(profile);
            Collections.sort(items);
            cboxProfileNamesSelected.removeAllItems();

            for(String str: items){
                cboxProfileNamesSelected.addItem(str);
            }
            cboxProfileNamesAvailable.removeItemAt(index);
            lblProfileError.setVisible(false);
            
            if(cboxProfileNamesAvailable.getItemCount() == 0){
                btnAddProfile.setEnabled(false);
            }
            btnRemoveProfile.setEnabled(true);

        }
    }
}//GEN-LAST:event_btnAddProfileActionPerformed

private void btnGenerateReportActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnGenerateReportActionPerformed
        generateReport();
}//GEN-LAST:event_btnGenerateReportActionPerformed

private void btnGenerateCSVReportActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnGenerateCSVReportActionPerformed
   exportToCSV();   
}//GEN-LAST:event_btnGenerateCSVReportActionPerformed

private void btnGenerateReport1ActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnGenerateReport1ActionPerformed
   this.dispose();
}//GEN-LAST:event_btnGenerateReport1ActionPerformed




    private void exportToCSV() {

        String fileName = getExportCSVFileName();
        if (!fileName.trim().equals("")) {

            //set cursor to wait cursor as the report view is about to be displayed
            this.setCursor(new Cursor(Cursor.WAIT_CURSOR));
            //String filter = txtSearchCriteria.getText().trim();
            String filter = ((String) cboxSearchCriteria.getEditor().getItem()).trim();

            //if filter string is an empty string show error label and exit the method
            if (cboxSearchCriteria.isEnabled() && filter.trim().equals("")) {
                this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
                lblError.setVisible(true);
                return;
            } else {
                //Jasper reports uses % as its wildcard character, so we replace all * with %
                filter = filter.replace('*', '%').trim();
            }


            //retrieve the report name
            String reportName = "";
            if (cboxReportNames.getItemCount() > 0) {
                reportName = cboxReportNames.getSelectedItem().toString().trim();
            }

            //Its only when the profile name combo box is enabled that we select profile name as a filter
            java.util.List<String> profileNames = new ArrayList<String>();
            boolean errorOccured = false;
            if (cboxProfileNamesSelected.isEnabled()) {
                if (cboxProfileNamesSelected.getItemCount() > 0) {
                    for (int i = 0; i < cboxProfileNamesSelected.getItemCount(); i++) {
                        profileNames.add(cboxProfileNamesSelected.getItemAt(i).toString().toUpperCase().trim());
                    }
                    lblProfileError.setVisible(false);

                } else {
                    lblProfileError.setVisible(true);
                    errorOccured = true;
                }
            }

            if (!filter.trim().equals("%")) {
                //retrieve selected operator for numeric fields in this case the "Date" field
                lblError.setVisible(false);
                if (getReportNameSuffix().trim().equalsIgnoreCase("Date")) {
                    int index = cboxFilterOperator.getSelectedIndex();
                    String operator = "";
                    switch (index) {
                        case 0:
                            operator = "=";
                            break;
                        case 1:
                            operator = ">";
                            break;
                        case 2:
                            operator = ">=";
                            break;
                        case 3:
                            operator = "<";
                            break;
                        case 4:
                            operator = "<=";
                            break;
                        case 5:
                            operator = "!=";
                            break;
                        case 6:
                            operator = "like";
                            break;
                    }

                    if (index == 6 && filter.trim().equals("%")) {
                        filter = "";
                    } else if (index == 6) { //filter for a LIKE search (search type using the "LIKE" keyword)

                        filter = " AND (UCASE(cast(YEAR(fileobject.lastmodified)as char(4))) LIKE UCASE(''" + filter.trim() + "''))";
                    } else { //numeric search type
                        int year = 0;
                        try {
                            //test whether or not the filter is a valid interger number
                            year = Integer.parseInt(filter);
                        } catch (NumberFormatException ex) {
                            lblError.setVisible(true);
                            this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
                            errorOccured = true;
                        }
                        filter = Integer.toString(year);

                        filter = " AND ( YEAR(fileobject.lastmodified) " + operator.trim() + " " + filter.trim() + ")";
                    }
                } else {
                    filter = " AND " + getFilterQuery(getReportNameSuffix()) + " UCASE(''" + filter + "'') ";
                }
            }
            //if no error occured, show the report
            if (!errorOccured) {
                //show the report viewer
                File outputFile = new File(fileName);
                String error = manager.export(outputFile, profileNames, reportName, filter);
                if (error.trim().equals("")) {

                    JOptionPane.showMessageDialog(this, "CSV export has completed successfully", "Export Complete", JOptionPane.INFORMATION_MESSAGE);
                } else {
                    error = error.replace(". ", ".\n");
                    error = error.replace(":", ":\n");
                    JOptionPane.showMessageDialog(this, "CSV export unsuccessful. The following error occured:\n" + error, "Export Complete", JOptionPane.ERROR_MESSAGE);
                }
            }
            //revert cursor to default cursor
            this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        }

    }


private String getFilterQuery(String parameterName){
        String query = "";
        if(parameterName.equalsIgnoreCase("FileName")){
            //UCASE(format.version) like
            query = " UCASE(fileobject.name) like ";
        }else if(parameterName.equalsIgnoreCase("FilePath")){
            query = " UCASE(fileobject.path) like ";
        }else if(parameterName.equalsIgnoreCase("Format")){
            query = " UCASE(format.name) like ";
        }else if(parameterName.equalsIgnoreCase("IDStatus")){
            query = " UCASE(fileobject.CLASSIFICATIONTEXT) like ";
        }else if(parameterName.equalsIgnoreCase("MIME")){
            query = " UCASE(format.mimetype) like ";
        }else if(parameterName.equalsIgnoreCase("PUID")){
            query = " UCASE(format.puid) like ";
        }else if(parameterName.equalsIgnoreCase("VERSION")){
            query = " UCASE(format.version) like ";
        }else if(parameterName.equalsIgnoreCase("WARNING")){
               query = " UCASE(format.warning) like ";
        }else if(parameterName.equalsIgnoreCase("Date")){
             query = "";
        }
        
        return query;
    }
    
    
    private String getExportCSVFileName() {

        String fileName = "";
        java.io.File path = null;
        try {
            //Setup save file dialog
            javax.swing.JFileChooser fc = new javax.swing.JFileChooser();
            fc.addChoosableFileFilter(new CustomFileFilter("csv", "Comma Separated Values (*.csv)"));
            fc.setAcceptAllFileFilterUsed(false);
            //Set dialog title to same as dialog boxess 
            fc.setDialogTitle("Select destination file to export to");

            //show file dialog
            int returnVal = fc.showSaveDialog(this);

            //Save file if user has chosen a file
            if (returnVal == javax.swing.JFileChooser.APPROVE_OPTION) {

                //Get file user selected
                path = fc.getSelectedFile();

                fileName = path.getAbsolutePath();
                //if no extension was specified add one
                if (!(path.getName().endsWith(".csv"))) {
                    path = new java.io.File(path.getParentFile(), path.getName() + ".csv");
                    fileName = path.getAbsolutePath();
                }

                //if path exists check confirm with user if they want to overwrite
                if (path.exists()) {
                    int option = javax.swing.JOptionPane.showConfirmDialog(this, "The specified file \"" + fileName + "\" exists, overwrite?");
                    if (option != javax.swing.JOptionPane.YES_OPTION) {
                        fileName = "";
                    } else {
                        path.delete();
                    }

                }


            }
        } catch (Exception ex) {
            fileName = "";
        }
        return fileName;
    }



public void setEnablePropOperatorsList(boolean enable){
    this.cboxFilterOperator.setEnabled(enable);
}

/**
 *  Set enabled properties of Filter components
 */
public void setFilterComponents(){
        boolean enableFilter = enableFilterParameters();
        cboxSearchCriteria.setEnabled(enableFilter);
        cboxFilterParam.setEnabled(enableFilter);
        cboxFilterParam.setSelectedIndex(0);
        cboxFilterParamActionPerformed(null);
        
        //        setEnablePropOperatorsList(enableFilter);
        
}

/**
 *  This functions returns a boolean value to determine if filters are to be used for this report
 * @return A value to determine if the filter combo box is enabled or disabled
 */
private boolean enableFilterParameters(){
        //flag used to abort Report Generation if an error occurs
        boolean canFilter = true;
       
        //Its only when the profile name combo box is enabled that we need profile name as a filter
        if (cboxProfileNamesAvailable.isEnabled()) {
            if (cboxProfileNamesAvailable.getItemCount() == 0 && cboxProfileNamesSelected.getItemCount() == 0) {
               canFilter = false;
            }
        }

            String reportName = "";
        if (cboxReportNames.getItemCount() > 0) {
            reportName = cboxReportNames.getSelectedItem().toString().trim();
        }else{
            canFilter = false;
        }

        if (!canFilter) {
            System.out.println("Either the Profile Names or Report Names or both could not retrieved.");
            log.error("Either the Profile Names or Report Names or both could not retrieved.");
            this.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
            return false;
        }
        
        

        //Filtering is not done for the following reports "Entire Database" and "Profile Summary" reports
        if ((reportName.toUpperCase().contains("ENTIRE DATABASE")  || reportName.toUpperCase().contains("PROFILE SUMMARY")) ) {
             return false;
        }

        return canFilter;

        


}

/**
 * Appends the correct suffix to the selected report name
 * @return
 */
private String getReportNameSuffix(){
    String suffix = "";
    
    if(cboxFilterParam.getItemCount() > 0){
        int index = cboxFilterParam.getSelectedIndex();
        switch(index){
            case 0:
                suffix = "FileName";
                break;
            case 1:
                suffix = "FilePath";
                break;
            case 2:
                suffix = "Format";
                break;
            case 3:
                suffix = "IDStatus";
                break;
            case 4:
                suffix = "Mime";
                break;
            case 5:
                suffix = "PUID";
                break;                
            case 6:
                suffix = "Version";
                break;
            case 7:
                 suffix = "Warning";
                break;
            case 8:
                   suffix = "Date";
                break;
         }
      }
     return suffix;
}
   /**
     * Used to populate the Combo boxes with the report and profile names
     */
    public void populateProfileAndReportNames() {
        
        //Empty combo boxes before populating them again
        if(cboxProfileNamesAvailable.getItemCount() > 0)
        {
            cboxProfileNamesAvailable.removeAllItems();
        }
        
        if(cboxReportNames.getItemCount() > 0)
        {
            cboxReportNames.removeAllItems();
        }

        for (String rptName :  rptManager.getReportNames()) {
            boolean addReport = false;
            if(reportType == ReportType.DATABASE && rptName.toUpperCase().contains("DATABASE")){
                addReport = true;
                lblProfileName.setEnabled(false);
                lblSelectedProfileName.setEnabled(false);
            }
            else if(reportType == ReportType.PROFILE && rptName.toUpperCase().contains("PROFILE")){
                addReport = true;
                lblProfileName.setEnabled(true);
                lblSelectedProfileName.setEnabled(true);
            }
            else if(reportType == ReportType.ALL_REPORTS){
                addReport = true;
                lblProfileName.setEnabled(true);
                lblSelectedProfileName.setEnabled(true);
            }
            if(addReport){
              cboxReportNames.addItem(rptName.trim());
            }    

        }
        if(cboxReportNames.getItemCount() >0){
            cboxReportNames.setSelectedIndex(0);
        }
        if (reportType == ReportType.PROFILE || reportType == ReportType.ALL_REPORTS) {
            java.util.List<String> profileNames = manager.getAllProfileNames();
            if (profileNames.size() > 0) {
                Collections.sort(profileNames);
            }
            for (String profName : profileNames) {
                cboxProfileNamesAvailable.addItem(profName.trim());
            }
            if (cboxProfileNamesAvailable.getItemCount() > 0) {
                cboxProfileNamesAvailable.setSelectedIndex(0);
            }
        }


    }

    /**
     * Get the dialog box to have the same look and feel as the main DROID form
     */
    private void setCustomLookAndFeel() {
        try {
            com.jgoodies.plaf.plastic.PlasticLookAndFeel lf = new com.jgoodies.plaf.plastic.PlasticLookAndFeel();
            PlasticLookAndFeel.setMyCurrentTheme(new com.jgoodies.plaf.plastic.theme.SkyBluer());
            javax.swing.UIManager.setLookAndFeel(lf);
            javax.swing.UIManager.put("jgoodies.popupDropShadowEnabled", Boolean.TRUE);
        } catch (Exception e) {
            //Silently ignore exception
        }
    }


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton btnAddProfile;
    private javax.swing.JButton btnGenerateCSVReport;
    private javax.swing.JButton btnGenerateReport;
    private javax.swing.JButton btnGenerateReport1;
    private javax.swing.JButton btnRemoveProfile;
    private javax.swing.JComboBox cboxFilterOperator;
    private javax.swing.JComboBox cboxFilterParam;
    private javax.swing.JComboBox cboxProfileNamesAvailable;
    private javax.swing.JComboBox cboxProfileNamesSelected;
    private javax.swing.JComboBox cboxReportNames;
    private javax.swing.JComboBox cboxSearchCriteria;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JLabel jStatus;
    private javax.swing.JLabel lblError;
    private javax.swing.JLabel lblInfo;
    private javax.swing.JLabel lblProfileError;
    private javax.swing.JLabel lblProfileName;
    private javax.swing.JLabel lblProfileName1;
    private javax.swing.JLabel lblProfileName2;
    private javax.swing.JLabel lblProfileName3;
    private javax.swing.JLabel lblSelectedProfileName;
    private javax.swing.JPanel pnlProfiles;
    // End of variables declaration//GEN-END:variables

}
