/**
 *  � The National Archives 2005-2008.  All rights reserved.
 * See Licence.txt for full licence details.
 * <p/>
 *
 *  DROID DCS Profile Tool
 * <p/>
 * <p/>
 * Developed By:
 * Tessella Support Services
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * <p/>
 * email:  info@tessella.com
 * web:    www.tessella.com
 * <p/>
 * <p/>
 * Created Date:   1-Nov-2008
 */
package uk.gov.nationalarchives.droid.profile;

import uk.gov.nationalarchives.droid.profile.domain.Profile;
import uk.gov.nationalarchives.droid.profile.domain.ProfileVolume;
import uk.gov.nationalarchives.droid.profile.service.ProfilingManager;
import org.jargp.ArgumentProcessor;
import org.jargp.ParameterDef;
import org.jargp.StringDef;

import java.io.File;
import java.net.InetAddress;
import org.apache.commons.logging.*;

/**
 * The primary entry point for utilising profiling tools.
 */
public class ProfileEntryPoint {

    private static Log log = LogFactory.getLog(ProfileEntryPoint.class);
    private static String DEFAULT_PATH_FOR_DATABASE = "C:\\DROID\\DATABASE";
    private static String DEFAULT_PROFILE_NAME = "DEFAULT_PROFILE";
    private static String DEFAULT_DESCRIPTION = "DROID_PROFILING";

    private static String myArgumentHelp =
            "The arguments should be \n" +
                    "    The disk volume location to run profiling tool on (Mandatory)\n" +
                    "        -V\n" +
                    "    The name of the profile to use (Optional)\n" +
                    "        -N\n" +
                    "    The location of the droid signature file (Mandatory)\n" +
                    "        -S\n" +
                    "    The path to where the database should be stored (Optional)\n" +
                    "        -D\n" +
                    "    The profile description (Optional)\n" +
                    "        -d\n";

    private String databasePath = "";
    private String profileName = "";
    private String signatureFileLocation = "";
    private String diskVolumeLocation = "";
    private String description = "";

    private static final ParameterDef[] PARAM_DEFS = {
            new StringDef('V', "diskVolumeLocation"),
            new StringDef('N', "profileName"),
            new StringDef('S', "signatureFileLocation"),
            new StringDef('D', "databasePath"),
            new StringDef('d', "description")
    };

    /**
     * Static entry method for profiling
     *
     * @param args the arguments
     */
    public static void main(String[] args) {
        if (args.length > 0) {
            ProfileEntryPoint inst = new ProfileEntryPoint();
            ArgumentProcessor.processArgs(args, PARAM_DEFS, inst);
            if (inst.diskVolumeLocation.length() == 0 || inst.signatureFileLocation.length() == 0) {
                log.info("Invalid arguments - Volume location (-V) and Signature file location (-S) must be specified\n" + myArgumentHelp);
                System.out.println("Invalid arguments - Volume location (-V) and Signature file location (-S) must be specified\n" + myArgumentHelp);
                System.exit(1);
            }
            if (inst.databasePath.length() == 0) {
                inst.databasePath = DEFAULT_PATH_FOR_DATABASE;
            }
            if (inst.profileName.length() == 0) {
                inst.profileName = DEFAULT_PROFILE_NAME;
            }
            if (inst.description.length() == 0) {
                inst.description = DEFAULT_DESCRIPTION;
            }
            ProfilingManager manager = inst.performProfilingChecks();
            if (manager != null) {
                try {
                    inst.commenceProfiling(manager);
                    log.info("Profiling completed successfully");
                    System.out.println("Profiling completed successfully");
                    System.exit(0);
                } catch (Exception e) {
                    e.printStackTrace();
                    log.error(e.toString());
                    log.error("Error starting profiling");
                    System.out.println("Error starting profiling");
                    System.exit(0);
                }
            }

        } else {
            System.out.println("Invalid arguments - Volume location (-V) and Signature file location (-S) must be specified\n" + myArgumentHelp);
            log.info("Invalid arguments - Volume location (-V) and Signature file location (-S) must be specified\n" + myArgumentHelp);
            System.exit(1);
        }
    }

    /**
     * Sets the manager and DB providing database path is correctly specified
     * @return  the manager
     */
    public ProfilingManager performProfilingChecks() {
        ProfilingManager manager = null;
        try{
        manager = new ProfilingManager(new File(databasePath));
        }catch(Exception ex){
            //ignore 
        }
        if (manager.isDBAccessSet()) {
            System.out.println("Error creating database at specified path");
            log.info("Error creating database at specified path");
            return null;
        } else {
            return manager;
        }
    }

    /**
     * Starts profiling
     * @param manager   the manager
     * @throws Exception    on error
     */
    public void commenceProfiling(ProfilingManager manager) throws Exception {
        Profile prof = manager.getProfileByName(profileName);
        if (prof == null) {
            prof = manager.createProfile(profileName, description);
        }
        String hostname = InetAddress.getLocalHost().getHostName();
        ProfileVolume vol = manager.createVolumeForProfileToStartNow(prof, diskVolumeLocation, hostname);
        manager.performProfilingOnDirectory(vol, signatureFileLocation);
        manager.labelVolumeComplete(vol);
        System.out.println("Total files: " + manager.getNumberOfFilesInVolume(vol));
        log.info("Total files: " + manager.getNumberOfFilesInVolume(vol));
        System.out.println("Total size: " + manager.getTotalSizeOfVolume(vol));
        log.info("Total size: " + manager.getTotalSizeOfVolume(vol));
        System.out.println("Average size: " + manager.getAverageSizeOfFileInVolume(vol));
        log.info("Average size: " + manager.getAverageSizeOfFileInVolume(vol));
        System.out.println("Max file: " + manager.getMaxSizeOfFileInVolume(vol));
        log.info("Max file: " + manager.getMaxSizeOfFileInVolume(vol));
        System.out.println("Min file: " + manager.getMinSizeOfFileInVolume(vol));
        log.info("Min file: " + manager.getMinSizeOfFileInVolume(vol));
        System.out.println("\n\nYear\t\tTotal Size (bytes)\t\tNum. Of Files");
        log.info("\n\nYear\t\tTotal Size (bytes)\t\tNum. Of Files");
        for (Object[] obj : manager.getVolumeYearInfo(vol)) {
            System.out.println(cleanObject(obj[0]) + "\t\t" + cleanObject(obj[1]) + "\t\t" + cleanObject(obj[2]));
            log.info(cleanObject(obj[0]) + "\t\t" + cleanObject(obj[1]) + "\t\t" + cleanObject(obj[2]));
        }
        System.out.println("\n\nPUID \tMIME\tName\tVersion\tNum. Of Files\tTotal Size (bytes)");
        log.info("\n\nPUID \tMIME\tName\tVersion\tNum. Of Files\tTotal Size (bytes)");
        for (Object[] obj : manager.getVolumeFormatInfo(vol)) {
            if (obj[2] == null) {
                obj[2] = "Unknown";
            }
            System.out.println(cleanObject(obj[0]) + "\t" + cleanObject(obj[1]) + "\t" + cleanObject(obj[2]) + "\t" + cleanObject(obj[3]) + "\t" + cleanObject(obj[4]) + "\t" + cleanObject(obj[5]) + "\t");
            log.info(cleanObject(obj[0]) + "\t" + cleanObject(obj[1]) + "\t" + cleanObject(obj[2]) + "\t" + cleanObject(obj[3]) + "\t" + cleanObject(obj[4]) + "\t" + cleanObject(obj[5]) + "\t");
        }
    }

    /**
     * Replaces null or blank with the word none
     * @param obj   the object
     * @return  object post cleaning
     */
    private Object cleanObject(Object obj) {
        if (obj == null || obj.equals("")) {
            return "none";
        } else {
            return obj;
        }
    }
}