/**
 *  � The National Archives 2005-2008.  All rights reserved.
 * See Licence.txt for full licence details.
 * <p/>
 *
 *  DROID DCS Profile Tool
 * <p/>
 * <p/>
 * Developed By:
 * Tessella Support Services
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * <p/>
 * email:  info@tessella.com
 * web:    www.tessella.com
 * <p/>
 * <p/>
 * Created Date:   12-Nov-2008
 */
package uk.gov.nationalarchives.droid.profile.domain;

import java.io.Serializable;
import java.io.File;
import java.util.*;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import uk.gov.nationalarchives.droid.MessageDisplay;

/**
 * This is the model for the summary data displayed as the user creates a profile
 */
public class VolumeSummary implements Serializable {

    private Integer numOfFiles;
    private Long totalSize;
    private Long maxSize;
    private Long minSize;
    private Double averageSize;
    private String currentFile;

    private Map<Integer, YearData> yearData;
    private Map<String, PuidData> puidData;

    /**
     * Default constructor. Initialises all fields
     */
    public VolumeSummary() {
        this.numOfFiles = 0;
        this.totalSize = 0L;
        this.maxSize = null;
        this.minSize = null;
        this.averageSize = 0.0;
        this.yearData = new ConcurrentSkipListMap<Integer, YearData>();
        this.puidData = new ConcurrentSkipListMap<String, PuidData>(new PuidComparator());
        this.currentFile = "";
    }

    /**
     * Getter
     *
     * @return total number of files in volume
     */
    public Integer getNumOfFiles() {
        return numOfFiles;
    }

    /**
     * Getter
     *
     * @return total size of volume
     */
    public Long getTotalSize() {
        return totalSize;
    }

    /**
     * Getter
     *
     * @return max size of volume
     */
    public Long getMaxSize() {
        return maxSize;
    }

    /**
     * Getter
     *
     * @return min size of volume
     */
    public Long getMinSize() {
        return minSize;
    }

    /**
     * Getter
     *
     * @return average size of volume
     */
    public Double getAverageSize() {
        return averageSize;
    }

    /**
     * Getter
     *
     * @return  the current file location
     */
    public String getCurrentFile() {
        return currentFile;
    }

    /**
     * Get the map of year data where the year is the key
     *
     * @return map
     */
    public Map<Integer, YearData> getYearData() {
        return yearData;
    }

    /**
     * Get the map of puid data where the puid is the key
     *
     * @return map
     */
    public Map<String, PuidData> getPuidData() {
        return puidData;
    }

    /**
     * Add a file and its formats to this summary
     *
     * @param fileObject the file object
     * @param formats    its formats
     */
    public boolean addFileAndFormats(FileObject fileObject, Set<Format> formats) {
        numOfFiles++;
        Long size = fileObject.getFileSize();
        totalSize += size;
        if (maxSize == null || size > maxSize) {
            maxSize = size;
        }
        if (minSize == null || size < minSize) {
            minSize = size;
        }
        averageSize = calculateNewAverage(numOfFiles, averageSize, size);
        currentFile = new File(fileObject.getPath()).getParent();

        Calendar cal = Calendar.getInstance();
        cal.setTime(fileObject.getLastModified());
        int year = cal.get(Calendar.YEAR);
        if(yearData.get(year) == null) {
            YearData yearDat = new YearData(fileObject);
            yearData.put(yearDat.getYear(), yearDat);
        } else {
            yearData.get(year).addFileToTotals(fileObject);
        }
        try{
        for (Format format : formats) {
            if (puidData.get(format.getPuid()) == null) {
                PuidData data = new PuidData(fileObject, format);
                puidData.put(data.getPuid(), data);
            } else {
                puidData.get(format.getPuid()).addFileOfSameFormat(fileObject);
            }
        }
        }catch(Exception ex){
            try {
                MessageDisplay.generalErrorjDialog("An error occured during profiling.Please make sure your signature file is up to date");
            } catch (Exception ex1) {
                
            }
            return false;
        }
        return true;
                
    }

    /**
     * Static method for on the fly average calculating when one item has been added
     * and the previous average is available
     *
     * @param newTotalItems the new total of items (including the added one)
     * @param oldAverage    the average previously
     * @param newValue      the new value
     * @return the new average
     */
    public static Double calculateNewAverage(Integer newTotalItems, Double oldAverage, Long newValue) {
        return (((newTotalItems.doubleValue() - 1.0) / newTotalItems.doubleValue()) * oldAverage + (newValue.doubleValue() / newTotalItems.doubleValue()));
    }

    /**
     * Compares puid names and sorts them first alphabetically by prefix, and then numerically by suffix
     */
    private class PuidComparator implements Comparator<String> {
        /**
         * Compares the two puids
         *
         * @param format1 the first puid
         * @param format2 the second puid
         * @return sort
         */
        public int compare(String format1, String format2) {
            String[] format1split = format1.split("/");
            String[] format2split = format2.split("/");
            if (format1split.length != format2split.length) {
                return (format1split.length - format2split.length);
            }
            if (format1split[0].compareToIgnoreCase(format2split[0]) != 0) {
                return format1split[0].compareToIgnoreCase(format2split[0]);
            } else {
                Integer format1Int = Integer.decode(format1split[1]);
                Integer format2Int = Integer.decode(format2split[1]);
                return (format1Int - format2Int);
            }
        }
    }
}
