/**
 * (c) Copyright  2008 Tessella Support Services plc.  
 * All rights reserved. 
 * <p/>
 * Project: 5890
 * DROID DCS Profile Tool
 * <p/>
 * <p/>
 * Developed By:
 * Tessella Support Services
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * <p/>
 * email:  info@tessella.com
 * web:    www.tessella.com
 * <p/>
 * <p/>
 * Created By:     Lawrence Owusu (owul)
 * Created Date:   2-Nov-2008
 */
package uk.gov.nationalarchives.droid.profile.service;


import net.sf.jasperreports.engine.*;
import net.sf.jasperreports.engine.fill.JRFileVirtualizer;
import net.sf.jasperreports.view.JasperViewer;
import uk.gov.nationalarchives.droid.AnalysisController;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.sf.jasperreports.engine.export.JExcelApiExporter;
import net.sf.jasperreports.engine.export.JRCsvExporter;
import net.sf.jasperreports.engine.export.JRXlsExporterParameter;
import org.apache.log4j.Logger;

/**
 * This is the primary service class for report generation
 *
 * @author owul
 */
public class ReportManager {

    private AnalysisController controller = new AnalysisController();
    static Logger log = Logger.getLogger(ReportManager.class);
   
    
    
    public static List<String> getReports() {
        return new ReportManager().getReportNames();
    }

    public static void saveReportToFile(String report, String outputFile, String type, List<String> profileName, ProfilingManager manager, String filterString) {

        JasperPrint jasperPrint = new ReportManager().getJasperReport(report,  profileName,  manager, filterString);
        try {
           if (type.equalsIgnoreCase("PDF"))
                JasperExportManager.exportReportToPdfFile(jasperPrint, outputFile);
            else if(type.equalsIgnoreCase("XML"))
                JasperExportManager.exportReportToXmlFile(jasperPrint, outputFile, true);
            else if(type.equalsIgnoreCase("CSV")){
                 JRCsvExporter exporter = new JRCsvExporter();
                 exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
                 exporter.setParameter(JRExporterParameter.OUTPUT_FILE_NAME, outputFile);
                 exporter.exportReport();
            }else if(type.equalsIgnoreCase("XLS")){
                 JExcelApiExporter exporter = new JExcelApiExporter();
                 exporter.setParameter(JRExporterParameter.JASPER_PRINT, jasperPrint);
                 exporter.setParameter(JRExporterParameter.OUTPUT_FILE_NAME, outputFile);
                 exporter.setParameter(JRXlsExporterParameter.IS_ONE_PAGE_PER_SHEET, false);
                 exporter.setParameter(JRXlsExporterParameter.IS_WHITE_PAGE_BACKGROUND, false);
                 exporter.setParameter(JRXlsExporterParameter.IS_REMOVE_EMPTY_SPACE_BETWEEN_COLUMNS, true);
                 exporter.setParameter(JRXlsExporterParameter.IS_REMOVE_EMPTY_SPACE_BETWEEN_ROWS, true);
                 exporter.setParameter(JRXlsExporterParameter.IS_DETECT_CELL_TYPE, true);
                 exporter.exportReport();
            }
           System.out.println("\nReport generation complete.  "+ outputFile);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("An error occured in generating report :"+outputFile);
            System.out.println("An error occured in generating report :"+outputFile);
        }

    }

    /**
     * This method is used to retrieve the report names from the report file names
     *
     * @return A list containing the report names
     */
    public List<String> getReportNames() {
        List<String> reportNames = new ArrayList<String>();
        File reportPath = new File(getReportPath());
        //extract report name by stripping the extension and path details from the full path names
        if (reportPath.isDirectory()) {
            File[] reports = reportPath.listFiles();
            for (File report : reports) {
                if (report.getName().trim().endsWith(".jrxml") && !report.getName().trim().contains("subreport")) {
                    try {
                        String strReportType = "";
                        if(report.getName().trim().contains("_")){
                                strReportType = report.getName().trim().substring(0, report.getName().indexOf("_"));
                        }else{
                            strReportType = report.getName().trim().substring(0, report.getName().indexOf(".jrxml"));
                        }
                        if(!reportNames.contains(strReportType.trim())){
                            reportNames.add(strReportType);
                        }
                    } catch (Exception e) {
                        System.out.println(e.getMessage());
                        log.error(e.toString());
                    }
                }
            }
        } else {
            System.out.println("Report path is not a directory");
            log.error("Report path is not a directory");
        }
        if(reportNames.size() > 0)
            Collections.sort(reportNames);
        return reportNames;
    }

    public JasperPrint getJasperReport(String reportName, List<String> profileNames, ProfilingManager manager, String filter) {

        JasperPrint jasperPrint = null;
        JRFileVirtualizer JRFile = null;
        try {

            //set the jasper reports temporary directory
            File tempDir = null;  
            boolean tempDirSet = controller.setJRPropertiesTempDir();
            if(tempDirSet){
                tempDir = new File(controller.getReportTempDir());                      
            }
            
            
            String reportPath = getReportPath();
      
            Map<String, Object> parameters = new HashMap<String, Object>();

            File reportPathFile = new File(reportPath);

            //The Entire Database Summary report is the only one that
            //have subreports. This implies that a SUBREPORT_DIR parameter has to be filled
            // with the path of the directory in which the subreport(s) resides.
            //Secondly these subreports must be compiled before the main report are compiled
            // and executed.

            //Each of the reports needs to specify the value for the DIR parameter which specifies
            // the directory in which the National Archive Logo resides
            //This is placed in the Reports folder.
            if (reportName.trim().equalsIgnoreCase("Entire Database Summary")) {
                
                File reportFile = new File(reportPathFile, reportName + "_subreport.jrxml");
                if(tempDirSet){
                    //parameters.put("SUBREPORT_DIR", reportPathFile.getAbsolutePath() + File.separator);
                    parameters.put("SUBREPORT_DIR", tempDir.getAbsolutePath() + File.separator);
                    JasperCompileManager.compileReportToFile(reportFile.getAbsolutePath(),tempDir.getAbsolutePath() + File.separator+"Entire Database Summary_subreport.jasper");
                }else{
                    parameters.put("SUBREPORT_DIR", reportPathFile.getAbsolutePath() + File.separator);
                    JasperCompileManager.compileReportToFile(reportFile.getAbsolutePath());
                }
                parameters.put("Database",controller.getProfileDatabasePath());

            } else {
                parameters.put("Filter", filter);
            }
            parameters.put("DIR", reportPathFile.getAbsolutePath() + File.separator);
            File actualReport = new File(reportPathFile, reportName + ".jrxml");



            if (reportName.toUpperCase().contains("PROFILE")) {
                String profileName = "";
                if (profileNames != null) {
                    profileName = "(";
                    for (int i = 0; i < profileNames.size(); i++) {
                        if (i != 0) {
                            profileName += ",";
                        }
                        profileName += ("'" + profileNames.get(i).toUpperCase().trim() + "'");
                    }
                    profileName += ")";
                    parameters.put("ProfileName", profileName.trim());
                }
            }

            // reduce memory usage in laoding report 
            if(tempDirSet){
                JRFile = new JRFileVirtualizer(10,tempDir.getAbsolutePath());
            }else{
                JRFile = new JRFileVirtualizer(10);
            }
            parameters.put(JRParameter.REPORT_VIRTUALIZER,JRFile );


            //Compile the main report
            JasperReport jasperReport = JasperCompileManager.compileReport(actualReport.getAbsolutePath());


            //Execute the report and store the executed object as a Jasper print
            jasperPrint = manager.fillReport(jasperReport, parameters);
        } catch (JRException e) {
            System.out.println(e.toString() + "Could not compile report:  " + reportName);
            log.error(e.toString() + "Could not compile report:  " + reportName);
        }finally{
            if(JRFile != null){
               // JRFile.cleanup();
            }
        }
        return jasperPrint;
    }

    /**
     * Display the final report
     *
     * @param reportPath  The name of the report. This is used to reconstruct the path of the report
     * @param profileName This is the name of the profile which is passed as a parameter for generating the report
     * @param manager     The profiling manager object in contains a method that returns the Jasper Print
     */
    public JasperViewer showReport(String reportPath, List<String> profileName, ProfilingManager manager, String filter) {
        
        JasperViewer viewer = null;
        try {

            //Execute the report and store the executed object as a Jasper print
            JasperPrint jasperPrint = getJasperReport(reportPath, profileName, manager, filter);
            if (jasperPrint != null) {
                //Display the report and set the Exit on close parameter to false, so that the dialo
                //box is not closed after the report is closed
                viewer = new JasperViewer(jasperPrint, false);
                
                //if report has filters remove the "_" character to get the report Name
                if(reportPath.contains("_")){
                    reportPath = reportPath.substring(0,reportPath.indexOf("_"));
                }
                viewer.setTitle("DROID REPORTING: " + reportPath);
                viewer.setFocusable(true);
                viewer.toFront();
            }
            //  viewer.setAlwaysOnTop(true);
            try {
                viewer.setIconImage(javax.imageio.ImageIO.read(getClass().getResource("/uk/gov/nationalarchives/droid/GUI/Icons/DROID16.gif")));
            } catch (java.io.IOException e) {
                //Silently ignore exception
            }
            
        }
        catch (Exception e) {
            
            System.out.println(e.toString()+"Could not compile report:  " + reportPath);
            log.error(e.toString()+"Could not compile report:  " + reportPath);
        }
        return viewer;
    }

    /**
     * Get path of reports folder
     *
     * @return Returns the path in which the reports reside from the configuration file
     */
    public String getReportPath() {
        String rptPath = "";
        try {
            controller.readConfiguration();
            rptPath = controller.getProfileReportPath();
            if (rptPath == null) {
                rptPath = "";
            }
        } catch (Exception ex) {
            System.out.println("Error in retrieving Report Directory path");
            log.error("Error in retrieving Report Directory path");
        }
        return rptPath;
    }
    

}
