/*
 * 
 *  The National Archives 2005-2006.  All rights reserved.
 * See Licence.txt for full licence details.
 *
 * Developed by:
 * Tessella Support Services plc
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * email: info@tessella.com
 * web:   www.tessella.com
 *
 * Project Number:  Tessella/NPD/4950
 * Author (loadComponents method: owul )                
 */
package uk.gov.nationalarchives.droid.stats.GUI;

import uk.gov.nationalarchives.droid.profile.domain.ProfileVolume;
import uk.gov.nationalarchives.droid.profile.domain.PuidData;
import uk.gov.nationalarchives.droid.profile.domain.VolumeSummary;
import uk.gov.nationalarchives.droid.profile.domain.YearData;
import uk.gov.nationalarchives.droid.profile.service.ProfilingManager;
import uk.gov.nationalarchives.droid.AnalysisController;

import javax.swing.*;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;
import javax.swing.table.AbstractTableModel;
import java.awt.*;
import java.awt.event.*;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.*;
import uk.gov.nationalarchives.droid.MessageDisplay;

/**
 * Dialog that watches a given volume, and regularly updates with the most recent data
 *
 * @author trir
 */
public class StatsResultDialog extends javax.swing.JDialog {

    
    /**
     * Add files button pressed
     */
    public static final int ACTION_ADD = 1;
    /**
     * Cancel button pressed
     */
    public static final int ACTION_CANCEL = 0;
    /**
     * Timer to regularly update display during profile generation
     */
    private Timer timer;
    private ProfileVolume volume;
    private Log log = LogFactory.getLog(this.getClass());
    /**
     * Reference to parent controller
     */
    private AnalysisController analysisControl;
    private ProfilingManager manager;
    private VolumeSummary summary;

    /**
     * Creates new form StatsResultDialog
     *
     * @param parent          parent GUI frame
     * @param modal           whether this should block interaction with the main window
     * @param vol             the relevant volume
     * @param analysisControl main application controller
     */
    private StatsResultDialog(Frame parent, boolean modal, ProfileVolume vol, AnalysisController analysisControl) {
        super(parent, modal);
        this.volume = vol;
        this.analysisControl = analysisControl;
        this.manager = analysisControl.getManager();
        setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
        //Intialise the form components
        loadComponents();
        makeDisposeOnEscapeKey(this);
        setMnemonics();
        setMinimumSize(new Dimension(510, this.getHeight()));
        
        //Set "Pause" as the default button
        this.getRootPane().setDefaultButton(jButtonPause);

        //This window should open centralised on the main application window.
        this.setLocationRelativeTo(parent);

        this.addWindowListener(new WindowAdapter() {

            public void windowClosing(WindowEvent e) {
                 if (!(StatsResultDialog.this.analysisControl.isAnalysisCancelled() || StatsResultDialog.this.analysisControl.isAnalysisComplete())) {
                    int outcome = JOptionPane.showConfirmDialog(StatsResultDialog.this, "The profiling is not complete, closing this will cause it to be cancelled. Are you sure you want to cancel?",
                            "Cancel profiling?", JOptionPane.YES_NO_OPTION, JOptionPane.INFORMATION_MESSAGE);
                    if (outcome == JOptionPane.YES_OPTION) {
                        StatsResultDialog.this.analysisControl.cancelAnalysis();
                        if (manager != null) {
                            //manager.cancelWalker();
                            //volume.setDateCompleted(new Date());
                            manager.saveVolume(volume);
                            StatsResultDialog.this.dispose();
                            log.info("Profiling was cancelled");
                        }
                    }
                } else {
                    StatsResultDialog.this.dispose();
                }
            }
        });
        // Begin querying database for new results
        startTimer();
    }

    /**
     * Update the summary statistics with values from the DB
     */
    private void updateSummaryFields() {
        if (summary != null) {
            jTextFieldResults[0].setText(" " + String.valueOf(summary.getCurrentFile()));
            jTextFieldResults[1].setText(" " + String.valueOf(summary.getNumOfFiles()));
            jTextFieldResults[2].setText(" " + String.valueOf(summary.getTotalSize()));
            jTextFieldResults[3].setText(" " + String.valueOf(summary.getMinSize()));
            jTextFieldResults[4].setText(" " + String.valueOf(summary.getMaxSize()));
            DecimalFormat twoDP = new DecimalFormat("#0.00");
            Double avg;
            if (summary.getAverageSize() == null) {
                avg = 0.0;
            } else {
                avg = summary.getAverageSize();
            }
            jTextFieldResults[5].setText(" " + String.valueOf(twoDP.format(avg)));
        }
    }

    /**
     * Begin timer to update GUI from the model - runs the update from the volume summary
     */
    private void startTimer() {
        timer = new Timer(2000, new ActionListener() {

            public void actionPerformed(ActionEvent evt) {
                //We use an in memory summary to keep the GUI and DB responsive - the total amount of data in memory should
                // never be more than ~1000 strings.
                //First we check to see if the walker has finished or been cancelled
                if ((analysisControl.isAnalysisComplete() || analysisControl.isAnalysisCancelled())) {
                    //if so stop the timer
                    timer.stop();
                    // Update all tables with complete data
                    updateFromSummary();
                    jButtonCancel.setEnabled(false);
                    jButtonPause.setEnabled(false);
                    jButtonExit.setEnabled(true);
                    // Check whether any files were found
                    if (analysisControl.getManager().getNumberOfFilesInVolume(volume) == 0) {
                        JOptionPane.showMessageDialog(StatsResultDialog.this,
                                "No files were found in the location you selected",
                                "No files found",
                                JOptionPane.INFORMATION_MESSAGE);
                        log.info("No files were found in the location you selected for profiling");
                    } else {
                        if (!analysisControl.isAnalysisCancelled()) {
                            JOptionPane.showMessageDialog(StatsResultDialog.this,
                                    "Profiling has completed successfully",
                                    "Profiling Complete",
                                    JOptionPane.INFORMATION_MESSAGE);
                            log.info("Profiling was successfully completed");
                        }
                    }
                }
                //Otherwise update all the fields
                updateFromSummary();
            }
        });
        timer.start();
    }

    private void updateFromSummary() {
        if (summary == null && analysisControl.getManager() != null) {
            summary = analysisControl.getManager().getWalkerSummary();
        }
        if (summary != null) {
            updateSummaryFields();
            ((ByYearTableModel) jTableByYear.getModel()).updateSortedYears();
            ((ByFormatTableModel) jTableByFormat.getModel()).updateFormats();
            ((AbstractTableModel) jTableByYear.getModel()).fireTableDataChanged();
            ((AbstractTableModel) jTableByFormat.getModel()).fireTableDataChanged();
        }
    }

    /**
     * Utility method to cancel dialog when escape button pressed
     *
     * @param rootPane dialog to close on escape key press
     */
    private static void makeDisposeOnEscapeKey(final RootPaneContainer rootPane) {
        //Create action to dispose
        Action action = new AbstractAction() {

            public void actionPerformed(ActionEvent arg0) {
                ((Window) rootPane).dispose();
            }
        };
        //Get keystroke for escape key
        KeyStroke stroke = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);
        //Add escape key to action map for dialog
        rootPane.getRootPane().getActionMap().put(action, action);
        rootPane.getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(stroke, action);
    }

    /**
     * Creates and shows a File select dialog
     *
     * @param frameComp Component dialog is owned by
     * @param vol       the affected volume
     * @param ac        analysis controller
     */
    public static void showDialog(Component frameComp, ProfileVolume vol, AnalysisController ac) {
        Frame f = JOptionPane.getFrameForComponent(frameComp);
        StatsResultDialog dialog = new StatsResultDialog(f, true, vol, ac);
        MessageDisplay.initialiseMainPane(dialog);
        dialog.setVisible(true);
    }

    /**
     * Set the mnemonics (Keyboard shortcuts) for menu items on this form
     * Can only be called after initComponents()
     */
    private void setMnemonics() {
        jButtonPause.setMnemonic('E');
        jButtonCancel.setMnemonic('S');
    }

    /**
     * This method is called from within the constructor to
     * initialize the form.
     */
    private void loadComponents() {

        setTitle("Profile");
        setName("StatsResultDialog");

        JPanel jPanelMaster = new JPanel();
        jPanelMaster.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(BorderFactory.createEtchedBorder(EtchedBorder.LOWERED), "Report"),
                BorderFactory.createEmptyBorder(0, 0, 0, 0)));


        // Create results summary fields
        JPanel jPanelSummary = new JPanel();
        JLabel[] jLabelSummaryLabels = new JLabel[6];
        jTextFieldResults = new JTextField[6];



        for (int i = 0; i < jLabelSummaryLabels.length; i++) {
            jLabelSummaryLabels[i] = new JLabel();
            jTextFieldResults[i] = new JTextField();
            jTextFieldResults[i].setBorder(BorderFactory.createLineBorder(Color.black));
            jTextFieldResults[i].setEditable(false);
            jTextFieldResults[i].setBackground(Color.white);

        }

        jLabelSummaryLabels[0].setText("Currently scanning: ");
        jLabelSummaryLabels[1].setText("Total readable files: ");
        jLabelSummaryLabels[2].setText("Total file size (bytes): ");
        jLabelSummaryLabels[3].setText("Smallest file's size (bytes): ");
        jLabelSummaryLabels[4].setText("Largest file's size (bytes): ");
        jLabelSummaryLabels[5].setText("Mean file size (bytes): ");


        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanelSummary);
        jPanelSummary.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
                jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jPanel1Layout.createSequentialGroup()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addComponent(jLabelSummaryLabels[0]).addComponent(jLabelSummaryLabels[1])
                .addComponent(jLabelSummaryLabels[2]).addComponent(jLabelSummaryLabels[3])
                .addComponent(jLabelSummaryLabels[4]).addComponent(jLabelSummaryLabels[5]))
                .addGap(80, 80, 80)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addComponent(jTextFieldResults[0], javax.swing.GroupLayout.DEFAULT_SIZE, 160, Short.MAX_VALUE)
                .addComponent(jTextFieldResults[1], javax.swing.GroupLayout.DEFAULT_SIZE, 160, Short.MAX_VALUE)
                .addComponent(jTextFieldResults[2], javax.swing.GroupLayout.DEFAULT_SIZE, 160, Short.MAX_VALUE)
                .addComponent(jTextFieldResults[3], javax.swing.GroupLayout.DEFAULT_SIZE, 160, Short.MAX_VALUE)
                .addComponent(jTextFieldResults[4], javax.swing.GroupLayout.DEFAULT_SIZE, 160, Short.MAX_VALUE)
                .addComponent(jTextFieldResults[5], javax.swing.GroupLayout.DEFAULT_SIZE, 160, Short.MAX_VALUE))));

        jPanel1Layout.setVerticalGroup(
                jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(jPanel1Layout.createSequentialGroup() 
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                .addComponent(jLabelSummaryLabels[0])
                .addComponent(jTextFieldResults[0], javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                .addComponent(jLabelSummaryLabels[1])
                .addComponent(jTextFieldResults[1], javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                .addComponent(jLabelSummaryLabels[2])
                .addComponent(jTextFieldResults[2], javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                .addComponent(jTextFieldResults[3], javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addComponent(jLabelSummaryLabels[3])).addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                .addComponent(jTextFieldResults[4], javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addComponent(jLabelSummaryLabels[4])).addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                .addComponent(jTextFieldResults[5], javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addComponent(jLabelSummaryLabels[5])) 
                ));

  
        // Create by year table        
        JPanel byYearPanel = new JPanel();
        byYearPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(BorderFactory.createEmptyBorder(), "Files by modified date", TitledBorder.LEFT, TitledBorder.DEFAULT_POSITION),
                BorderFactory.createEmptyBorder(5, 0, 5, 0)));
        jTableByYear = new JTable(new ByYearTableModel());
        
        jTableByYear.setPreferredScrollableViewportSize(jTableByYear.getPreferredSize());
       
        JScrollPane jScrollPaneByYear = new JScrollPane();
        jScrollPaneByYear.setViewportView(jTableByYear);
        jScrollPaneByYear.getViewport().setBackground(Color.WHITE);

        javax.swing.GroupLayout jPanel2Layout = new javax.swing.GroupLayout(byYearPanel);
        byYearPanel.setLayout(jPanel2Layout);
        jPanel2Layout.setHorizontalGroup(
                jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(jPanel2Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jScrollPaneByYear, javax.swing.GroupLayout.DEFAULT_SIZE, 335, Short.MAX_VALUE)
                .addContainerGap()));
        jPanel2Layout.setVerticalGroup(
                jPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(jPanel2Layout.createSequentialGroup() 
                .addGap(8, 8, 8).
                addComponent(jScrollPaneByYear, javax.swing.GroupLayout.PREFERRED_SIZE, 200, javax.swing.GroupLayout.PREFERRED_SIZE) 
                .addGap(8, 8, 8)));


        // Create by format table
        JPanel byFormatPanel = new JPanel();
        byFormatPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(BorderFactory.createEmptyBorder(), "Files by format"),
                BorderFactory.createEmptyBorder(5, 0, 5, 0)));
        jTableByFormat = new JTable(new ByFormatTableModel());
        jTableByFormat.setPreferredScrollableViewportSize(jTableByFormat.getPreferredSize());
        JScrollPane jScrollPaneByFormat = new JScrollPane();
        jScrollPaneByFormat.setViewportView(jTableByFormat);
        jScrollPaneByFormat.getViewport().setBackground(Color.WHITE);
  

        javax.swing.GroupLayout jPanel3Layout = new javax.swing.GroupLayout(byFormatPanel);
        byFormatPanel.setLayout(jPanel3Layout);
        jPanel3Layout.setHorizontalGroup(
                jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(jPanel3Layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(jScrollPaneByFormat, javax.swing.GroupLayout.DEFAULT_SIZE, 335, Short.MAX_VALUE)
                .addContainerGap()));
        jPanel3Layout.setVerticalGroup(
                jPanel3Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(jPanel3Layout.createSequentialGroup() 
                .addGap(8, 8, 8)
                .addComponent(jScrollPaneByFormat, javax.swing.GroupLayout.PREFERRED_SIZE, 200, javax.swing.GroupLayout.PREFERRED_SIZE)));





        //create a new panel comprising of the summary, year and format panels
        javax.swing.GroupLayout jMasterLayout = new javax.swing.GroupLayout(jPanelMaster);
        jPanelMaster.setLayout(jMasterLayout);
        jMasterLayout.setHorizontalGroup(
                jMasterLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, jMasterLayout.createSequentialGroup()
                .addContainerGap()
                .addGroup(jMasterLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                .addComponent(jPanelSummary, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addComponent(byYearPanel, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addComponent(byFormatPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addContainerGap()));
        jMasterLayout.setVerticalGroup(
                jMasterLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(jMasterLayout.createSequentialGroup()
                .addComponent(jPanelSummary, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addGap(10, 10, 10).addComponent(byYearPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addComponent(byFormatPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)));




        //create a panel for the buttons
        JPanel jPanelActions = new JPanel();
        jButtonPause = new JButton();
        jButtonCancel = new JButton();
        jButtonExit = new JButton();

        // Add pause button
        jButtonPause.setText("Pause");
        jButtonPause.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent evt) {
                jButtonPauseActionPerformed();
            }
        });
        jPanelActions.add(jButtonPause);

        // Add cancel button
        jButtonCancel.setText("Cancel");
        jButtonCancel.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent evt) {
                jButtonCancelActionPerformed();
            }
        });
        jPanelActions.add(jButtonCancel);

        // Add Close button
        jButtonExit.setText("Close");
        jButtonExit.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent evt) {
                jButtonExitActionPerformed();
            }
        });
        jButtonExit.setEnabled(false);
        jPanelActions.add(jButtonExit);


        //create the final layout for the form comprising of all the panels
        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
                layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup() 
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                .addComponent(jPanelActions, javax.swing.GroupLayout.Alignment.LEADING, 500, 500, Short.MAX_VALUE)
                .addComponent(jPanelMaster, javax.swing.GroupLayout.Alignment.LEADING, 500, 500, Short.MAX_VALUE))));
        layout.setVerticalGroup(
                layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING).addGroup(layout.createSequentialGroup()
                .addComponent(jPanelMaster, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addComponent(jPanelActions, 35, 35, javax.swing.GroupLayout.PREFERRED_SIZE)));


        
      
        pack();
        
    }

    /**
     * Exit the window
     */
    private void jButtonExitActionPerformed() {
        this.dispose();
    }

    /**
     * Cancels the profiling
     */
    private void jButtonCancelActionPerformed() {
        analysisControl.cancelAnalysis();
        jButtonCancel.setEnabled(false);
        jButtonPause.setEnabled(false);
        jButtonExit.setEnabled(true);
    }

    /**
     * Pause the profiling
     */
    private void jButtonPauseActionPerformed() {
        Boolean pause = analysisControl.isAnalysisPaused();
        if (pause.equals(Boolean.TRUE)) {
            analysisControl.restartAnalysis();
            jButtonPause.setText("Pause");
             log.info("Profiling was restarted");
        }
        if (pause.equals(Boolean.FALSE)) {
            analysisControl.pauseAnalysis();
            jButtonPause.setText("Resume");
            log.info("Profiling was paused");
        }
    }
    private JButton jButtonPause;
    private JButton jButtonCancel;
    private JButton jButtonExit;
    private JTextField[] jTextFieldResults;
    private JTable jTableByYear;
    private JTable jTableByFormat;

    /**
     * Table model to show stats by year
     */
    private class ByYearTableModel extends AbstractTableModel {

        private String[] columnNames = {"Last Modified Year", "Total Size (bytes)", "Number of Files"};
        private List<Object[]> years;

        /**
         * Constructor - initialises the list
         */
        public ByYearTableModel() {
            years = new ArrayList<Object[]>();
        }

        @Override
        public String getColumnName(int column) {
            return columnNames[column];
        }

        /**
         * Returns number of columns in the table
         * PUID,Format,Status,Version,Warnings
         *
         * @return number of columns in table
         */
        public int getColumnCount() {
            return columnNames.length;
        }

        /**
         * Gets the number of rows displayed in the table
         *
         * @return Number of files held by the filelist
         */
        public int getRowCount() {
            return years.size();
        }

        /**
         * Gets the object for a specified cell
         *
         * @param row row id
         * @param col column id
         * @return object to display in cell
         */
        public Object getValueAt(int row, int col) {
            if (years.size() == 0) {
                return "";
            }
            try {
                return years.get(row)[col];
            } catch (Exception e) {
                return "";
            }
        }

        /**
         * Get the files' years logged, sorted from old to new
         */
        public void updateSortedYears() {
            List<Object[]> tempYears = new ArrayList<Object[]>();
            for (YearData tempData : summary.getYearData().values()) {
                Object[] temp = new Object[]{tempData.getYear(), tempData.getTotalSize(), tempData.getNumOfFiles()};
                tempYears.add(temp);
            }
            years = tempYears;
        }

        /**
         * Get the Class for a column
         *
         * @param col column to find class
         * @return Class type found
         */
        public Class getColumnClass(int col) {
            try {
                return getValueAt(0, col).getClass();
            } catch (NullPointerException e) {
                return "".getClass();
            }
        }

        /**
         * Determines whether table cell is editable
         * ALWAYS returns false
         *
         * @param row Row of cell
         * @param col Column of cell
         * @return true if cell editable , false otherwise
         */
        public boolean isCellEditable(int row, int col) {
            return false;
        }

        /**
         * Sets the value for a specific cell
         * DOES NOTHING IN THIS IMPLEMENTATION
         *
         * @param aValue Object to set in cell
         * @param row    Row of cell to enter object
         * @param column Column of cell to enter object
         */
        public void setValueAt(Object aValue, int row, int column) {
            //We do not allow cell editing in this table model
        }
    }

    /**
     * Table model to show stats by format
     */
    private class ByFormatTableModel extends AbstractTableModel {

        private String[] columnNames = new String[]{"PUID", "MIME", "Format", "Version", "Number of Files", "Total Bytes"};
        private List<Object[]> formats;

        /**
         * Constructor - intitilises the map, and sets the column headers
         */
        public ByFormatTableModel() {
            this.formats = new ArrayList<Object[]>();
        }

        @Override
        public String getColumnName(int column) {
            return columnNames[column];
        }

        /**
         * Returns number of columns in the table
         *
         * @return number of columns in table
         */
        public int getColumnCount() {
            return columnNames.length;
        }

        /**
         * Gets the number of rows displayed in the table
         *
         * @return Number of files held by the filelist
         */
        public int getRowCount() {
            return formats.size();
        }

        /**
         * Gets the object for a specified cell
         *
         * @param row row id
         * @param col column id
         * @return object to display in cell
         */
        public Object getValueAt(int row, int col) {
            if (formats.size() == 0) {
                return "";
            }
            if (col == 2) {
                Object type = formats.get(row)[2];
                if (type == null) {
                    return "Unknown format";
                } else {
                    return type;
                }
            } else {
                try {
                    return formats.get(row)[col];
                } catch (Exception e) {
                    return "";
                }
            }
        }

        /**
         * Get the Class for a column
         *
         * @param col column to find class
         * @return Class type found
         */
        public Class getColumnClass(int col) {
            try {
                return getValueAt(0, col).getClass();
            } catch (NullPointerException e) {
                return "".getClass();
            }
        }

        /**
         * Update the formats logged
         */
        public void updateFormats() {
            List<Object[]> formatTemp = new ArrayList<Object[]>();
            for (PuidData data : summary.getPuidData().values()) {
                Object[] temp = new Object[]{data.getPuid(), data.getMimeType(), data.getName(), data.getVersion(), data.getNumOfFiles(), data.getTotalSize()};
                formatTemp.add(temp);
            }
            formats = formatTemp;
        }

        /**
         * Determines whether table cell is editable
         * ALWAYS returns false
         *
         * @param row Row of cell
         * @param col Column of cell
         * @return true if cell editable , false otherwise
         */
        public boolean isCellEditable(int row, int col) {
            return false;
        }

        /**
         * Sets the value for a specific cell
         * DOES NOTHING IN THIS IMPLEMENTATION
         *
         * @param aValue Object to set in cell
         * @param row    Row of cell to enter object
         * @param column Column of cell to enter object
         */
        public void setValueAt(Object aValue, int row, int column) {
            //We do not allow cell contents editing in this model
        }
    }
}