/*
 * � The National Archives 2005-2006.  All rights reserved.
 * See Licence.txt for full licence details.
 *
 * Developed by:
 * Tessella Support Services plc
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * http://www.tessella.com
 *
 * Tessella/NPD/4305
 * PRONOM 4
 *
 * $Id: CmdController.java,v 1.7 2006/03/13 15:15:26 linb Exp $
 * 
 * $Log: CmdController.java,v $
 * Revision 1.7  2006/03/13 15:15:26  linb
 * Changed copyright holder from Crown Copyright to The National Archives.
 * Added reference to licence.txt
 * Changed dates to 2005-2006
 *
 * Revision 1.6  2006/02/13 11:27:29  linb
 * - Correct spelling in error message
 *
 * Revision 1.5  2006/02/08 12:03:37  linb
 * - add more comments
 *
 * Revision 1.4  2006/02/08 11:45:48  linb
 * - add support for streams
 *
 * Revision 1.3  2006/02/08 08:56:35  linb
 * - Added header comments
 *
 *
 * $History: CmdController.java $
 * 
 * *****************  Version 8  *****************
 * User: Walm         Date: 20/10/05   Time: 15:31
 * Updated in $/PRONOM4/FFIT_SOURCE/commandLine
 * When using the web service, check whether or not connection was
 * successful, and if not, display a helpful message to user
 * 
 * *****************  Version 7  *****************
 * User: Walm         Date: 12/05/05   Time: 12:12
 * Updated in $/PRONOM4/FFIT_SOURCE/commandLine
 * read configuration file when running -C and -D options
 * 
 * *****************  Version 6  *****************
 * User: Walm         Date: 5/05/05    Time: 10:29
 * Updated in $/PRONOM4/FFIT_SOURCE/commandLine
 * Save to file at end of run has been moved to setAnalysisComplete() in
 * the AnalysisController
 * 
 * *****************  Version 5  *****************
 * User: Walm         Date: 26/04/05   Time: 17:29
 * Updated in $/PRONOM4/FFIT_SOURCE/commandLine
 * change name from uk to DROID
 * 
 * *****************  Version 4  *****************
 * User: Walm         Date: 5/04/05    Time: 18:08
 * Updated in $/PRONOM4/FFIT_SOURCE/commandLine
 * review headers
 * 
 * *****************  Version 3  *****************
 * User: Walm         Date: 31/03/05   Time: 15:26
 * Updated in $/PRONOM4/FFIT_SOURCE/commandLine
 * Download a signature file
 * + some changes of output for other options
 *
 * Created on 15 March 2005, 14:43
 */

package uk.gov.nationalarchives.droid.commandLine;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import uk.gov.nationalarchives.droid.AnalysisController;
import uk.gov.nationalarchives.droid.MessageDisplay;
import uk.gov.nationalarchives.droid.binFileReader.UrlByteReader;
import uk.gov.nationalarchives.droid.profile.ProfileEntryPoint;
import uk.gov.nationalarchives.droid.profile.service.ReportManager;
import uk.gov.nationalarchives.droid.xmlReader.PronomWebService;

import java.io.File;
import java.util.List;

/**
 * @author walm
 */
public class CmdController {

    private Log log = LogFactory.getLog(this.getClass());

    private static final String myArgumentHelp =
            "The arguments should be \n" +
                    "    Display the command line options\n" +
                    "        -H\n" +
                    "    Display the analysis code version\n" +
                    "        -V\n" +
                    "    Display the signature file version\n" +
                    "        -Vxxx  where xxx is the signature file name\n" +
                    "    Check whether the PRONOM website has a newer signature file\n" +
                    "        -Cxxx  where xxx is the local signature file\n" +
                    "    Download the latest signature file from PRONOM website\n" +
                    "        -Dxxx  where xxx is the name of the file to download to\n" +
                    "    Quiet mode: only print errors to the command line\n" +
                    "        -Q\n" +
                    "    Run the analysis on a comma separated list of files\n" +
                    "        -Lxxx  where xxx is a comma-separated list of files\n" +
                    "        -Syyy  where yyy is the signature file\n" +
                    "        -Ozzz  where zzz is the base name for the output files\n" +
                    "        -Faaa  where aaa is a comma-separated list of output formats from XML,CSV\n" +
                    "    Run the analysis on the files in a file list file\n" +
                    "        -Axxx  where xxx is the file list file\n" +
                    "        -Syyy  where yyy is the signature file\n" +
                    "        -Ozzz  where zzz is the base name for the output files\n" +
                    "        -Faaa  where aaa is a comma-separated list of output formats from XML,CSV\n" +
                    "    Generate profile information on a directory\n" +
                    "        -Mxxx  where xxx is the directory\n" +
                    "        -Syyy  where yyy is the signature file\n" +
                    "        -Yzzz  where zzz is the location of the embedded database (will be created if it doesn't exist)\n" +
                    "        -Paaa  where aaa is the profile name to write to\n" +
                    "    Generate profile reports\n" +
                    "        -all-reports   List all available reports\n" +
                    "        -database      The database location if different from default\n" +
                    "        -report-type   The ouput format, options are PDF, XML, CSV. Deafult is PDF\n" +
                    "        -report-name   The report to use\n" +
                    "        -output        The ouput file name\n";

    //These are the possible actions requested
    private boolean isHelpDisplay = false;  //Display argument options (args must be -H)
    private boolean isAnalysisVersionDisplay = false;  //Display system version (args must be -V)
    private boolean isSigFileVersionDisplay = false;  //Display signature file version (args must be -S)
    private boolean isSigFileCheck = false;  //Checks whether a more recent signature file is available (args must be -C -S)
    private boolean isDownload = false;  //Download a signature file (args must be -D)
    private boolean isAnalysis = false;  //Run the analysis (args must be -A -S -O -F or -L -S -O -F)
    private boolean isProfiling = false;

    private AnalysisController myAnalysis;

    private File profilingDirectory = null;
    private String profileName = "";
    private File profileDBLocation = null;

    /**
     * Creates a new instance of CmdController
     *
     * @param theArgs the incoming arguments
     */
    public CmdController(String[] theArgs, AnalysisController control) {

        myAnalysis = control;
        this.parseArgs(theArgs);
    }

    private void parseArgs(String[] theArgs) {
        boolean isWellFormed = true;
        String theSignatureFile = "";
        String theListFileName = "";
        String theOutputFile = "DROID_out";
        String theOutputFormats = "XML";
        boolean profilingOk = true;
        //boolean isXMLOutput = false;
        //boolean isCSVOutput = false;
        String proxyHost = null;
        String proxyPort;

        //parse arguments
        for (String theArg : theArgs) {
            if (!theArg.substring(0, 1).equals("-")) {
                isWellFormed = false;
                break;
            }

            // print out a list of reports and exit
            if (theArg.contains("all-reports")) {
                List<String> reports = ReportManager.getReports();
                for (String r : reports) {
                    System.out.println(r);
                }
                System.exit(1);
            }

            if (theArg.contains("report-name") ) {

            }


            if (theArg.substring(1, 2).equals("V")) {
                theSignatureFile = theArg.substring(2);
                if (theSignatureFile.length() == 0) {
                    isAnalysisVersionDisplay = true;
                } else {
                    isSigFileVersionDisplay = true;
                }
            } else if (theArg.substring(1, 2).equals("C")) {
                theSignatureFile = theArg.substring(2);
                isSigFileCheck = true;
            } else if (theArg.substring(1, 2).equals("D")) {
                theSignatureFile = theArg.substring(2);
                isDownload = true;
            } else if (theArg.substring(1, 2).equals("L")) {
                String[] files = theArg.substring(2).split(",");
                for (String fileName : files) {
                    if (UrlByteReader.isURL(fileName) && proxyHost == null) {
                        /* Get proxy settings from configuration file, if not already set */
                        try {
                            myAnalysis.readConfiguration();
                            proxyHost = myAnalysis.getProxyHost();
                            proxyPort = Integer.toString(myAnalysis.getProxyPort());
                            if (!"".equals(proxyHost)) {
                                System.setProperty("http.proxyHost", proxyHost);
                                System.setProperty("http.proxyPort", proxyPort);
                            }
                        } catch (Exception e) {
                            MessageDisplay.generalInformation("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                        }
                    }

                    myAnalysis.addFile(fileName);
                }
                isAnalysis = true;
            } else if (theArg.substring(1, 2).equals("A")) {
                theListFileName = theArg.substring(2);
                isAnalysis = true;
            } else if (theArg.substring(1, 2).equals("M")) {
                theListFileName = theArg.substring(2);
                profilingDirectory = new File(theListFileName);
                if (!(profilingDirectory.exists() && profilingDirectory.isDirectory())) {
                    profilingOk = false;
                    break;
                } else {
                    isProfiling = true;
                }
            } else if (theArg.substring(1, 2).equals("S")) {
                theSignatureFile = theArg.substring(2);
            } else if (theArg.substring(1, 2).equals("O")) {
                theOutputFile = theArg.substring(2);
            } else if (theArg.substring(1, 2).equals("F")) {
                theOutputFormats = theArg.substring(2);
            } else if (theArg.substring(1, 2).equals("H")) {
                isHelpDisplay = true;
            } else if (theArg.substring(1, 2).equalsIgnoreCase("Q")) {
                myAnalysis.setVerbose(false);
            } else if (theArg.substring(1, 2).equalsIgnoreCase("P")) {
                profileName = theArg.substring(2).toUpperCase();
                if (profileName.equals("")) {
                    profilingOk = false;
                    break;
                }
            } else if (theArg.substring(1, 2).equalsIgnoreCase("Y")) {
                profileDBLocation = new File(theArg.substring(2));
            } else {
                isWellFormed = false;
                break;
            }
        }

        if (!isWellFormed) {
            MessageDisplay.fatalError("The command line arguments were incorrectly formed\n" + myArgumentHelp);
        }

        if (!profilingOk || isProfiling && (profileName.equalsIgnoreCase("") || profilingDirectory == null || profileDBLocation == null || theSignatureFile.equalsIgnoreCase(""))) {
            MessageDisplay.fatalError("The command line arguments were incorrect for profiling\n" +
                    "The target must be a directory and the profile name cannot be blank\n" + myArgumentHelp);
        }

        //check command is complete and consistent and run it
        if (isAnalysisVersionDisplay && !isSigFileVersionDisplay && !isSigFileCheck && !isDownload && !isAnalysis && !isHelpDisplay) {
            //display analysis version
            System.out.println("DROID " + AnalysisController.getDROIDVersion());
            log.info("DROID " + AnalysisController.getDROIDVersion());

        } else
        if (!isAnalysisVersionDisplay && isSigFileVersionDisplay && !isSigFileCheck && !isDownload && !isAnalysis && !isHelpDisplay) {
            //read signature file this automatically displays its version once it has read it in
            try {
                myAnalysis.readSigFile(theSignatureFile, false);
            } catch (Exception e) {
                MessageDisplay.fatalError("Error reading the signature file: " + theSignatureFile);
            }

        } else
        if (!isAnalysisVersionDisplay && !isSigFileVersionDisplay && isSigFileCheck && !isDownload && !isAnalysis && !isHelpDisplay) {
            //check whether a later signature file exists on PRONOM website
            //first read in the signature file
            try {
                myAnalysis.readSigFile(theSignatureFile, false);
            } catch (Exception e) {
                MessageDisplay.fatalError("Error reading the signature file: " + theSignatureFile);
            }
            //then read in configuration file to get URL for PRONOM web services
            try {
                myAnalysis.readConfiguration();
            } catch (Exception e) {
                MessageDisplay.fatalError("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
            }
            if (myAnalysis.isNewerSigFileAvailable()) {
                System.out.println("Signature file " + theSignatureFile + " is out of date.  Please download a new one from the PRONOM web service");
                log.info("Signature file " + theSignatureFile + " is out of date.  Please download a new one from the PRONOM web service");
            } else if (PronomWebService.isCommSuccess) {
                System.out.println("Signature file " + theSignatureFile + " is up to date");
                log.info("Signature file " + theSignatureFile + " is up to date");
            } else {
                String failureMessage = "Unable to connect to the PRONOM web service. Make sure that the following settings in your configuration file (DROID_config.xml) are correct:\n";
                failureMessage += "    1- <SigFileURL> is the URL of the PRONOM web service.  This should be '" + AnalysisController.PRONOM_WEB_SERVICE_URL + "'\n";
                failureMessage += "    2- <ProxyHost> is the IP address of the proxy server if one is required\n";
                failureMessage += "    3- <ProxyPort> is the port to use on the proxy server if one is required";
                MessageDisplay.fatalError(failureMessage);
            }

        } else
        if (!isAnalysisVersionDisplay && !isSigFileVersionDisplay && !isSigFileCheck && isDownload && !isAnalysis && !isHelpDisplay) {
            if (theSignatureFile.length() == 0) {
                MessageDisplay.fatalError("No signature file name was provided.\n" + myArgumentHelp);
            }
            //Read in configuration file to get URL for PRONOM web services
            try {
                myAnalysis.readConfiguration();
            } catch (Exception e) {
                MessageDisplay.fatalError("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
            }
            //download latest signature file from PRONOM website
            System.out.println("Downloading latest signature file ...");
            log.info("Downloading latest signature file ...");
            myAnalysis.downloadwwwSigFile(theSignatureFile, false);
            if (!PronomWebService.isCommSuccess) {
                String failureMessage = "Unable to connect to the PRONOM web service. Make sure that the following settings in your configuration file (DROID_config.xml) are correct:\n";
                failureMessage += "    1- <SigFileURL> is the URL of the PRONOM web service.  This should be '" + AnalysisController.PRONOM_WEB_SERVICE_URL + "'\n";
                failureMessage += "    2- <ProxyHost> is the IP address of the proxy server if one is required\n";
                failureMessage += "    3- <ProxyPort> is the port to use on the proxy server if one is required";
                MessageDisplay.fatalError(failureMessage);
            } else if (myAnalysis.isFileFound(theSignatureFile)) {
                System.out.println("A new signature file has been downloaded to " + theSignatureFile);
                log.info("A new signature file has been downloaded to " + theSignatureFile);
            } else {
                System.out.println("Signature file download has failed");
                log.info("Signature file download has failed");
            }

        } else
        if (!isAnalysisVersionDisplay && !isSigFileVersionDisplay && !isSigFileCheck && !isDownload && isAnalysis && !isHelpDisplay) {
            if (theSignatureFile.length() == 0) {
                MessageDisplay.fatalError("No signature file name was provided.\n" + myArgumentHelp);
            } else if (theOutputFile.length() == 0) {
                MessageDisplay.fatalError("No output file name was provided.\n" + myArgumentHelp);
            } else if (theOutputFormats.length() == 0) {
                MessageDisplay.fatalError("No output format was provided.\n" + myArgumentHelp);
            }

            //get list of files
            if (theListFileName.length() > 0) {
                try {
                    myAnalysis.readFileCollection(theListFileName);
                } catch (Exception e) {
                    MessageDisplay.fatalError("Error reading the file collection file: " + theListFileName);
                }
            }
            if (myAnalysis.getNumFiles() == 0) {
                MessageDisplay.fatalError("No file was provided for identification.\n" + myArgumentHelp);
            }
            //read signature file
            try {
                myAnalysis.readSigFile(theSignatureFile, false);
            } catch (Exception e) {
                MessageDisplay.fatalError("Error reading the signature file: " + theSignatureFile);
            }
            myAnalysis.checkSignatureFile();

            //run analysis
            if (myAnalysis.isVerbose()) {
                System.out.println("Run analysis");
                log.info("Run analysis");
            }
            myAnalysis.setAnalysisStart();
            myAnalysis.runFileFormatAnalysis(theOutputFormats, theOutputFile);

        } else if (isProfiling) {
            ProfileEntryPoint.main(new String[]{"-V", profilingDirectory.getAbsolutePath(),
                    "-N", profileName, "-S", theSignatureFile, "-D", profileDBLocation.getAbsolutePath()});
        } else if (isHelpDisplay) {
            MessageDisplay.generalInformation(myArgumentHelp);
        } else {
            MessageDisplay.fatalError("Too many command line options were provided\n" + myArgumentHelp);
        }

    }
}
