/**
 * (c) Copyright  2009 Tessella Support Services plc.  
 * All rights reserved. 
 * <p/>
 * Project: 5890
 * Droid Profiler
 * <p/>
 * <p/>
 * Developed By: carj and owul
 * Tessella Support Services
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * <p/>
 * email:  info@tessella.com
 * web:    www.tessella.com
 * <p/>
 * <p/>
 */
package uk.gov.nationalarchives.droid.commandLine;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import uk.gov.nationalarchives.droid.AnalysisController;
import uk.gov.nationalarchives.droid.MessageDisplay;
import uk.gov.nationalarchives.droid.profile.domain.Profile;
import uk.gov.nationalarchives.droid.profile.domain.ProfileVolume;
import uk.gov.nationalarchives.droid.profile.service.ProfilingManager;
import uk.gov.nationalarchives.droid.profile.service.ReportManager;
import uk.gov.nationalarchives.droid.xmlReader.PronomWebService;
import uk.gov.nationalarchives.droid.binFileReader.UrlByteReader;

import java.io.File;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

public class CommandLineParser {

    private AnalysisController myAnalysis;
    private Log log = LogFactory.getLog(this.getClass());
    private String sigFile = "";
    private String proxyHost = null;
    private String proxyPort;
    private String outputFormat = "XML";
    private String outputBaseName = "output";
    private String profileLocation = "";
    private String databaseLocation = null;
    private ProfilingManager manager = null;
    private String profileName = "profile-" + new Date().toString();
    private List<String> profiles = new ArrayList<String>();
    private File bulkOutputName = null;
    
     
    public CommandLineParser(String[] args, AnalysisController control) {

        myAnalysis = control;
        Options options = createOptions();
        org.apache.commons.cli.CommandLineParser parser = new PosixParser();

        try {
            

            
            CommandLine line = parser.parse(options, args);

            // turn off verbose debugging
            if (line.hasOption('q')) {
                myAnalysis.setVerbose(false);
            }

            //list reports
            if (line.hasOption('i')) {
                iOption();
                return;
            }

            //list parameters that can be used for filtering
            if (line.hasOption('k')) {
                kOption();
                return;
            }
            
            // display the help
            if (line.hasOption('h')) {
                hOption();
                return;
            }

            // display the droid version
            if (line.hasOption('v')) {
                System.out.println("\nDROID Version: " + AnalysisController.getDROIDVersion());
                return;
            }
             
            // set the signature file 
            if (line.hasOption('s')) {
                 sigFile = sOption(line.getOptionValue('s'));
                 if(sigFile.trim().equals("")){
                     return;
                 }
              }else{
                 sigFile = myAnalysis.getSignatureFileName();
              }

            // display the signature file version
            if (line.hasOption('x')) {
                xOption();
            }

            if (line.hasOption('d')) {
                 if(dOption(line.getOptionValue('d'))){
                     return;
                 }
            }

            if (line.hasOption('c')) {
               if(cOption()){
                   return;
               }
            }

            if (line.hasOption('f')) {
                String format = line.getOptionValue('f');
                if(!format.trim().equals("") && format != null){
                    outputFormat = format;
                }
            }

            if (line.hasOption('o')) {
                outputBaseName = line.getOptionValue('o');
            }

            // list of file names
            if (line.hasOption('l')) {
                  lOption(line.getOptionValue('l'));
                //System.exit(1);
            }

            if (line.hasOption('a')) {
                if(aOption(line.getOptionValue('a'))){
                    return;
                }
            }

            if (line.hasOption('m') || line.hasOption('n') || line.hasOption('b') || line.hasOption('z')) {

                myAnalysis.readConfiguration();
                myAnalysis.readSigFile(sigFile, false);

                //retrieve profile location/volume
                if (line.hasOption('m')) {
                    profileLocation = line.getOptionValue('m');
                    if (mOption()) {
                        return;
                    }
                }

                //retrieve and connect to the database (y Option)
                yOption(line);

                // display a list of profile names
                if (line.hasOption('z')) {
                    zOption();
                    return;
                }
                
                //retrieve profile name
                if (line.hasOption("p")) {
                    pOtion(line);
                }

                //run profiling
                if (line.hasOption('m')) {
                    System.out.println("\nProfiling started ...\n");
                    commenceProfiling(manager, profileName, "Profile Created By Command Line", profileLocation, myAnalysis);
                    System.out.println("\nProfiling completed ");
                }
                                
                // bulk export
                if (line.hasOption('b')) {
                      bOption(line.getOptionValue('b'));
                }
                                               
                // run report
                if (line.hasOption('n') || bulkOutputName!=null ) {

                    //retrieve report name
                    String reportName = line.getOptionValue('n');
                    if (reportName != null) {
                        if (reportName.startsWith("=")) {
                            reportName = reportName.replaceFirst("=", "");
                        }
                    }else{
                        reportName = "";
                    }

                    if (line.hasOption("j") || reportName.trim().equalsIgnoreCase("Profile Summary")|| reportName.trim().equalsIgnoreCase("Entire Database Summary") || bulkOutputName!=null) {
                        //Generate report
                         generateReport(reportName, line, profiles, manager,bulkOutputName);
                    }
                }                       
            }

        } catch (Exception e) {
            System.out.println("\nAn error occured: "+e.getMessage()+"\n");
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("droid", options);
           
        }
        return;
    }

    
    private void iOption() {
        List<String> reports = ReportManager.getReports();
        System.out.println("\n");
        int i = 0;
        for (String r : reports) {
            i++;
            System.out.println(i + ")\t" + r);
        }
    }
    
    private void kOption(){
                System.out.println("\n1)\tFile Name\t[NAME]");
                System.out.println("2)\tFile Path\t[PATH]");
                System.out.println("3)\tFile Format\t[FORMAT]");
                System.out.println("4)\tIdentification Status\t[STATUS]");
                System.out.println("5)\tMime Type\t[MIME]");
                System.out.println("6)\tPUID\t[PUID]");
                System.out.println("7)\tVersion\t[VERSION]");
                System.out.println("8)\tWarning\t[WARNING]");
                System.out.println("9)\tYear Modified\t[YEAR]");
                System.out.println("\nNote: Use the name in brackets when specifying parameter name for option j\n");        
    }
    
    private void hOption(){
                        HelpFormatter formatter = new HelpFormatter();
                System.out.println("\nDROID Command Line Options\n");
                formatter.printHelp("droid", createOptions());
    }
    
    private String sOption(String sigFileName){

        try {

            if (sigFileName.startsWith("=")) {
                sigFileName = sigFileName.replaceFirst("=", "");
            }
            if (!sigFileName.trim().equals("") && sigFileName != null) {
                sigFile = sigFileName;
            } else {
                sigFile = myAnalysis.getSignatureFileName();
                MessageDisplay.fatalError("Invalid sig file " + sigFileName + " specified. Using Sig file: " + sigFile);
                System.out.println("Invalid sig file " + sigFileName + " specified. Using Sig file: " + sigFile);
            }
            System.out.println("\nSetting Signature File to: " + sigFile);
            myAnalysis.readSigFile(sigFile, true);
        } catch (Exception e) {
            MessageDisplay.fatalError("Invalid sig file specified: " + sigFile + "\n" + e.getMessage());
            System.out.println("Invalid sig file specified: " + sigFile + "\n" + e.getMessage());
            sigFile = "";
        }
        return sigFile;
    }
    
    private void xOption(){
                        try {
                    myAnalysis.readConfiguration();
                    System.out.println("\nVersion of Signature file: "+ sigFile);
                    myAnalysis.readSigFile(sigFile, false);
                } catch (Exception e) {
                    MessageDisplay.fatalError("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                    System.out.println("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                }
               
    }
    
    private boolean dOption(String sigFileName){
        boolean hasErrorOccured = false;
                        try {
                    
                    if (sigFileName.startsWith("=")) {
                        sigFileName = sigFileName.replaceFirst("=", "");
                    }
                    if (!sigFileName.trim().equals("") && sigFileName != null) {
                        sigFile = sigFileName;
                    } else {
                        MessageDisplay.fatalError("Invalid sig file " + sigFileName + " specified.");
                        System.out.println("Invalid sig file " + sigFileName + " specified.");
                        hasErrorOccured = true;
                    }
                    System.out.println("\nDownloading latest signature file ...");
                    log.info("\nDownloading latest signature file ...");
                    myAnalysis.downloadwwwSigFile(sigFile, false);
                    if (!PronomWebService.isCommSuccess) {
                        String failureMessage = "\n Unable to connect to the PRONOM web service. Make sure that the following settings in your configuration file (DROID_config.xml) are correct:\n";
                        failureMessage += "    1- <SigFileURL> is the URL of the PRONOM web service.  This should be '" + AnalysisController.PRONOM_WEB_SERVICE_URL + "'\n";
                        failureMessage += "    2- <ProxyHost> is the IP address of the proxy server if one is required\n";
                        failureMessage += "    3- <ProxyPort> is the port to use on the proxy server if one is required";
                        MessageDisplay.fatalError(failureMessage);
                        System.out.println(failureMessage);
                    } else if (myAnalysis.isFileFound(sigFile)) {
                        System.out.println("A new signature file has been downloaded to " + sigFile);
                        log.info("A new signature file has been downloaded to " + sigFile);
                    } else {
                        System.out.println("Signature file download has failed");
                        log.info("Signature file download has failed");
                    }

                } catch (Exception e) {
                    MessageDisplay.fatalError("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                    System.out.println("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                    hasErrorOccured = true;
                }
        return hasErrorOccured;
    }
    
    private boolean cOption(){
        boolean hasErrorOccured = false;
                        try {
                    System.out.println("\nChecking Signature file: "+ sigFile);
                    myAnalysis.readConfiguration();
                    myAnalysis.readSigFile(sigFile, false);
                    if (myAnalysis.isNewerSigFileAvailable()) {
                        System.out.println("Signature file " + sigFile + " is out of date.  Please download a new one from the PRONOM web service");
                        log.info("Signature file " + sigFile + " is out of date.  Please download a new one from the PRONOM web service");
                    } else if (PronomWebService.isCommSuccess) {
                        System.out.println("Signature file " + sigFile + " is up to date");
                        log.info("Signature file " + sigFile + " is up to date");
                    } else {
                        String failureMessage = "Unable to connect to the PRONOM web service. Make sure that the following settings in your configuration file (DROID_config.xml) are correct:\n";
                        failureMessage += "    1- <SigFileURL> is the URL of the PRONOM web service.  This should be '" + AnalysisController.PRONOM_WEB_SERVICE_URL + "'\n";
                        failureMessage += "    2- <ProxyHost> is the IP address of the proxy server if one is required\n";
                        failureMessage += "    3- <ProxyPort> is the port to use on the proxy server if one is required";
                        MessageDisplay.fatalError(failureMessage);
                        System.out.println(failureMessage);
                        hasErrorOccured = true;
                    }
                } catch (Exception e) {
                    MessageDisplay.fatalError("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                    System.out.println("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                    hasErrorOccured = true;
                }
        return hasErrorOccured;
    }
    
    private void lOption( String files){
        if (files.startsWith("=")) {
            files = files.replaceFirst("=", "");
        }
        for (String file : files.split(",")) {
            if (UrlByteReader.isURL(file) && proxyHost == null) {
                /* Get proxy settings from configuration file, if not already set */
                try {
                    myAnalysis.readConfiguration();
                    proxyHost = myAnalysis.getProxyHost();
                    proxyPort = Integer.toString(myAnalysis.getProxyPort());
                    if (!"".equals(proxyHost)) {
                        System.setProperty("http.proxyHost", proxyHost);
                        System.setProperty("http.proxyPort", proxyPort);
                    }
                } catch (Exception e) {
                    MessageDisplay.generalInformation("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                    System.out.println("Error reading the congifuration file: " + AnalysisController.CONFIG_FILE_NAME + "\n" + e.getMessage());
                }
            }
            myAnalysis.addFile(file);
        }

        System.out.println("\nPerforming File Format Analysis: ");
        try {
            myAnalysis.readConfiguration();
            myAnalysis.readSigFile(sigFile, false);
            myAnalysis.checkSignatureFile();
            myAnalysis.setAnalysisStart();
            myAnalysis.runFileFormatAnalysis(outputFormat, outputBaseName);
            System.out.println("File Format Analysis completed successfully. Results saved in file: " + outputBaseName + "\n");
        } catch (Exception ex) {
            System.out.println("File Format Analysis failed\n ");
        }
    }
    
    private boolean aOption(String xmlfilename){
            try{      
                if (xmlfilename.startsWith("=")) {
                    xmlfilename = xmlfilename.replaceFirst("=", "");
                }
                //get list of files
                if (xmlfilename.length() > 0) {
                    try {
                        myAnalysis.readFileCollection(xmlfilename);
                    } catch (Exception e) {
                        MessageDisplay.fatalError("\nError reading the file collection file: " + xmlfilename);
                        System.out.println("\nError reading the file collection file: " + xmlfilename);
                    }
                }
                if (myAnalysis.getNumFiles() == 0) {
                    MessageDisplay.fatalError("\nNo file was provided for identification.\n" + xmlfilename);
                    System.out.println("\nNo file was provided for identification.\n" + xmlfilename);
                }
                System.out.println("\n");
                myAnalysis.readConfiguration();
                myAnalysis.readSigFile(sigFile, false);
                myAnalysis.setAnalysisStart();
                myAnalysis.runFileFormatAnalysis(outputFormat, outputBaseName);
                return true;
            }catch(Exception e){
                            System.out.println("\nAn error occured: "+e.getMessage()+"\n");
                            HelpFormatter formatter = new HelpFormatter();
                formatter.printHelp("droid", createOptions());
                return false;
            }
    }
    
    private boolean mOption(){
                     
                    if (profileLocation == null || profileLocation.length() < 2) {
                        System.out.println("Profile location: "+ profileLocation+" is invalid\n");
                        HelpFormatter formatter = new HelpFormatter();
                        formatter.printHelp("droid", createOptions());
                        return true;
                    }
                    if (profileLocation.startsWith("=")) {
                        profileLocation = profileLocation.replaceFirst("=", "");
                    }
                    if (!new File(profileLocation).exists() || !new File(profileLocation).isDirectory()) {
                        System.out.println("An error occured. Either the path " + profileLocation + " does not exist or it is not a directory.\n");
                        HelpFormatter formatter = new HelpFormatter();
                        formatter.printHelp("droid",createOptions());
                        return true;
                    }
                    return false;
    }
    
    private void yOption(CommandLine line){
            try{
                
                try {
                    databaseLocation = line.getOptionValue('y');
                } catch (Exception e) {
                    databaseLocation = null;
                } finally {
                    if (databaseLocation == null || databaseLocation.trim().equals("") ) {
                        System.out.println("\nInvalid database location specified: "+databaseLocation+". Using defualt database: "+myAnalysis.getProfileDatabasePath());
                        databaseLocation = myAnalysis.getProfileDatabasePath();
                    }
                }
                if (databaseLocation != null || databaseLocation.length() > 0) {
                    if (new File(databaseLocation).exists() && new File(databaseLocation).isDirectory()) {
                        manager = new ProfilingManager(new File(databaseLocation));
                        myAnalysis.setDatabaseLocation(new File(databaseLocation).getAbsolutePath());
                    } else if(!new File(databaseLocation).exists()){
                        System.out.println("\nDatabase location \""+databaseLocation+"\" does not exist. A new database \""+databaseLocation+"\" will be created");
                        manager = new ProfilingManager(new File(databaseLocation)); 
                        myAnalysis.setDatabaseLocation(new File(databaseLocation).getAbsolutePath());
                    }else{
                        System.out.println("\nInvalid database location specified; "+databaseLocation+". Either it is not a directory or an error occured. Using defualt database: "+myAnalysis.getProfileDatabasePath());
                        manager = new ProfilingManager(new File(myAnalysis.getProfileDatabasePath()));
                    }
                } else {
                    System.out.println("\nNo database location specified. Using defualt database: "+myAnalysis.getProfileDatabasePath());
                    manager = new ProfilingManager(new File(myAnalysis.getProfileDatabasePath()));
                }
            }catch(Exception e){
                                            System.out.println("\nAn error occured: "+e.getMessage()+"\n");
                            HelpFormatter formatter = new HelpFormatter();
                formatter.printHelp("droid", createOptions());
            }
    }
    
    private void zOption(){
                        try {
                    if(manager != null){
                    java.util.List<String> profileNames = manager.getAllProfileNames();
                    if (profileNames.size() > 0) {
                            Collections.sort(profileNames);
                     }
                    if(profileNames.size() > 0){
                        System.out.println("\nProfiles\n");
                    }
                    int cnt  = 0;
                    for (String profName : profileNames) {
                           
                            System.out.println(++cnt+")\t"+profName.trim());
                     }
                    System.out.println("\n ");
                    }else{
                        System.out.println("\nError occured in retrieving profile names \n");
                    }
                } catch (Exception e) {
                    MessageDisplay.fatalError("\nError occured in retrieving profile names \n"+e.getMessage());
                    System.out.println("\nError occured in retrieving profile names \n");
                }
    }
    
    private void pOtion(CommandLine line) {

        if (line.getOptionValue('p') != null || line.getOptionValue('p').length() > 0) {
            String profileArg = line.getOptionValue('p');
            if (profileArg.startsWith("=")) {
                profileArg = profileName.replaceFirst("=", "");
            }

            //retrieve profiles and add to the profiles list by using the "," as a delimeter
            //For reporting purposes, this list of profiles is used for multiple profile reports
            for (String profile : profileArg.split(",")) {
                profiles.add(profile.toUpperCase());
            }
            if (profiles.size() == 0) {
                profileName = "";
            } else {
                //set the profileName to the first. In case the user wants to create a profile only the fist one will be created i.e if there is a list of profile names
                profileName = profiles.get(0);
            }

        } else {
            System.out.println("Invalid profile name specified: " + line.getOptionValue('p'));
        }
    }
    
    private void bOption(String outputFileName){
                          
                    if (outputFileName.startsWith("=")) {
                        outputFileName = outputFileName.replaceFirst("=", "");
                    }
                    if (!outputFileName.trim().equals("")) {
                        
                        try {
                            bulkOutputName = new File(outputFileName);
                            //TODO~: create a method like generateReport where you append the filter string and prifle names (if they exist) to the query. This is for file level reporting only
                            // Also check the enable and disabling of the report manager
                           // manager.export(bulkOutputName, profileName);
                        } catch (Exception ex) {
                            System.out.println("\nInvalid Output File Name: " + outputFileName + "\n");
                            log.error("\nInvalid Output File Name: " + outputFileName + "\n");
                            bulkOutputName = null;
                            HelpFormatter formatter = new HelpFormatter();
                            formatter.printHelp("droid", createOptions());
                        }


                    } else {
                        System.out.println("\nInvalid Output File Name: " + outputFileName + "\n");
                        log.error("\nInvalid Output File Name: " + outputFileName + "\n");
                        bulkOutputName = null;
                        HelpFormatter formatter = new HelpFormatter();
                        formatter.printHelp("droid", createOptions());
                    }

    }    
    
    /**
     * This is used to generate the report
     * @param reportName The name of the report to be generated
     * @param line The command line 
     * @param profiles List of profile names
     * @param manager The Profiling Manager
     */
    private void generateReport(String reportName, CommandLine line, List<String> profiles, ProfilingManager manager, File bulkOutputName) {
        List<String> reports = ReportManager.getReports();


        //Run reports if only there are reports available
        if (reports != null && reports.size() > 0) {

            //retrieve parameter name
            String parameterName = line.getOptionValue('j');
            if (parameterName != null && parameterName.startsWith("=")) {
                parameterName = parameterName.replaceFirst("=", "");
            } else if (parameterName == null) {
                parameterName = "";
            }

            boolean onlyBulkExport = false;
            if (reportName.trim().equals("") && bulkOutputName != null) {
                reportName = "File Detail of Database";
                onlyBulkExport = true;
            } else if (reportName.trim().equals("") && bulkOutputName == null) {
                System.out.println("\nNo report name specified");
                return;
            }
            //Profile Summary and Entire database reports do not need filters
            boolean noFilters = false;
            if (reportName.trim().equalsIgnoreCase("Profile Summary") || reportName.trim().equalsIgnoreCase("Entire Database Summary") || bulkOutputName != null) {
                noFilters = true;
            }

            //retrieve report name suffix, an empty string is returned if the parameter name is incorrect
            String reportNameSuffix = getReportSuffix(parameterName);
            if (!reportNameSuffix.trim().equals("") || noFilters || bulkOutputName != null) {

                //retrieve filter string and replace all wildcard characters "*" with the Derby database wildcard character "%"
                String filterString = line.getOptionValue('g');
                if (filterString == null || filterString.trim().equals("")) {
                    filterString = "%";
                } else {
                    filterString = filterString.replace("*", "%");
                }
                if (filterString.startsWith("=")) {
                    filterString = filterString.replaceFirst("=", "");
                }

                boolean errorOccured = false;  //flag to record if an error occured

                if (reportNameSuffix.trim().equals("Date")) {
                    //for a Date search, it can be a String search (using the LIKE keyword) or a numeric search using numerical operators
                    if (isLIKEsearch(filterString)) {
                        if (reportName.toUpperCase().contains("DATABASE")) {
                            filterString = "WHERE UCASE(cast(YEAR(fileobject.lastmodified)as char(4))) LIKE UCASE('" + filterString.trim() + "')";
                        } else {
                            filterString = " (UCASE(cast(YEAR(fileobject.lastmodified)as char(4))) LIKE UCASE('" + filterString.trim() + "'))";
                        }
                    } else {
                        String operators[] = {">=", "<=", "!=", "=", ">", "<"};

                        boolean isOperatorFound = false; //flags whether or not a valid numerical operator was found in the filter string

                        for (int i = 0; i < operators.length; i++) {
                            if (filterString.trim().startsWith(operators[i].trim())) {
                                isOperatorFound = true;
                                String integerPart = "";
                                if (i > 2) {
                                    //if the filter string contains a valid operator with 1 character e.g =,<,>
                                    // then retrieve the integer portion of the filter string
                                    integerPart = filterString.substring(1, filterString.length());
                                } else {
                                    integerPart = filterString.substring(2, filterString.length());
                                }
                                try {
                                    //verify that the integer portion retrieved is a valid integer
                                    Integer.parseInt(integerPart);
                                } catch (NumberFormatException ex) {
                                    System.out.println("\nInvalid integer number specified: " + integerPart);
                                    errorOccured = true;
                                }
                                if (!errorOccured) {
                                    if (reportName.toUpperCase().contains("DATABASE")) {
                                        filterString = "WHERE YEAR(fileobject.lastmodified) " + filterString.trim();
                                    } else {
                                        filterString = " ( YEAR(fileobject.lastmodified) " + filterString.trim() + ")";
                                    }
                                }
                                break;
                            }
                        }
                        //if none of the operators were found, flag that an error has occured
                        if (!isOperatorFound) {
                            System.out.println("\nInvalid operator used. Valid operators are: \">=\", \"<=\", \"!=\", \"=\", \">\", \"<\" ");
                            errorOccured = true;
                        }
                    }
                }

                String type = line.getOptionValue('t');
                if (type == null) {
                    if (!onlyBulkExport) {
                        System.out.println("\nNo report type was specified. Setting output report file type to PDF");
                    }
                    type = "PDF";
                }

                if (type.startsWith("=")) {
                    type = type.replaceFirst("=", "");
                }

                //if type is incorrect set it to the default type (PDF)
                if (!type.equalsIgnoreCase("PDF") && !type.equalsIgnoreCase("CSV") && !type.equalsIgnoreCase("XML") && !type.equalsIgnoreCase("XLS")) {
                    System.out.println("\nInvalid report type specified [PDF,CSV,XML,XLS]. Report type set to PDF");
                    type = "PDF";
                }


                String outputName = reportName + "." + type;

                boolean isValidReportName = false;

                //verify that the report name is a valid report name
                for (String rptName : reports) {
                    if (rptName.trim().equalsIgnoreCase(reportName.trim())) {
                        reportName = rptName.trim();
                        isValidReportName = true;
                        break;
                    }
                }

                if (!isValidReportName) {
                    System.out.println("\nInvalid report name: " + reportName);
                }

                //Generate bulk CSV export report
                if (bulkOutputName != null) {
                    String filterString_bulk = "";
                    if (filterString.trim().startsWith("WHERE")) {
                        filterString_bulk = filterString.replace("WHERE", " ");
                    } else if (!filterString.trim().equals("%")) {
                        filterString_bulk = filterString;
                    }

                    if (!parameterName.trim().equals("")) {
                        if (!filterString_bulk.trim().equals("") && !filterString_bulk.trim().equals("%")) {
                            if (!reportNameSuffix.trim().equalsIgnoreCase("DATE")) {
                                filterString_bulk = " AND " + getFilterQuery(parameterName) + " UCASE(''" + filterString_bulk + "'') ";
                            } else {
                                filterString_bulk = " AND " + filterString_bulk.replace("'", "''");
                            }
                        }
                    } else if(parameterName.trim().equals("") && !filterString_bulk.trim().equals("%") ){
                        if(!filterString_bulk.trim().equals("")){
                            System.out.println("\nNo parameter specified. Filter String reset to \"*\"");
                        }
                        filterString_bulk = "%";
                    }

                    String bulk_ReportName = "";
                    if (profiles.size() > 0) {
                        bulk_ReportName = "File Detail of Profile";
                    } else {
                        bulk_ReportName = "File Detail of Database";
                    }
                    
                    
                    manager.export(bulkOutputName, profiles, bulk_ReportName, filterString_bulk);
                }

                if (onlyBulkExport) {
                    return;
                }
                //flag an error if a profile report is requested but no profile name was supplied
                if (reportName.toUpperCase().contains("PROFILE") && profiles.size() == 0) {
                    errorOccured = true;
                }

                if (isValidReportName && !errorOccured) {
                    //retrieve output file name
                    if (line.hasOption('r')) {
                        outputName = line.getOptionValue('r');
                        if (outputName.startsWith("=")) {
                            outputName = outputName.replaceFirst("=", "");
                        }
                        if (!outputName.toUpperCase().trim().endsWith(".PDF") && !outputName.toUpperCase().trim().endsWith(".CSV") && !outputName.toUpperCase().trim().endsWith(".XML") && !outputName.toUpperCase().trim().endsWith(".XLS")) {
                            outputName = outputName + "." + type;

                        }
                    }

                    //Append a suffix to the report name if its is not any of the following reports "Profile Summary" and "Entire Database" reports
                    if (!(reportName.toUpperCase().contains("ENTIRE DATABASE") || reportName.toUpperCase().contains("PROFILE SUMMARY")) ) {
                        if(!reportNameSuffix.trim().equals("")){
                            reportName = reportName + "_" + reportNameSuffix;
                        }else{
                            System.out.println("\nInvalid parameter specified: \""+parameterName+"\"");
                            return;
                        }
                    }

                    //generate and save the file
                    System.out.println("\nGenerating report ... ");
                    ReportManager.saveReportToFile(reportName, outputName, type, profiles, manager, filterString);

                }
            }
        }
    }

    

    
    
    /**
     *
     * @param filterString The filter string whose search type is to be determined
     * @return True if the filter string is not intended for a numerical search otherwise, false
     */
    private boolean isLIKEsearch(String filterString){
        String operators[] = {"=",">",">=","<","<=","!="};
        boolean isNumericalSearch = false;
        for(int i=0;i<operators.length;i++){
            if(filterString.contains(operators[i].trim())){
                isNumericalSearch = true;
                break;
            }
        }
        return !isNumericalSearch;
    }

    /**
     * This method is used to get a part of the SQL query used to call the Derby CSV export function
     * @param parameterName The parameter name selected for filtering
     * @return A portion of the SQL Query
     */
    private String getFilterQuery(String parameterName){
                String query = "";
        if(parameterName.equalsIgnoreCase("NAME")){
            //UCASE(format.version) like
            query = " UCASE(fileobject.name) like ";
        }else if(parameterName.equalsIgnoreCase("PATH")){
            query = " UCASE(fileobject.path) like ";
        }else if(parameterName.equalsIgnoreCase("FORMAT")){
            query = " UCASE(format.name) like ";
        }else if(parameterName.equalsIgnoreCase("STATUS")){
            query = " UCASE(fileobject.CLASSIFICATIONTEXT) like ";
        }else if(parameterName.equalsIgnoreCase("MIME")){
            query = " UCASE(format.mimetype) like ";
        }else if(parameterName.equalsIgnoreCase("PUID")){
            query = " UCASE(format.puid) like ";
        }else if(parameterName.equalsIgnoreCase("VERSION")){
            query = " UCASE(format.version) like ";
        }else if(parameterName.equalsIgnoreCase("WARNING")){
               query = " UCASE(format.warning) like ";
        }else if(parameterName.equalsIgnoreCase("YEAR")){
             query = "";
        }
        
        return query;
    }
    
    /**
     *
     * @param parameterName
     * @return A string denoting the type of report to be run. returns an empty string is the parameter name is incorrect
     */
    private String getReportSuffix(String parameterName){
        String suffix = "";
        if(parameterName.equalsIgnoreCase("NAME")){
            suffix = "FileName";
        }else if(parameterName.equalsIgnoreCase("PATH")){
            suffix = "FilePath";
        }else if(parameterName.equalsIgnoreCase("FORMAT")){
            suffix = "Format";
        }else if(parameterName.equalsIgnoreCase("STATUS")){
            suffix = "IDStatus";
        }else if(parameterName.equalsIgnoreCase("MIME")){
            suffix = "Mime";
        }else if(parameterName.equalsIgnoreCase("PUID")){
            suffix = "PUID";
        }else if(parameterName.equalsIgnoreCase("VERSION")){
            suffix = "Version";
        }else if(parameterName.equalsIgnoreCase("WARNING")){
               suffix = "Warning";
        }else if(parameterName.equalsIgnoreCase("YEAR")){
             suffix = "Date";
        }
        
        return suffix;
    }
    
    private Options createOptions() {

        Options options = new Options();
        options.addOption("h", "help", false, "Display the command line options");
        options.addOption("v", "version", false, "Display DROID application version");
        options.addOption("q", "quiet", false, "Quiet mode: only print errors to the command line");
        options.addOption("x", "signature-file-version", false, "Display the signature file version");
        options.addOption("s", "signature-file", true, "Set the signature file");
        options.addOption("c", "check-signature-file", false, "Check whether the PRONOM website has a newer signature file");
        options.addOption("d", "download-signature-file", true, "Download the latest signature file from PRONOM website");


        options.addOption("a", "xml-file-list", true, "xml file containing input files");
        options.addOption("l", "list-input-files", true, "comma separated list of files, -l=file1.txt,file2.txt");
        options.addOption("o", "output-file-base-name", true, "base name for the output files");
        options.addOption("f", "format-output", true, "output format from XML,CSV");

        options.addOption("y", "profile-database", true, "The database location if different from default");
        options.addOption("m", "profile-start-location", true, "Directory to profile");
        options.addOption("p", "profile-name", true, "profile name");

        options.addOption("i", "list-reports", false, "List all available reports");
        options.addOption("k", "list-parameters", false, "List all parameters available for filtering.");          
        options.addOption("z", "list-profiles", false, "List all available profiles");
        options.addOption("g", "filter-string", true, "The filter string. Default is *. Valid operators(=,>,>=,<,<=,!=) can be used when filtering by Year e.g. -g\">=1998\".");
        options.addOption("j", "parameter-name", true, "The parameter to filter by." );          
        options.addOption("n", "name-of-report", true, "The report to run");
        options.addOption("t", "type-of-report", true, "The output report format, options are PDF,CSV,XLS and XML. Default is PDF");
        options.addOption("r", "report-output", true, "The output report file name");
      

        options.addOption("b", "bulk-profile-output", true, "Output csv profile data from database");


        return options;
    }

    public void commenceProfiling(ProfilingManager manager, String profileName, String description, String target, AnalysisController controller) throws Exception {
        Profile prof = manager.getProfileByName(profileName);
        if (prof == null) {
            prof = manager.createProfile(profileName, description);
        }
        String hostname = InetAddress.getLocalHost().getHostName();
        ProfileVolume vol = manager.createVolumeForProfileToStartNow(prof, target, hostname);
        manager.performProfilingOnDirectory(vol, controller);
        manager.labelVolumeComplete(vol);
        System.out.println("Total files: " + manager.getNumberOfFilesInVolume(vol));
        log.info("Total files: " + manager.getNumberOfFilesInVolume(vol));
        System.out.println("Total size: " + manager.getTotalSizeOfVolume(vol));
        log.info("Total size: " + manager.getTotalSizeOfVolume(vol));
        System.out.println("Average size: " + manager.getAverageSizeOfFileInVolume(vol));
        log.info("Average size: " + manager.getAverageSizeOfFileInVolume(vol));
        System.out.println("Max file: " + manager.getMaxSizeOfFileInVolume(vol));
        log.info("Max file: " + manager.getMaxSizeOfFileInVolume(vol));
        System.out.println("Min file: " + manager.getMinSizeOfFileInVolume(vol));
        log.info("Min file: " + manager.getMinSizeOfFileInVolume(vol));
        System.out.println("\n\nYear\t\tTotal Size (bytes)\t\tNum. Of Files");
        log.info("\n\nYear\t\tTotal Size (bytes)\t\tNum. Of Files");
        for (Object[] obj : manager.getVolumeYearInfo(vol)) {
            System.out.println(cleanObject(obj[0]) + "\t\t" + cleanObject(obj[1]) + "\t\t" + cleanObject(obj[2]));
            log.info(cleanObject(obj[0]) + "\t\t" + cleanObject(obj[1]) + "\t\t" + cleanObject(obj[2]));
        }
        System.out.println("\n\nPUID \tMIME\tName\tVersion\tNum. Of Files\tTotal Size (bytes)");
        log.info("\n\nPUID \tMIME\tName\tVersion\tNum. Of Files\tTotal Size (bytes)");
        for (Object[] obj : manager.getVolumeFormatInfo(vol)) {
            if (obj[2] == null) {
                obj[2] = "Unknown";
            }
            System.out.println(cleanObject(obj[0]) + "\t" + cleanObject(obj[1]) + "\t" + cleanObject(obj[2]) + "\t" + cleanObject(obj[3]) + "\t" + cleanObject(obj[4]) + "\t" + cleanObject(obj[5]) + "\t");
            log.info(cleanObject(obj[0]) + "\t" + cleanObject(obj[1]) + "\t" + cleanObject(obj[2]) + "\t" + cleanObject(obj[3]) + "\t" + cleanObject(obj[4]) + "\t" + cleanObject(obj[5]) + "\t");
        }
    }

    /**
     * Replaces null or blank with the word none
     *
     * @param obj the object
     * @return object post cleaning
     */
    private Object cleanObject(Object obj) {
        if (obj == null || obj.equals("")) {
            return "none";
        } else {
            return obj;
        }
    }

}
