/**
 * (c) Copyright  2008 Tessella Support Services plc.  
 * All rights reserved. 
 * <p/>
 * Project: 5890
 * DCS
 * <p/>
 * <p/>
 * Developed By:
 * Tessella Support Services
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * <p/>
 * email:  info@tessella.com
 * web:    www.tessella.com
 * <p/>
 * <p/>
 * Created By:     trir
 * Created Date:   29-Oct-2008
 */
package uk.gov.nationalarchives.droid.profile.database;

import java.io.BufferedWriter;
import java.io.IOException;
import uk.gov.nationalarchives.droid.profile.domain.FileObject;
import uk.gov.nationalarchives.droid.profile.domain.Format;
import uk.gov.nationalarchives.droid.profile.domain.Profile;
import uk.gov.nationalarchives.droid.profile.domain.ProfileVolume;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import org.hibernate.*;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.io.File;

import java.io.FileWriter;
import org.apache.commons.logging.*;

/**
 * This class is the access point to the database. All database access should occur through here.
 * Only the Profiling Manager should have access to this class. Any other class wanting DB access should
 * go through the profiling manager.
 */
public class ProfilingDBAccessPoint {

    private SessionFactory sessionFactory;
    private Log log = LogFactory.getLog(this.getClass());
    /**
     * Constructor from session factory
     *
     * @param sessionFactory the session factory
     */
    public ProfilingDBAccessPoint(SessionFactory sessionFactory) {
        this.sessionFactory = sessionFactory;
    }

    /**
     * Searches for a profile by name, returns null if none found
     *
     * @param name The name of the profile
     * @return the profile
     */
    public Profile getProfileByName(String name) {
        Profile prof = null;
        Session session = null;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            prof = getProfileInsideSession(name, session);
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return prof;
    }



    public String bulkExport(File outputFile,List<String> profiles,String reportName, String filterString){

        String occuredError = "";
        String selectQuery = "select DISTINCT profile.name, CAST(profile.datecreated as VARCHAR(25)), profilevolume.hostname, profilevolume.volume, CAST(profilevolume.datestarted AS VARCHAR(25)), CAST(profilevolume.datecompleted AS VARCHAR(25)),  fileobject.path , fileobject.name, CAST(fileobject.filesize AS CHAR(50)) , CAST(fileobject.lastmodified AS VARCHAR(25)), fileobject.CLASSIFICATIONTEXT , format.name , format.puid , format.version , format.mimetype ,format.warning  from profile , profilevolume , fileobject left outer join format on format.fileobject_id = fileobject.id where profilevolume.profile_id = profile.id and fileobject.profilevolume_id = profilevolume.id ";
        //String selectQuery = "select profile.name, profile.datecreated, profilevolume.hostname, profilevolume.volume, profilevolume.datestarted, profilevolume.datecompleted,  fileobject.path , fileobject.name, fileobject.filesize , fileobject.lastmodified, fileobject.CLASSIFICATIONTEXT , format.name , format.puid , format.version , format.mimetype ,format.warning  from profile , profilevolume , fileobject , format where profilevolume.profile_id = profile.id and fileobject.profilevolume_id = profilevolume.id and format.fileobject_id = fileobject.id";
        // String selectQuery = "select p.name as \"Profile Name\", p.datecreated as \"Profile Created\", pv.hostname as \"Hostname\", pv.volume  as \"Profile Directory\", pv.datestarted as \"Profile Started\", pv.datecompleted as \"Profile Completed\",  fo.path as \"File path\", fo.name as \"File Name\", fo.filesize as \"FileSize\",fo.lastmodified as \"Last Modified Date\",fo.CLASSIFICATIONTEXT, f.name as \"Format Name\", f.puid as \"PUID\", f.version as \"Format Version\", f.mimetype as \"Mimetype\",f.warning  from profile p, profilevolume pv, fileobject fo, format f where pv.profile_id = p.id and fo.profilevolume_id = pv.id and f.fileobject_id = fo.id";
               if (reportName.toUpperCase().contains("PROFILE")) {
                if(profiles.size() == 0){
                    return occuredError;
                }
                String profileName = "";
                if (profiles != null) {
                    profileName = "(";
                    for (int i = 0; i < profiles.size(); i++) {
                        if (i != 0) {
                            profileName += ",";
                        }
                        profileName += ("''" + profiles.get(i).toUpperCase().trim() + "''");
                    }
                    profileName += ")";
                    
                    profileName = " AND profile.name in "+ profileName;
                    
                    selectQuery += profileName;
                      //selectQuery = "select p.name as \"Profile Name\", p.datecreated as \"Profile Created\", pv.hostname as \"Hostname\", pv.volume  as \"Profile Directory\", pv.datestarted as \"Profile Started\", pv.datecompleted as \"Profile Completed\",  fo.path as \"File path\", fo.name as \"File Name\", fo.filesize as \"FileSize\",fo.lastmodified as \"Last Modified Date\",fo.CLASSIFICATIONTEXT, f.name as \"Format Name\", f.puid as \"PUID\", f.version as \"Format Version\", f.mimetype as \"Mimetype\",f.warning from profile p, profilevolume pv, fileobject fo, format f where pv.profile_id = p.id and fo.profilevolume_id = pv.id and f.fileobject_id = fo.id "+ profileName;
                    
                }
            }
           
        if(!filterString.trim().equals("%") && !filterString.trim().equals("")){
            selectQuery += filterString;
        }
        
      //  System.out.println(selectQuery);
        

//        if (profileName != null) {
//            Profile profile = getProfileByName(profileName);
//            if (profile != null) {
//                selectQuery = "select p.name as \"Profile Name\", p.datecreated as \"Profile Created\", pv.hostname as \"Hostname\", pv.volume  as \"Profile Directory\", pv.datestarted as \"Profile Started\", pv.datecompleted as \"Profile Completed\",  fo.path as \"File path\", fo.name as \"File Name\", fo.filesize as \"FileSize\",fo.lastmodified as \"Last Modified Date\",fo.CLASSIFICATIONTEXT, f.name as \"Format Name\", f.puid as \"PUID\", f.version as \"Format Version\", f.mimetype as \"Mimetype\",f.warning from profile p, profilevolume pv, fileobject fo, format f where pv.profile_id = p.id and fo.profilevolume_id = pv.id and f.fileobject_id = fo.id ";
//            }
//
//        }
        Session session = null;
       
        try {
            System.out.println("\nStarting bulk export..");
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            String coloumnHeaders = "select ''Profile Name'', '' Date Created '','' Hostname '',  ''Profile Directory '', '' Date Started '', '' Date Completed '',"+
                                    "'' File Path '' , ''File Name'' ,   ''File Size'' ,   ''Last Modified Date'' , ''Identification  Status '',   ''Format Name '', '' PUID '',  ''Format Version'' ,  ''Mime Type'' , ''Warning''"+
                                    "from sysibm.sysdummy1 UNION ALL   ";
            
            selectQuery = coloumnHeaders + selectQuery;
            
            String outputName = outputFile.getAbsolutePath();
            if(!outputName.trim().toUpperCase().endsWith(".CSV")){
                outputName+=".csv";
            }
            
            SQLQuery query = session.createSQLQuery("CALL SYSCS_UTIL.SYSCS_EXPORT_QUERY ('"  + selectQuery + "', '" + outputName + "' ,null,null,null)");
            query.executeUpdate();
            tx.commit();
            //String coloumnHeaders = "  Profile Name ,  Date Created , Hostname ,  Profile Directory ,  Date Started ,  Date Completed , " +
            //                " File path , File Name ,   File Size ,   Last Modified Date , Identification  Status ,   Format Name ,  PUID ,  Format Version ,  Mime Type , Warning  " ;

            
            
            

            System.out.println("Bulk export complete. " + outputName);
            
        } catch (Exception e) {
           
            log.error(e.toString());
            
            System.out.println(e.getLocalizedMessage());
            occuredError = e.getLocalizedMessage();
            if (e.getCause() instanceof SQLException)  {
                occuredError+="\n"+e.getCause().getLocalizedMessage();
                System.out.println(e.getCause().getLocalizedMessage());
            }
        } finally {
            if (session != null) {
                session.close();
            }
  
        }
        return occuredError;

    }


    /**
     * To be called within an open session, gets a profile by name
     *
     * @param name    The name
     * @param session The open session
     * @return The profile
     */
    private Profile getProfileInsideSession(String name, Session session) {
        Query query = session.createQuery("FROM Profile WHERE name = \'" + name.toUpperCase() + "\'");
        return (Profile) query.uniqueResult();
    }

    /**
     * Saves or updates a profile object
     *
     * @param profile the profile
     * @return the profile (updated)
     */
    public Profile saveOrUpdateProfile(Profile profile) {
        Session session = null;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            session.saveOrUpdate(profile);
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return profile;
    }

    /**
     * Saves a file object and its corresponding formats
     *
     * @param file    The file
     * @param formats The formats
     */
    public void saveFileAndFormats(FileObject file, Set<Format> formats) {
        Session session = null;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            session.saveOrUpdate(file);
            for (Format format : formats) {
                session.save(format);
            }
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
    }

    /**
     * Delete an existing profile if it exists
     *
     * @param profileName the name of the profile to delete
     */
    public void deleteProfile(String profileName) {
        Session session = null;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            Profile prof = getProfileInsideSession(profileName, session);
            if (prof != null) {
                session.delete(prof);
            }
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
    }

    /**
     * Gets all the profiles
     *
     * @return list
     */
    public List<Profile> getAllProfiles() {
        Session session = null;
        List<Profile> profiles = null;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            profiles = session.createQuery("from Profile").list();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return profiles;
    }

    /**
     * Gets all the profile names
     *
     * @return lists
     */
    public List<String> getAllProfileNames() {
        
        Session session = null;
        List<String> profiles = null;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            profiles = session.createQuery("SELECT name from Profile").list();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error("An error occured in retrieving profile names "+e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
      return profiles;
    }

    /**
     * Saves a volume
     *
     * @param vol the volume
     * @return the updated volume
     */
    public ProfileVolume saveVolume(ProfileVolume vol) {
        Session session = null;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            session.saveOrUpdate(vol);
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return vol;
    }

    /**
     * Get the total number of files in a profile
     *
     * @param profileName the profile name
     * @return total
     */
    public Long getTotalFilesInProfile(String profileName) {
        Session session = null;
        Long total = 0L;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            /*total = (Long)session.createQuery(
                    "SELECT COUNT (fileObject) " +
                    "FROM FileObject fileObject " +
                    "JOIN ProfileVolume profileVolume ON fileObject.profileVolume_Id = profileVolume.id " +
                    "WHERE profileVolume.profile.name = \'" + profileName + "\'").uniqueResult();*/
            total = (Long) session.createQuery("SELECT COUNT(fo) FROM FileObject fo WHERE fo.profileVolume.profile.name = \'" + profileName + "\'").uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Get the total size of files in a profile
     *
     * @param profileName the name
     * @return the total size
     */
    public Long getTotalBytesOfProfile(String profileName) {
        Session session = null;
        Long total = 0L;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Long) session.createQuery("SELECT SUM(fo.fileSize) FROM FileObject fo WHERE fo.profileVolume.profile.name = \'" + profileName + "\'").uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Get the average size of file in a profile
     *
     * @param profileName the profile name
     * @return the average file size
     */
    public Double getAverageBytesOfProfile(String profileName) {
        Session session = null;
        Double total = 0.0;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Double) session.createQuery("SELECT AVG(fo.fileSize) FROM FileObject fo WHERE fo.profileVolume.profile.name = \'" + profileName + "\'").uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Get the max file size in a profile
     *
     * @param profileName the profile name
     * @return max size
     */
    public Long getMaxSizeOfFileOfProfile(String profileName) {
        Session session = null;
        Long total = 0L;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Long) session.createQuery("SELECT MAX(fo.fileSize) FROM FileObject fo WHERE fo.profileVolume.profile.name = \'" + profileName + "\'").uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Get the min size of file in a profile
     *
     * @param profileName the profile name
     * @return the min size
     */
    public Long getMinFileSizeOfProfile(String profileName) {
        Session session = null;
        Long total = 0L;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Long) session.createQuery("SELECT MIN(fo.fileSize) FROM FileObject fo WHERE fo.profileVolume.profile.name = \'" + profileName + "\'").uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Get all volumes for a profile
     *
     * @param profileName the profile's name
     * @return list of volumes
     */
    public List<ProfileVolume> getAllVolumesForProfile(String profileName) {
        Session session = null;
        List<ProfileVolume> volumes = null;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            volumes = session.createQuery("from ProfileVolume pv where pv.profile.name = \'" + profileName + "\'").list();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return volumes;
    }

    /**
     * Get the number of files in a volume
     *
     * @param volId the volume id
     * @return number of files
     */
    public Long getNumberOfFilesInVolume(Long volId) {
        Session session = null;
        Long total = 0L;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Long) session.createQuery("SELECT COUNT (fo) FROM FileObject fo WHERE fo.profileVolume.id  = " + volId).uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Gets the total size of files in the volume
     *
     * @param volId the volume id
     * @return the total size
     */
    public Long getTotalSizeOfVolume(Long volId) {
        Session session = null;
        Long total = 0L;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Long) session.createQuery("SELECT SUM(fo.fileSize) FROM FileObject fo WHERE fo.profileVolume.id  = " + volId).uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Get the average file size in a volume
     *
     * @param volumeId the volume id
     * @return the average size
     */
    public Double getAverageBytesInVolume(Long volumeId) {
        Session session = null;
        Double total = 0.0;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Double) session.createQuery("SELECT AVG(fo.fileSize) FROM FileObject fo WHERE fo.profileVolume.id = " + volumeId).uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Get the maximum size of file in the volume
     *
     * @param volumeId the volume id
     * @return max size
     */
    public Long getMaxSizeOfFileInVolume(Long volumeId) {
        Session session = null;
        Long total = 0L;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Long) session.createQuery("SELECT MAX(fo.fileSize) FROM FileObject fo WHERE fo.profileVolume.id = " + volumeId).uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Get the min size of file in the volume
     *
     * @param volumeId the volume id
     * @return min size
     */
    public Long getMinFileSizeInVolume(Long volumeId) {
        Session session = null;
        Long total = 0L;
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            total = (Long) session.createQuery("SELECT MIN(fo.fileSize) FROM FileObject fo WHERE fo.profileVolume.id = " + volumeId).uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return total;
    }

    /**
     * Close the current database
     */
    public void close() {
        if (sessionFactory != null) {
            sessionFactory.close();
            sessionFactory = null;
        }
    }

    @Override
    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }

    /**
     * Compile a report against the database
     *
     * @param report the report
     * @param params the parameters
     * @return the print report
     */
    public JasperPrint fillReport(JasperReport report, Map<String, Object> params) {
        JasperPrint jasperPrint = null;
        StatelessSession session = null;
        Connection connection = null;
        try {
            session = sessionFactory.openStatelessSession();
            connection = session.connection();
            jasperPrint = JasperFillManager.fillReport(report, params, connection);
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            try {
                if (connection != null) {
                    connection.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
                log.error(e.toString());
            } finally {
                if (session != null) {
                    session.close();
                }
            }
        }
        return jasperPrint;
    }

    /**
     * Get information about the file age distribution in a volume
     *
     * @param volumeId the volume id
     * @return the age information
     */
    public List<Object[]> getVolumeYearsData(Long volumeId) {
        Session session = null;
        List<Object[]> result = new ArrayList<Object[]>();
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            StringBuilder build = new StringBuilder("select YEAR(fileobject.lastmodified),  sum(fileobject.filesize), count(fileobject.id) from fileobject ");
            build.append("where fileobject.profilevolume_id =").append(volumeId).append(" group by YEAR(fileobject.lastmodified) order by YEAR(fileobject.lastmodified)");
            SQLQuery query = session.createSQLQuery(build.toString());
            result = query.list();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return result;
    }

    /**
     * Get data about the format of files in a volume
     *
     * @param volumeId the volume id
     * @return list
     */
    public List<Object[]> getVolumeFormatData(Long volumeId) {
        Session session = null;
        List<Object[]> result = new ArrayList<Object[]>();
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            StringBuilder build = new StringBuilder("select format.puid, format.mimeType, format.name, format.version, count(fileobject.id),  sum(fileobject.filesize) from fileobject left outer join format ");
            build.append("on fileobject.id = format.fileobject_id where fileobject.profilevolume_id =").append(volumeId).append(" group by format.puid,format.mimeType, format.name,format.version order by format.puid");
            SQLQuery query = session.createSQLQuery(build.toString());
            result = query.list();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return result;
    }

    /**
     * Get the summary of a volume
     *
     * @param volumeId the volume id
     * @return the summary (as object[])
     */
    public Object[] getVolumeSummary(Long volumeId) {
        Session session = null;
        Object[] result = new Object[]{};
        try {
            session = sessionFactory.openSession();
            Transaction tx = session.beginTransaction();
            SQLQuery query = session.createSQLQuery("select COUNT(fileobject.id), SUM(fileobject.fileSize), MIN(fileobject.fileSize), MAX(fileobject.fileSize), AVG(fileobject.fileSize) FROM fileobject WHERE fileobject.profileVolume_id=" + volumeId);
            result = (Object[]) query.uniqueResult();
            tx.commit();
        } catch (Exception e) {
            e.printStackTrace();
            log.error(e.toString());
        } finally {
            if (session != null) {
                session.close();
            }
        }
        return result;
    }
}
