/**
 *  � The National Archives 2005-2008.  All rights reserved.
 * See Licence.txt for full licence details.
 * <p/>
 *
 *  DROID DCS Profile Tool
 * <p/>
 * <p/>
 * Developed By:
 * Tessella Support Services
 * 3 Vineyard Chambers
 * Abingdon, OX14 3PX
 * United Kingdom
 * <p/>
 * email:  info@tessella.com
 * web:    www.tessella.com
 * <p/>
 * <p/>
 * Created Date:   1-Nov-2008
 */
package uk.gov.nationalarchives.droid.profile.service;

import uk.gov.nationalarchives.droid.profile.FileFormatWalker;
import uk.gov.nationalarchives.droid.profile.database.ProfilingDBAccessPoint;
import uk.gov.nationalarchives.droid.profile.domain.*;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.AnnotationConfiguration;
import uk.gov.nationalarchives.droid.AnalysisController;
import uk.gov.nationalarchives.droid.MessageDisplay;
        

import java.io.File;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.sql.Connection;
import java.util.*;
import org.apache.commons.logging.*;




/**
 * This is the primary service class for profiling. It holds the DB access point. All DB access should go through this class.
 * The manager has the ability to change the current database, so it is critical that no other class has a direct link to the database.
 */
public class ProfilingManager {

    private ProfilingDBAccessPoint access;
    private File databaseLocation;
    private FileFormatWalker walker;
    private Log log = LogFactory.getLog(this.getClass());
    /**
     * Creates a manager with a DB handler for a database at the specified file location.
     *
     * @param derbyDatabaseLocation the DB location
     */
    public ProfilingManager(File derbyDatabaseLocation) throws Exception{
        
        boolean errorOccured = false; 
        try {
            if (access != null) {
                access.close();
                access = null;
            }
            SessionFactory sessFac = createDerbyFactoryFromFileObject(derbyDatabaseLocation);
            if (sessFac == null) {
                access = null;
            } else {
                databaseLocation = derbyDatabaseLocation;
                access = new ProfilingDBAccessPoint(sessFac);
            }

        } catch (Exception ex) {
            log.error("An error occured in connecting/creating Derby Database \n" + ex.toString());
            errorOccured = true;
        } finally {
            if (access == null || errorOccured) {
                throw new Exception();
            }
        }
       
    }

    public String export(File outputFile, List<String> profiles,String reportName, String filterString) {
       return  access.bulkExport(outputFile, profiles,reportName,filterString);
    }

    /**
     * Gets a profile by name
     *
     * @param profileName the name
     * @return the profile
     */
    public Profile getProfileByName(String profileName) {
        return access.getProfileByName(profileName.toUpperCase());
    }

    /**
     * Delete a profile
     *
     * @param profileName profile name
     */
    public void deleteProfile(String profileName) {
        access.deleteProfile(profileName);
    }

    /**
     * Returns all profile objects
     *
     * @return a list of profiles
     */
    public List<Profile> getAllProfiles() {
        return access.getAllProfiles();
    }

    /**
     * Creates a new profile, providing one with that name doesn't already exist
     *
     * @param name        the name
     * @param description the description
     * @return a profile
     */
    public Profile createProfile(String name, String description) {
        if (getProfileByName(name) != null) {
            return null;
        } else {
            String firstCleanName = name.replace("\'", "");
            String cleanName = firstCleanName.replace("\"", "");
            Profile profile = new Profile();
            profile.setName(cleanName.toUpperCase());
            profile.setDescription(description);
            profile.setDateCreated(new Date());
            return saveProfile(profile);
        }
    }

    /**
     * Create a new volume for the given profile
     *
     * @param profile the profile
     * @param target  the location for the volume
     * @param host    the host name
     * @return the new volume object
     */
    public ProfileVolume createVolumeForProfile(Profile profile, String target, String host) {
        ProfileVolume vol = new ProfileVolume();
        vol.setVolume(target);
        vol.setHostname(host);
        vol.setProfile(profile);
        profile.getProfileVolumes().add(vol);
        saveProfile(profile);
        return vol;
    }

    /**
     * Creates a new volume and marks it as profiling started
     *
     * @param profile the profile
     * @param target  the volume location
     * @param host    the host name
     * @return the new volume
     */
    public ProfileVolume createVolumeForProfileToStartNow(Profile profile, String target, String host) {
        ProfileVolume vol = new ProfileVolume();
        vol.setVolume(target);
        vol.setHostname(host);
        vol.setProfile(profile);
        vol.setDateStarted(new Date());
        profile.getProfileVolumes().add(vol);
        saveProfile(profile);
        return vol;
    }

    /**
     * Saves a profile
     *
     * @param prof the profile
     * @return the saved profile
     */
    public Profile saveProfile(Profile prof) {
        return access.saveOrUpdateProfile(prof);
    }

    /**
     * Save a file and its relevant format objects
     *
     * @param file    the file
     * @param formats list of formats
     */
    public void saveFileAndFormats(FileObject file, Set<Format> formats) {
        access.saveFileAndFormats(file, formats);
    }

    /**
     * Start profiling on a directory
     *
     * @param vol     the volume
     * @param control the analysis controller
     */
    public boolean performProfilingOnDirectory(ProfileVolume vol, AnalysisController control) {
        walker = new FileFormatWalker(vol, control, this);
        return walker.call();
    }

    /**
     * Start profiling on on a directory
     *
     * @param vol           the volume
     * @param signatureFile the signature file location
     */
    public boolean performProfilingOnDirectory(ProfileVolume vol, String signatureFile) {
        walker = new FileFormatWalker(this, vol, signatureFile);
        return walker.call();
    }

    /**
     * Has the DB access object been set
     *
     * @return boolean
     */
    public boolean isDBAccessSet() {
        return (access == null);
    }

    /**
     * Marks a volume as complete
     *
     * @param vol the vol
     */
    public void labelVolumeComplete(ProfileVolume vol) {
        vol.setDateCompleted(new Date());
        saveVolume(vol);
    }

    /**
     * Save a volume
     *
     * @param vol the volume
     * @return the updated volume
     */
    public ProfileVolume saveVolume(ProfileVolume vol) {
        return access.saveVolume(vol);
    }

    /**
     * Get the total number of files in the profile
     *
     * @param profileName the name
     * @return the total
     */
    public Long getTotalFilesInProfile(String profileName) {
        return access.getTotalFilesInProfile(profileName);
    }

    /**
     * Gets the total size of files in the profiles
     *
     * @param profileName the name
     * @return the total
     */
    public Long getTotalSizeOfProfile(String profileName) {
        return access.getTotalBytesOfProfile(profileName);
    }

    /**
     * Return the size of the largest file in a profile
     *
     * @param profileName the profile name
     * @return the size
     */
    public Long getMaxFileSizeOfProfile(String profileName) {
        return access.getMaxSizeOfFileOfProfile(profileName);
    }

    /**
     * Return the size of the smallest file in a profile
     *
     * @param profileName the profile name
     * @return the size
     */
    public Long getMinFileSizeOfProfile(String profileName) {
        return access.getMinFileSizeOfProfile(profileName);
    }

    /**
     * Return the average size of files in a profile
     *
     * @param profileName the profile name
     * @return the average size
     */
    public Double getAverageFileSizeInProfile(String profileName) {
        return access.getAverageBytesOfProfile(profileName);
    }

    /**
     * Get all volumes associated with the given profile
     *
     * @param profileName the profile name
     * @return the volumes
     */
    public List<ProfileVolume> getAllVolumesForProfile(String profileName) {
        return access.getAllVolumesForProfile(profileName);
    }

    /**
     * Return the number of files in a volume
     *
     * @param vol the volume
     * @return the number of files
     */
    public Long getNumberOfFilesInVolume(ProfileVolume vol) {
        return access.getNumberOfFilesInVolume(vol.getId());
    }

    /**
     * Returns the total size of a volume
     *
     * @param vol the volume
     * @return the total size
     */
    public Long getTotalSizeOfVolume(ProfileVolume vol) {
        return access.getTotalSizeOfVolume(vol.getId());
    }

    /**
     * Return the maximum size of file in a volume
     *
     * @param vol the volume
     * @return the max size
     */
    public Long getMaxSizeOfFileInVolume(ProfileVolume vol) {
        return access.getMaxSizeOfFileInVolume(vol.getId());
    }

    /**
     * Return the minimum size of file in a volume
     *
     * @param vol the volume
     * @return the min size
     */
    public Long getMinSizeOfFileInVolume(ProfileVolume vol) {
        return access.getMinFileSizeInVolume(vol.getId());
    }

    /**
     * Return the average size of file in a volume
     *
     * @param vol the volume
     * @return the average size
     */
    public Double getAverageSizeOfFileInVolume(ProfileVolume vol) {
        return access.getAverageBytesInVolume(vol.getId());
    }

    /**
     * Change the location of the derby database
     *
     * @param location the new location
     * @return success/failure
     */
    public boolean changeDerbyDatabase(File location) {
        SessionFactory sessFac = createDerbyFactoryFromFileObject(location);
        if (sessFac == null) {
            this.access = null;
            return false;
        } else {
            this.access.close();
            this.access = new ProfilingDBAccessPoint(sessFac);
            this.databaseLocation = location;
            return true;
        }
    }

    /**
     * Creates a new volume for the given profile, and marks it as started profiling
     *
     * @param profileName the profile name
     * @param newLocation the location for volume
     * @return the volume object
     */
    public ProfileVolume prepareVolumeForProfilingImmediately(String profileName, File newLocation) {
        Profile profile = getProfileByName(profileName);

        if (profile == null) {
            profile = createProfile(profileName, "");
        }

        String hostname = null;
        try {
            hostname = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            e.printStackTrace();
            log.error(e.toString());
        }

        return createVolumeForProfileToStartNow(profile, newLocation.getAbsolutePath(), hostname);
    }

    /**
     * Get the table entries for swing GUI
     *
     * @param profileName the profile name
     * @return an object[] of data
     */
    public Object[] getTableEntriesForSwing(String profileName) {
        List<Object[]> list = new ArrayList<Object[]>();
        for (ProfileVolume vol : getAllVolumesForProfile(profileName)) {
            Object[] tempArray = new Object[]{vol.getHostname(), vol.getVolume(), vol.getDateStarted(), vol.getDateCompleted()};
            list.add(tempArray);
        }
        return list.toArray();
    }

    /**
     * Get a list of all available profile names
     *
     * @return list
     */
    public List<String> getAllProfileNames() {
        return access.getAllProfileNames();
    }
    
    
/**
 * 
 * @param dir File whose size is sought
 * @return The size of a file/directory in MB
 */
    
    long getDirSizeInMB(File dir) {
        long size = 0;
        if (dir.isFile()) {
            size = dir.length();
        } else {
            File[] subFiles = dir.listFiles();

            for (File file : subFiles) {
                if (file.isFile()) {
                    size += file.length();
                } else {
                    size += this.getDirSizeInMB(file);
                }

            }
        }

        return ((size/1024)/1024);
    }
    
    
    /**
     * Create a derby session factory from a file object
     *
     * @param file the file object
     * @return the session factory
     */
    private SessionFactory createDerbyFactoryFromFileObject(File file) {
        try{
        if (file.isFile()) {
            log.error("Invalid Derby Database folder was selected: "+ file.getAbsolutePath());
            return null;
        } else {
            AnnotationConfiguration annotationConfiguration = new AnnotationConfiguration();
            if (file.exists() && file.isDirectory()) {
                annotationConfiguration.setProperty("hibernate.connection.url", "jdbc:derby:" + file.getAbsolutePath());
                log.info("Connecting to existing Derby Database: "+ file.getAbsolutePath());
                System.out.println("Connecting to existing Derby Database: "+ file.getAbsolutePath());
            } else {
                annotationConfiguration.setProperty("hibernate.connection.url", "jdbc:derby:" + file.getAbsolutePath() + ";create=true");
                log.info("Creating new Derby Database: "+ file.getAbsolutePath());
                System.out.println("Creating new Derby Database: "+ file.getAbsolutePath());
                //ensure there is enough disk space to create database
                File parentFile = file.getParentFile();
                if (parentFile != null) {
                    double freeSpace = ((parentFile.getUsableSpace() / 1024d) / 1024d);
                    //mimium file space needed for creating a database is about 2MB
                    if (freeSpace < 3) {
                        String error = "Cannot create database: " + file.getAbsolutePath() + ".\nThere is not enough free disk space on location: " + parentFile.getAbsolutePath() + ".\nDROID needs a minimum of 3.0 MB to create a database.\nPlease free space on this drive and try again";
                        MessageDisplay.generalError(error);
                        System.out.println(error);
                        log.error(error);
                        return null;
                    }
                }

            }
            annotationConfiguration.setProperty("hibernate.connection.driver_class", "org.apache.derby.jdbc.EmbeddedDriver");
            annotationConfiguration.setProperty("hibernate.connection.username", "sa");
            annotationConfiguration.setProperty("hibernate.connection.password", "");
            annotationConfiguration.setProperty("hibernate.hbm2ddl.auto", "update");
            annotationConfiguration.setProperty("hibernate.dialect", "org.hibernate.dialect.DerbyDialect");
            annotationConfiguration.setProperty("hibernate.connection.release_mode", "after_transaction");

            annotationConfiguration.setProperty("hibernate.connection.provider_class", "org.hibernate.connection.C3P0ConnectionProvider");
            annotationConfiguration.setProperty("hibernate.c3p0.min_size", "4");
            annotationConfiguration.setProperty("hibernate.c3p0.max_size", "10");
            annotationConfiguration.setProperty("hibernate.c3p0.timeout", "1800");
            annotationConfiguration.setProperty("hibernate.c3p0.max_statements", "0");

            annotationConfiguration.setProperty("hibernate.connection.isolation", String.valueOf(Connection.TRANSACTION_READ_UNCOMMITTED));

            annotationConfiguration.addAnnotatedClass(Profile.class);
            annotationConfiguration.addAnnotatedClass(FileObject.class);
            annotationConfiguration.addAnnotatedClass(Format.class);
            annotationConfiguration.addAnnotatedClass(Property.class);
            annotationConfiguration.addAnnotatedClass(ProfileVolume.class);
            SessionFactory sF = annotationConfiguration.buildSessionFactory();
            return sF;
        }
        }catch(Exception ex){
            log.error("An error occured in connecting to the Derby database");
            return null;
        }
        
    }

    /**
     * Fill a report with info from the DB
     *
     * @param report the report
     * @param params the parameters
     * @return the filled report
     */
    public JasperPrint fillReport(JasperReport report, Map<String, Object> params) {
        return access.fillReport(report, params);
    }

    /**
     * Return info about a volume's file ages for GUI
     *
     * @param vol the volume
     * @return list
     */
    public List<Object[]> getVolumeYearInfo(ProfileVolume vol) {
        return access.getVolumeYearsData(vol.getId());
    }

    /**
     * Return info about a volume's format info for GUI
     *
     * @param vol the volume
     * @return list
     */
    public List<Object[]> getVolumeFormatInfo(ProfileVolume vol) {
        return access.getVolumeFormatData(vol.getId());
    }

    /**
     * Get the location of the current database
     *
     * @return the file location
     */
    public File getDatabaseLocation() {
        return databaseLocation;
    }

    /**
     * Pause the walker
     */
    public void pauseWalker() {
        if (walker != null) {
            walker.pause();
        }
    }

    /**
     * Cancel the walker
     */
    public void cancelWalker() {
        if (walker != null) {
            walker.cancel();
        }
    }

    /**
     * Restart the walker
     */
    public void restartWalker() {
        if (walker != null) {
            walker.restart();
        }
    }

    /**
     * Check to see if the walker is paused
     *
     * @return boolean
     */
    public Boolean isWalkerPaused() {
        if (walker != null) {
            return walker.isPaused();
        } else {
            return null;
        }
    }

    /**
     * Get the summary from the walker
     *
     * @return summary
     */
    public VolumeSummary getWalkerSummary() {
        return walker.getSummary();
    }

    /**
     * Get information to update the profiling dialog with
     *
     * @param vol the volume
     * @return summary data
     */
    public Object[] getProfilingUpdateInformation(ProfileVolume vol) {
        return access.getVolumeSummary(vol.getId());
    }
}
