package fedora.services.sipcreator;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.Properties;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JApplet;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.filechooser.FileFilter;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

//import org.apache.log4j.xml.DOMConfigurator;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import ch.docuteam.docutools.out.Logger;
import ch.docuteam.docutools.translations.I18N;

import uk.gov.nationalarchives.droid.AnalysisController;
import beowulf.gui.Utility;
import beowulf.util.ExtensionFileFilter;
import edu.harvard.hul.ois.jhove.Checksummer;
import fedora.services.sipcreator.mimetype.MimetypeDetector;
import fedora.services.sipcreator.tasks.ConversionRulesTask;
import fedora.services.sipcreator.tasks.FileSelectTask;
import fedora.services.sipcreator.tasks.MetadataEntryTask;
import fedora.services.sipcreator.tasks.SubmissionAgreementTask;

import java.io.FileInputStream;

/**
 * This class is the overarching "system" class for the entire application. A
 * lot of the public methods in this class exist as instance methods because of
 * problems with how different browsers handle static methods in applets.
 * Apparently, some browsers only instantiate one JVM *across multiple broswer
 * windows*, creating obvious issues with static fields. <br>
 * <br>
 * 
 * @author Andy Scukanec - (ags at cs dot cornell dot edu)
 */
public class SIPCreator extends JApplet implements Constants {

	/** The label used to display the currently open file/directory */
	private JLabel currentFileLabel = new JLabel();

	/**
	 * The context of using SIP-Creator (creating a SIP by producers or
	 * verifying an SIP by the archive)
	 */
	private boolean create = true;

	/** The data structure for storing non-java system wide properties */
	private Properties sipCreatorProperties = new Properties(); // @jve:decl-index=0:

	/** The tool for determining MIME type */
	private MimetypeDetector mimetypeDetector;

	/** The tool for parsing XML documents */
	private DocumentBuilder documentBuilder;

	private AnalysisController droid;

	/** The tool for file selection */
	private JFileChooser fileChooser = new JFileChooser(System.getProperty("user.home"));
	/** The filter which accepts XML files and directories */
	private FileFilter xmlFilter = new ExtensionFileFilter("xml");
	/** The filter which accepts ZIP files and directories */
	private FileFilter zipFilter = new ExtensionFileFilter("zip");

	/** The mapping between metadata classes and display names */
	private Properties knownMetadataClasses = new Properties();

	// Task elements, these are larger classes that encapsulate everything
	// necessary for the end user to perform a certain task.
	/** All the UI views and handlers to deal with conversion rule editing */
	private ConversionRulesTask conversionRulesTask;
	/** All the UI views and handlers to deal with metadata entry */
	private MetadataEntryTask metadataEntryTask;
	/** All the UI views and handlers to deal with selecting data sources */
	private FileSelectTask fileSelectTask;
	/** All the UI views and handlers to deal with the submission agreement */
	private SubmissionAgreementTask submissionAgreementTask;

	public void init() {
		Logger.setConfigFile(getURL(LOG4J_CONFIG_NAME));

		try {
			String lang = System.getProperty("user.language");
			I18N.initialize(lang, "ch.docuteam.sipcreator.translations.Translations");
			sipCreatorProperties.load(getInputStream(getURL(CONFIG_FILE_NAME).toString()));
			if (sipCreatorProperties.containsKey("sipcreator.language")) {
				lang = sipCreatorProperties.getProperty("sipcreator.language");
				I18N.initialize(lang, "ch.docuteam.sipcreator.translations.Translations");
			}
			Logger.info("Setting language to " + lang);
		} catch (IOException ioe) {
			Utility.showExceptionDialog(this, ioe, I18N.translate("ExPropsNotLoaded"));
		}

		submissionAgreementTask = new SubmissionAgreementTask(this);
		fileSelectTask = new FileSelectTask(this);
		metadataEntryTask = new MetadataEntryTask(this);
		conversionRulesTask = new ConversionRulesTask(this);

		// Instantiate the XML Parser
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			factory.setNamespaceAware(true);
			factory.setIgnoringElementContentWhitespace(true);
			factory.setIgnoringComments(true);
			documentBuilder = factory.newDocumentBuilder();
		} catch (ParserConfigurationException pce) {
			Utility.showExceptionDialog(this, pce, I18N.translate("ExXMLParserNotInitialized"));
		}

		// Add the JSplitPane
		Container cp = getContentPane();
		cp.setLayout(new BorderLayout(5, 5));
		cp.add(createCenterPanel(), BorderLayout.CENTER);
		cp.add(createHelpBar(), BorderLayout.SOUTH);

		// Perform default loading activities
		loadDefaults();
	}

	/**
	 * This method returns a JPanel containing the label displaying the
	 * currently open file and the help button bar. Collectively, they form the
	 * status bar. <br>
	 * <br>
	 * 
	 * @return A JPanel displaying system wide information that is visible
	 *         regardless of which task is active.
	 */
	private JComponent createHelpBar() {
		JPanel help = new JPanel(new BorderLayout());

		help.add(currentFileLabel, BorderLayout.CENTER);
		help.add(new JButton(new OpenHelpAction()), BorderLayout.EAST);

		return help;
	}

	/**
	 * This method returns a JTabbedPane showing with a single tab for each
	 * task. This is just a convenience method. <br>
	 * <br>
	 * 
	 * @return A JTabbedPane where each tab represents a task.
	 */
	private JComponent createCenterPanel() {
		JTabbedPane leftPanel = new JTabbedPane();

		leftPanel.addTab("1. " + I18N.translate("SubmissionAgreement"), submissionAgreementTask);
		leftPanel.addTab("2. " + I18N.translate("FileSelection"), fileSelectTask);
		if (Boolean.parseBoolean(sipCreatorProperties.getProperty("sipcreator.metadata.enabled")))
			leftPanel.addTab("3. " + I18N.translate("Metadata"), metadataEntryTask);
		// So far, the necessary conversion rules are already included in the
		// distribution or automatically inserted when adding metadata.
		// Nf/Docuteam, 17.06.2011
		// leftPanel.addTab("4. " + I18N.translate("ConversionRules"),
		// conversionRulesTask);

		return leftPanel;
	}

	/**
	 * This interprets the information stored in the properties file for the
	 * system. It will handle things such as loading in the default conversion
	 * rules file and loading in a list of known metadata classes. All errors
	 * will result in a message dialog being displayed to the user.
	 */
	private void loadDefaults() {
		String value;

		try {
			value = sipCreatorProperties.getProperty(DEFAULT_CRULES);
			if (value != null) {
				loadDefaultConversionRules(value);
			}
		} catch (Exception e) {
			Utility.showExceptionDialog(this, e, I18N.translate("ExCRulesNotInitialized"));
		}

		try {
			value = sipCreatorProperties.getProperty(METADATA_CLASS_LIST);
			if (value != null) {
				loadDefaultMetadataClasses(value);
			}
		} catch (Exception e) {
			Utility.showExceptionDialog(this, e, I18N.translate("ExMDClassDetectionNotInitialized"));
		}

		try {
			value = sipCreatorProperties.getProperty(MIMETYPE_CLASS_NAME);
			if (value != null) {
				loadDefaultDetector(value);
			}
		} catch (Exception e) {
			Utility.showExceptionDialog(this, e, I18N.translate("ExMimetypeDetectionNotInitialized"));
		}
	}

	/**
	 * This method handles loading in the default conversion rules file. The
	 * parameter is expected to be the path of the conversion rules file
	 * relative to the codebase, or an absolute URL. <br>
	 * <br>
	 * 
	 * @param value
	 *            The relative path or absolute URL of the conversion rules.
	 * @throws Exception
	 *             Thrown if there is a problem locating the file or parsing the
	 *             file.
	 */
	private void loadDefaultConversionRules(String value) throws Exception {
		try {
			ConversionRules crules = new ConversionRules(documentBuilder.parse(getURL(value).toString()));
			conversionRulesTask.updateRules(value, crules);
		} catch (Exception mue) {
			InputSource is = new InputSource(getInputStream(value));
			ConversionRules crules = new ConversionRules(documentBuilder.parse(is));
			conversionRulesTask.updateRules(value, crules);
		}
	}

	/**
	 * This method handles loading in the mapping of display names to metadata
	 * class names. The parameter is a path of the file defining the mapping
	 * using the java.util.Properties syntax, and is relative to the codebase. <br>
	 * <br>
	 * 
	 * @param value
	 *            The relative path to the file containing the mapping.
	 * @throws IOException
	 *             Thrown if there is a problem reading the file indicated.
	 */
	private void loadDefaultMetadataClasses(String value) throws IOException {
		knownMetadataClasses.load(getInputStream(value));
	}

	/**
	 * This method is responsible for instantiating the default mime type
	 * detection tool. The parameter is expected to be a fully qualified class
	 * name. The tool is instantiated using reflection, and the given class must
	 * be a subclass of MimetypeDetector and have a single argument constructor
	 * taking in a SIPCreator reference. <br>
	 * <br>
	 * 
	 * @param value
	 *            The fully qualified class name of the detection class.
	 * @throws Exception
	 *             Thrown if there is a problem with the underlying reflection
	 *             methods.
	 */
	private void loadDefaultDetector(String value) throws Exception {
		Class<?> detectorClass = Class.forName(value);

		if (!MimetypeDetector.class.isAssignableFrom(detectorClass)) {
			throw new RuntimeException(I18N.translate("ExMimetypeDetectionNotInitialized") + "\n" + I18N.translate("ExMimetypeDetectorNotExtended"));
		}

		Constructor<?> constructor = detectorClass.getConstructor(new Class[] { SIPCreator.class });
		mimetypeDetector = (MimetypeDetector) constructor.newInstance(new Object[] { this });
	}

	/**
	 * This method takes a relative path in string form, resolves it against the
	 * codebase URL, and returns the input stream associated with the resulting
	 * resource. If there is a problem opening the stream or creating the
	 * absolute URL, null is returned and an error message is printed to
	 * System.err. <br>
	 * <br>
	 * 
	 * @param filename
	 *            The name to resolve against the code base.
	 * @return The input stream of the associated resource.
	 */
	public InputStream getInputStream(String filename) {
		try {
			return (new URL(getCodeBase(), filename)).openStream();
		} catch (IOException ioe) {
			System.err.println(I18N.translate("ExIOinSIPCreatorIS") + ": " + ioe.getMessage());
			return null;
		}
	}

	/**
	 * This method takes a relative path in string form, resolves it against the
	 * code base URL, and returns the resulting URL. If there is a problem
	 * creating the URL, null is returned and an error message is printed to
	 * System.err. <br>
	 * <br>
	 * 
	 * @param filename
	 *            The name to resolve against the code base.
	 * @return The absolute URL of the associated resource.
	 */
	public URL getURL(String filename) {
		try {
			return new URL(getCodeBase(), filename);
		} catch (MalformedURLException mue) {
			System.err.println(I18N.translate("ExMalformedURLinSIPCreatorgetURL") + ": " + mue.getMessage());
			return null;
		}
	}

	/**
	 * Returns the properties for the sipcreator system. The file defining these
	 * proprties is indicated by the String Constants.CONFIG_FILE_NAME. <br>
	 * <br>
	 * 
	 * @return The properties for the sipcreator system.
	 */
	public Properties getProperties() {
		return sipCreatorProperties;
	}

	/**
	 * Given a File object, this method will return the guessed mime type of
	 * that file. The guess made will depend on the underlying mime type
	 * detection library used. This library is defined in the properties file
	 * for the SIPCreator system, using the key Constants.MIMETYPE_CLASS_NAME. <br>
	 * <br>
	 * 
	 * @param file
	 *            The file whose mime type will be returned.
	 * @return The mime type of the given file.
	 */
	public String getMimeType(File file) {
		return mimetypeDetector.getMimeType(file);
	}

	/**
	 * Given a File object, this method will return the calculated checksum of
	 * that file. <br>
	 * <br>
	 * 
	 * @param filePath
	 *            The file whose checksum will be returned.
	 * @return The checksum of the given file.
	 */
	public String calculateChecksum(String filePath) throws FileNotFoundException, IOException {
		Checksummer ckSummer = new Checksummer();

		DataInputStream instrm = new DataInputStream(new BufferedInputStream(new FileInputStream(filePath)));
		try {
			while (true) {
				int ch = instrm.readUnsignedByte();
				if (ckSummer != null) {
					ckSummer.update(ch);
				}
			}
		} catch (EOFException e) {
			/* This is the normal way for detecting we're done */
		}
		instrm.close();

		return ckSummer.getMD5();
	}

	/**
	 * Returns the metadata entry task object. <br>
	 * <br>
	 * 
	 * @return The metadata entry task object.
	 */
	public MetadataEntryTask getMetadataEntryTask() {
		return metadataEntryTask;
	}

	/**
	 * Returns the conversion rules task object. <br>
	 * <br>
	 * 
	 * @return The conversion rules task object.
	 */
	public ConversionRulesTask getConversionRulesTask() {
		return conversionRulesTask;
	}

	/**
	 * Returns the file selection task object. <br>
	 * <br>
	 * 
	 * @return The file selection task object.
	 */
	public FileSelectTask getFileSelectTask() {
		return fileSelectTask;
	}

	/**
	 * This method sets the text on the label used to indicate the currently
	 * open file/folder providing the root to the content tree in use. <br>
	 * <br>
	 * 
	 * @param text
	 *            The name of the currently open file/folder.
	 */
	public void setFileLabelText(String text) {
		currentFileLabel.setText(text);
	}

	/**
	 * Returns a properties object defining a mapping from display names to
	 * class names of metadata subclasses. This mapping is defined by a file
	 * whose name is accessed in the SIPCreator properties file with the key
	 * Constants.METADATA_CLASS_LIST. <br>
	 * <br>
	 * 
	 * @return A properties object mapping from display names to class names.
	 */
	public Properties getKnownMetadataClasses() {
		return knownMetadataClasses;
	}

	/**
	 * This method takes in an input source object and runs it through the
	 * system parser. This method exists so that only one parser need exist
	 * throughout the whole system. <br>
	 * <br>
	 * 
	 * @param is
	 *            The input source of the XML data to parsed.
	 * @return The parsed XML Document.
	 * @throws IOException
	 *             If there is a problem reading the input source.
	 * @throws SAXException
	 *             If the input source is syntactically invalid.
	 */
	public Document parseXML(InputSource is) throws IOException, SAXException {
		return documentBuilder.parse(is);
	}

	/**
	 * @return the droid
	 */
	public AnalysisController getDroid() {
		if (droid==null) {
			try {
				Logger.info("initializing DROID...");
				droid = new AnalysisController();
				Logger.info("reading DROID config file: " + PRONOM_CONFIG_NAME);
				droid.readConfiguration(getURL(PRONOM_CONFIG_NAME));
				Logger.info("reading DROID signature file: " + droid.getSignatureFileName());
				droid.readSigFile(getURL(droid.getSignatureFileName()));
			} catch (Exception e) {
				Utility.showExceptionDialog(this, e, I18N.translate("ExDROIDNotInitialized"));
			}
		}
		return droid;
	}

	/**
	 * Returns the system file chooser. This is done so that only one file
	 * chooser object need exist, and so that all the file chooser contexts
	 * remember the last used directory. <br>
	 * <br>
	 * 
	 * @return The system file chooser.
	 */
	public JFileChooser getFileChooser() {
		return fileChooser;
	}

	/**
	 * Returns the file filter used to accept XML files. <br>
	 * <br>
	 * 
	 * @return The file filter used to accept XML files.
	 */
	public FileFilter getXMLFilter() {
		return xmlFilter;
	}

	/**
	 * Returns the file filter used to accept ZIP files. <br>
	 * <br>
	 * 
	 * @return The file filter used to accept ZIP files.
	 */
	public FileFilter getZIPFilter() {
		return zipFilter;
	}

	/**
	 * Returns the submission agreement task object. <br>
	 * <br>
	 * 
	 * @return the submissionAgreementTask
	 */
	public SubmissionAgreementTask getSubmissionAgreementTask() {
		return submissionAgreementTask;
	}

	/**
	 * @return the create
	 */
	public boolean isCreate() {
		return create;
	}

	/**
	 * @param create
	 *            the create to set
	 */
	public void setCreate(boolean create) {
		this.create = create;
	}

	public class OpenHelpAction extends AbstractAction {

		private static final long serialVersionUID = 3763096349595678519L;

		private OpenHelpAction() {
			putValue(Action.NAME, I18N.translate("ShowHelp"));
			URL imgURL = getURL(HELP_IMAGE_NAME);
			putValue(Action.SMALL_ICON, new ImageIcon(getImage(imgURL)));
			putValue(Action.SHORT_DESCRIPTION, I18N.translate("ShowHelpTT"));
		}

		public void actionPerformed(ActionEvent ae) {
			try {
				java.awt.Desktop.getDesktop().browse(new URI("http://wiki.docuteam.ch/doku.php?id=oais:sip-creator"));
			} catch (Exception e) {
				Logger.error(I18N.translate("ErrorOpeningHelp"), e);
			}
		}

	}

}
