package fedora.services.sipcreator.metadata;

import java.util.Observable;

import org.w3c.dom.Element;

import fedora.services.sipcreator.Constants;
import fedora.services.sipcreator.SelectableEntry;

/**
 * This class defines the interface for all metadata objects.  Implementors
 * of this class need to implement all abstract methods, and are strongly
 * encouraged to create two constructors as well.  The blank (default)
 * constructor is required, and the single argument constructur is required
 * if the implementing subclass needs to be saved and reloaded from an XML
 * file.  The SIPCreator system uses reflection to call the blank constructor
 * when a new instance of the metadata subclass is created from scratch.  The
 * system also uses reflection when it reads in a serialized instance.  The
 * system attempts to recreate the serialized instance by passing the XML
 * node under which the instance was serialized to a constructor for the
 * target subclass.  If that constructor does not exist, deserialization will
 * fail.  Note that this does not mean the SIP file will be invalid, simply
 * that it will not be a SIP file which can be saved, reloaded, edited,
 * etc.
 * <br><br>
 * @author Andy Scukanec - (ags at cs dot cornell dot edu)
 */
public abstract class Metadata extends Observable implements Constants {

    /** This long holds the last id value generated */
    private static long lastID = 0;

    /**
     * This method generates a unique string to be used as an ID for Metadata
     * and SelectableEntry objects.  This id is generated by using the current
     * takeing system time and prepending an underscore character.  The
     * underscore is prepended so as to conform to the METS requirements on
     * the ID field.  This method is thread safe.
     * <br><br>
     * @return A unique String to be used as an ID.
     */
    public static synchronized String getNextID() {
        long nextID = System.currentTimeMillis();
        while (nextID == lastID) {
            try { Thread.sleep(10); }
            catch (InterruptedException ie) {}
            nextID = System.currentTimeMillis();
        }
        lastID = nextID;
        return "_" + Long.toString(lastID);
    }
    
    /**
     * This field holds the id for the metadata.  This field shold be unique
     * to this particular piece of metadata.  It is declared final to prevent
     * any problems from unplanned modifications.
     */
    private final String id;
    
    /** This string is the label associated with this Metadata  */
    private String label = new String();
    
    /**  */
    private String type = new String();
    
    /**  */
    private SelectableEntry entry;
    
    /**
     * The default constructor for this class simply sets the ID field to
     * be the result of calling <code>getNextID()</code>.  All other fields
     * are left blank.
     */
    public Metadata() {
        id = getNextID();
    }
    
    /**
     * This constructor grabs the id field from the ID attribute in the given
     * node.  If that attribute doesn't exist, <code>getNextID</code> is used
     * to generate the id field for the produced object.
     * <br><br>
     * @param xmlNode The XML node possibly holding the ID field.
     */
    public Metadata(Element xmlNode) {
        if (xmlNode.hasAttributeNS(METS_NS, "ID")) {
            id = xmlNode.getAttributeNS(METS_NS, "ID");
        } else {
            id = getNextID();
        }
    }
    
    /**
     * Returns the id of this Metadata object. 
     * <br><br>
     * @return The id of this Metadata object.
     */
    public String getID() {
        return id;
    }
    
    /**
     * Returns the user-defined label associated with this Metadata object.
     * <br><br>
     * @return The user-defined label associated with this Metadata object.
     */
    public String getLabel() {
        return label;
    }
    
    /**
     * Returns the user-defined type associated with this Metadata object.
     * <br><br>
     * @return The user-defined type associated with this Metadata object.
     */
    public String getType() {
        return type;
    }

    /**
     * Returns the SelectableEntry object containing this Metadata object.
     * <br><br>
     * @return The SelectableEntry object containing this Metadata object.
     */
    public SelectableEntry getEntry() {
        return entry;
    }
    
    /**
     * This method should return a "short" name for this Metadata object.  No
     * uniqueness constraints are placed upon this name, and it is meant mainly
     * for display purposes.
     * <br><br>
     * @return A string which can be used as a "short" name for this Metadata
     * object.
     */
    public abstract String getShortName();
    
    /**
     * This method uses <code>getShortName()</code> and generates a string
     * which represents this Metadata object and is globally unique.  This
     * method relies on the entry for this Metadata object being set.
     * <br><br>
     * @return A globally unique name for this Metadata object.
     */
    public String getDescriptiveName() {
        return entry.getDescriptiveName() + " - (" + id + ") " + getShortName();
    }

    /**
     * Sets the SelectableEntry object containing this Metadata object.
     * <br><br>
     * @param newEntry The new SelectableEntry containing this Metadata object.
     */
    public void setEntry(SelectableEntry newEntry) {
        entry = newEntry;
    }
    
    /**
     * Sets the user-defined label.
     * <br><br>
     * @param newLabel The user-defined label.
     */
    public void setLabel(String newLabel) {
        label = newLabel;
        setChanged();
        notifyObservers();
    }
    
    /**
     * Sets the user-defined type.
     * <br><br>
     * @param newType The user-defined type.
     */
    public void setType(String newType) {
        type = newType;
        setChanged();
        notifyObservers();
    }
    
    /**
     * This method should return an instance of MetadataPanel which will be
     * used to display and manipulate the information stored in this Metadata
     * object.  It is recommended that a new instance of MetadataPanel be
     * returned upon subsequent invocations of this method.
     * <br><br>
     * @return A MetadataPanel used to display and interact with this Metadata
     * object.
     */
    public abstract MetadataPanel getPanel();

    /**
     * This method should return an XML String for this piece of metadata.  If
     * namespaces need to be used, they should be declared inside of the
     * returned node, rather than assuming they will be declared at some higher
     * level in the containing Document.  This XML string must be well formed
     * and should validate as well.
     * <br><br>
     * @return A string that can be used to serialize this metadata object to
     * an XML file.
     */
    public abstract String getAsXML();
    
    public boolean equals(Object o) {
        if (!(o instanceof Metadata)) return false;
        Metadata m = (Metadata)o;
        return m.getID().equals(id);
    }
    
}