package fedora.services.sipcreator.tasks;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.lang.reflect.Constructor;
import java.net.URL;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import beowulf.gui.PopupListener;
import beowulf.gui.Utility;
import fedora.services.sipcreator.Constants;
import fedora.services.sipcreator.ConversionRules;
import fedora.services.sipcreator.FileSystemEntry;
import fedora.services.sipcreator.MetadataNode;
import fedora.services.sipcreator.MetadataPanelWrapper;
import fedora.services.sipcreator.SIPCreator;
import fedora.services.sipcreator.SelectableEntry;
import fedora.services.sipcreator.SelectableEntryNode;
import fedora.services.sipcreator.SelectableEntryPanel;
import fedora.services.sipcreator.acceptor.FilterAcceptor;
import fedora.services.sipcreator.acceptor.IntersectionAcceptor;
import fedora.services.sipcreator.acceptor.SelectionAcceptor;
import fedora.services.sipcreator.metadata.Metadata;
import fedora.services.sipcreator.utility.CheckRenderer;

import ch.docuteam.docutools.translations.I18N;

public class MetadataEntryTask extends JPanel implements Constants {

	private FilterAction filterAction;
	private RemoveMetadataAction removeMetadataAction;
	private AddMetadataAction addMetadataAction;
//	private SaveSIPAction saveSIPAction;
	private CloseCurrentTabAction closeCurrentTabAction;
	private EventHandler eventHandler = new EventHandler();

	private FilterAcceptor filterAcceptor = new FilterAcceptor();
	private IntersectionAcceptor acceptor = new IntersectionAcceptor();

	// Data structures and UI components involved with the metadata entry task
	private DefaultTreeModel metadataTreeModel = new DefaultTreeModel(null);
	private JTree metadataTreeDisplay = new JTree(metadataTreeModel);

	private JCheckBox filterEnabledBox;
	private JTextField filterField = new JTextField();
	private JTabbedPane metadataView = new JTabbedPane();

	private SIPCreator creator;

	public MetadataEntryTask(SIPCreator newCreator) {
		creator = newCreator;
		I18N.initialize(creator.getProperties().getProperty("sipcreator.language"), "ch.docuteam.sipcreator.translations.Translations");

		filterAction = new FilterAction();
		removeMetadataAction = new RemoveMetadataAction();
		addMetadataAction = new AddMetadataAction();
//		saveSIPAction = new SaveSIPAction();
		closeCurrentTabAction = new CloseCurrentTabAction();

		acceptor.addAcceptor(filterAcceptor);
		acceptor.addAcceptor(new SelectionAcceptor(FileSystemEntry.FULLY_SELECTED | FileSystemEntry.PARTIALLY_SELECTED));
		acceptor.setAcceptsMetadata(true);

		metadataTreeDisplay.addMouseListener(eventHandler);
		metadataTreeDisplay.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
		metadataTreeDisplay.addMouseListener(new PopupListener(createPopupMenu()));
		metadataTreeDisplay.setCellRenderer(new CheckRenderer(false, creator));

		filterEnabledBox = new JCheckBox(filterAction);
		filterField.getDocument().addDocumentListener(filterAction);

		JSplitPane centerPane = new JSplitPane();
		centerPane.setLeftComponent(createLeftPanel());
		centerPane.setRightComponent(createRightPanel());
		centerPane.setOneTouchExpandable(true);
		centerPane.setResizeWeight(0.5);

		setLayout(new BorderLayout());
		add(centerPane, BorderLayout.CENTER);
	}

	private JComponent createRightPanel() {
		return metadataView;
	}

	private JComponent createLeftPanel() {
		JPanel left = new JPanel(new BorderLayout(5, 5));
		left.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
		left.add(createNorthPanel(), BorderLayout.NORTH);
		left.add(new JScrollPane(metadataTreeDisplay), BorderLayout.CENTER);
		return left;
	}

	private JPanel createNorthPanel() {
		JPanel tempP2 = new JPanel(new GridLayout(1, 0));
		tempP2.add(filterEnabledBox);
		tempP2.add(new JButton(closeCurrentTabAction));
//		tempP2.add(new JButton(saveSIPAction));

		JPanel result = new JPanel(new BorderLayout());
		result.add(filterField, BorderLayout.CENTER);
		result.add(tempP2, BorderLayout.EAST);

		return result;
	}

	private JPopupMenu createPopupMenu() {
		JPopupMenu result = new JPopupMenu();

		result.add(addMetadataAction);
		result.add(removeMetadataAction);

		return result;
	}

	public void updateTree(SelectableEntry newRoot) {
		metadataTreeModel.setRoot(new SelectableEntryNode(newRoot, null, acceptor, metadataTreeModel));
	}

	public void refreshTree() {
		metadataTreeModel.nodeStructureChanged((SelectableEntryNode) metadataTreeModel.getRoot());
	}

	public void closeCurrentTab() {
		closeCurrentTabAction.closeCurrentTab();
	}

	public void closeAllTabs() {
		while (metadataView.getTabCount() > 0) {
			metadataView.remove(0);
		}
	}

	public int getIndexByToolTip(String tip) {
		int index = 0;
		for (index = 0; index < metadataView.getTabCount(); index++) {
			if (metadataView.getToolTipTextAt(index).equals(tip)) {
				return index;
			}
		}
		return -1;
	}

	public void updateMetadata() {
		for (int ctr = 0; ctr < metadataView.getTabCount(); ctr++) {
			Component c = metadataView.getComponent(ctr);
			if (c instanceof SelectableEntryPanel) {
				((SelectableEntryPanel) c).updateMetadata();
			} else if (c instanceof MetadataPanelWrapper) {
				((MetadataPanelWrapper) c).updateMetadata();
			}
		}
	}

	public TreeNode getSelectedNode() {
		TreePath path = metadataTreeDisplay.getSelectionPath();
		if (path == null) {
			return null;
		}
		return (TreeNode) path.getLastPathComponent();
	}

	public FilterAction getFilterAction() {
		return filterAction;
	}

	public CloseCurrentTabAction getCloseCurrentTabAction() {
		return closeCurrentTabAction;
	}

//	public SaveSIPAction getSaveSIPAction() {
//		return saveSIPAction;
//	}

	public AddMetadataAction getAddMetadataAction() {
		return addMetadataAction;
	}

	public RemoveMetadataAction getRemoveMetadataAction() {
		return removeMetadataAction;
	}

	private class EventHandler extends MouseAdapter {

		public void mouseClicked(MouseEvent me) {
			if (me.getClickCount() != 2)
				return;

			TreePath path = metadataTreeDisplay.getSelectionPath();
			if (path == null)
				return;

			if (path.getLastPathComponent() instanceof SelectableEntryNode) {
				SelectableEntryNode node = (SelectableEntryNode) path.getLastPathComponent();
				int index = getIndexByToolTip(node.getEntry().getDescriptiveName());

				if (index == -1) {
					SelectableEntryPanel listPanel = new SelectableEntryPanel(node.getEntry());
					metadataView.addTab(node.getEntry().getShortName(), null, listPanel, node.getEntry().getDescriptiveName());
					metadataView.setSelectedComponent(listPanel);
				} else {
					metadataView.setSelectedIndex(index);
				}
			} else if (path.getLastPathComponent() instanceof MetadataNode) {
				MetadataNode node = (MetadataNode) path.getLastPathComponent();
				int index = getIndexByToolTip(node.getMetadata().getDescriptiveName());

				if (index == -1) {
					MetadataPanelWrapper panel = new MetadataPanelWrapper(node.getMetadata(), creator);
					metadataView.addTab(node.getMetadata().getShortName(), null, panel, node.getMetadata().getDescriptiveName());
					metadataView.setSelectedComponent(panel);
				} else {
					metadataView.setSelectedIndex(index);
				}
			}
		}

	}

	public class RemoveMetadataAction extends AbstractAction {

		public RemoveMetadataAction() {
			putValue(Action.NAME, I18N.translate("RemoveMetadata"));
			putValue(Action.SHORT_DESCRIPTION, I18N.translate("RemoveMetadataTT"));
		}

		public void actionPerformed(ActionEvent ae) {
			try {
				TreeNode lastElement = getSelectedNode();
				if (!(lastElement instanceof MetadataNode))
					return;
				Metadata metadata = ((MetadataNode) lastElement).getMetadata();
				SelectableEntry entry = metadata.getEntry();
				metadata.setEntry(null);
				entry.removeMetadata(metadata);

				int index = getIndexByToolTip(metadata.getDescriptiveName());
				if (index != -1) {
					metadataView.removeTabAt(index);
				}
			} catch (Exception e) {
				Utility.showExceptionDialog(creator, e);
			}
		}

	}

	public class AddMetadataAction extends AbstractAction {

		/** */
		private static final long serialVersionUID = 414125917858160620L;

		public AddMetadataAction() {
			putValue(Action.NAME, I18N.translate("AddMetadata"));
			putValue(Action.SHORT_DESCRIPTION, I18N.translate("AddMetadataTT"));
		}

		public void actionPerformed(ActionEvent ae) {
			try {
				TreeNode lastElement = getSelectedNode();
				if (!(lastElement instanceof SelectableEntryNode))
					return;
				SelectableEntry entry = ((SelectableEntryNode) lastElement).getEntry();

				String title = I18N.translate("AddMetadata");
				String msg = I18N.translate("MessageSelectMetadataClass");
				Object[] options = creator.getKnownMetadataClasses().keySet().toArray();
				if (options.length == 0) {
					JOptionPane.showMessageDialog(creator, I18N.translate("MessageUnknownMetadataType"));
					return;
				}
				Object selection = JOptionPane.showInputDialog(creator, msg, title, JOptionPane.QUESTION_MESSAGE, null, options, options[0]);
				if (selection == null)
					return;
				String selectedClassName = (String) creator.getKnownMetadataClasses().get(selection);
				Class<?> selectedClass = Class.forName(selectedClassName);
				Constructor<?> constructor = selectedClass.getConstructor((Class[]) null);
				Metadata metadata = (Metadata) constructor.newInstance((Object[]) null);

				addMetadataAction(entry, metadata);
			} catch (Exception e) {
				Utility.showExceptionDialog(creator, e);
			}
		}

		public void addMetadataAction(SelectableEntry entry, Metadata newMetadata) {
			ConversionRules rules = creator.getConversionRulesTask().getRules();
			if (rules.getDatastreamTemplateCount() == 0) {
				String msg = I18N.translate("MessageMissingDatastreamTemplate");
				String newDT = JOptionPane.showInputDialog(creator, msg);
				if (newDT == null)
					return;
				rules.addDatastreamTemplate(new ConversionRules.DatastreamTemplate(newDT));
				creator.getConversionRulesTask().updateRules();
			}
			newMetadata.setType(rules.getDatastreamTemplate(0).getNodeType());

			for (int ctr = 0; ctr < entry.getMetadataCount(); ctr++) {
				if (entry.getMetadata(ctr).getType().equals(newMetadata.getType())) {
					String msg = I18N.translate("MessageTypeAlreadyExisting1") + newMetadata.getType() + I18N.translate("MessageTypeAlreadyExisting2");
					JOptionPane.showMessageDialog(creator, msg);
					break;
				}
			}

			entry.addMetadata(newMetadata);
		}

	}

	public class CloseCurrentTabAction extends AbstractAction {

		private static final long serialVersionUID = -1317113261942287869L;

		private CloseCurrentTabAction() {
			putValue(Action.NAME, I18N.translate("CloseTab"));
			URL imgURL = creator.getURL(CLOSE_IMAGE_NAME);
			putValue(Action.SMALL_ICON, new ImageIcon(creator.getImage(imgURL)));
			putValue(Action.SHORT_DESCRIPTION, I18N.translate("CloseTabTT"));
		}

		public void actionPerformed(ActionEvent ae) {
			closeCurrentTab();
		}

		public void closeCurrentTab() {
			int index = metadataView.getSelectedIndex();
			if (index < 0)
				return;

			Component c = metadataView.getComponentAt(index);
			if (c instanceof SelectableEntryPanel) {
				((SelectableEntryPanel) c).updateMetadata();
			} else if (c instanceof MetadataPanelWrapper) {
				((MetadataPanelWrapper) c).updateMetadata();
			}
			metadataView.remove(index);
		}

	}

	public class FilterAction extends AbstractAction implements DocumentListener {

		public FilterAction() {
			putValue(Action.NAME, I18N.translate("ActivateFilter"));
			putValue(Action.SHORT_DESCRIPTION, I18N.translate("ActivateFilterTT"));
		}

		public void actionPerformed(ActionEvent ae) {
			String text = filterField.getText();
			text = text.replaceAll("\\.", "\\\\.");
			text = text.replaceAll("\\*", ".*");
			if (text.length() == 0) {
				text = ".*";
			}

			filterAcceptor.setFilter(text);
			filterAcceptor.setEnabled(filterEnabledBox.isSelected());

			refreshTree();
		}

		public void changedUpdate(DocumentEvent e) {
			if (!filterEnabledBox.isSelected()) {
				return;
			}
			actionPerformed(null);
		}

		public void insertUpdate(DocumentEvent e) {
			if (!filterEnabledBox.isSelected()) {
				return;
			}
			actionPerformed(null);
		}

		public void removeUpdate(DocumentEvent e) {
			if (!filterEnabledBox.isSelected()) {
				return;
			}
			actionPerformed(null);
		}

	}

}
