/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;

import ch.docuteam.tools.file.FileChecksumCalculator.Algorithm;
import ch.docuteam.tools.out.Logger;
import edu.harvard.hul.ois.jhove.Module;
import edu.harvard.hul.ois.jhove.RepInfo;
import uk.gov.nationalarchives.droid.core.interfaces.IdentificationResult;

/**
 * @author denis
 *
 */
public class FileWithMetadata extends File
{

	private static final String					Unknown = "(n/a)";

	private Algorithm							DefaultChecksumAlgorithm = Algorithm.MD5;

	private IdentificationResult				DROIDData = null;
	private boolean								alreadyTriedToRetrieveDROID = false;

	private RepInfo								jhoveData = null;
	private boolean								alreadyTriedToRetrieveJhove = false;


	/**
	 * @param pathname
	 */
	public FileWithMetadata(String pathname)
	{
		super(pathname);
	}

	/**
	 * @param uri
	 */
	public FileWithMetadata(URI uri)
	{
		super(uri);
	}

	/**
	 * @param parent
	 * @param child
	 */
	public FileWithMetadata(String parent, String child)
	{
		super(parent, child);
	}

	/**
	 * @param parent
	 * @param child
	 */
	public FileWithMetadata(File parent, String child)
	{
		super(parent, child);
	}


	public Algorithm getDefaultChecksumAlgorithm()
	{
		return DefaultChecksumAlgorithm;
	}

	public void setDefaultChecksumAlgorithm(Algorithm algorithm)
	{
		DefaultChecksumAlgorithm = algorithm;
	}

	public String getFormatName()
	{
		//	Note: formatName from droid takes precedence over formatName from jhove:
		String formatNameDroid = getFileFormatName_droid();
		return (Unknown.equals(formatNameDroid)? getFormat_jhove(): formatNameDroid);
	}

	public String getFormatName(Provider provider)
	{
		if (provider == Provider.DROID)				return getFileFormatName_droid();
		else if (provider == Provider.JHOVE)		return getFormat_jhove();

		return Unknown;
	}

	public String getFormatPronomID()
	{
		return getFileFormatPUID_droid();
	}

	public String getMimeType()
	{
		//	Note: mimeType from droid takes precedence over mimeType from jhove:
		String mimeTypeDroid = getMimeType_droid();
		return (Unknown.equals(mimeTypeDroid)? getMimeType_jhove(): mimeTypeDroid);
	}

	public String getMimeType(Provider provider)
	{
		if (provider == Provider.DROID)				return getMimeType_droid();
		else if (provider == Provider.JHOVE)		return getMimeType_jhove();

		return Unknown;
	}

	public int getWellFormed()
	{
		return getWellFormed_jhove();
	}

	public int getValid()
	{
		return getValid_jhove();
	}

	public boolean isConsistent()
	{
		return isConsistent_jhove();
	}

	public long getSize()
	{
		return getSize_jhove();
	}

	public Module getModule()
	{
		return getModule_jhove();
	}

	public List<String> getProfile()
	{
		return getProfile_jhove();
	}

	public String getChecksum() throws NoSuchAlgorithmException, IOException
	{
		return DefaultChecksumAlgorithm.getDigest(getAbsolutePath());
	}

	public String getChecksum(Algorithm algorithm) throws NoSuchAlgorithmException, IOException
	{
		if (algorithm == null)					return null;
		
		return Algorithm.valueOf(algorithm.toString()).getDigest(getAbsolutePath());
	}


	/**
	 * @return The property jHoveInfo, if it is already initialized. If not, initialize the property jHoveInfo and return it.
	 * It can however happen that it can't be initialized; in this case, return null.
	 */
	private RepInfo getJhoveMetadata()
	{
		Logger.getLogger().debug("Getting jhove Metadata...");

		if (jhoveData == null)		initializeJhoveMetadata();
		return jhoveData;
	}


	private void initializeJhoveMetadata()
	{
		//	If I already tried to get Jhove, don't try again:
		if (alreadyTriedToRetrieveJhove)
		{
			Logger.getLogger().debug("NOT retrying to retrieve jhove Metadata for file:" + getPath());
			return;
		}

		String filePath;
		try
		{
			filePath = getCanonicalPath();
		}
		catch (IOException e)
		{
			Logger.getLogger().warn("Could not get canonical path for file", e);
			return;
		}

		Logger.getLogger().debug("Retrieving jhove Metadata for file:" + filePath + "...");

		jhoveData = MetadataProviderJHOVE.getRepInfo(filePath);		//	May be null in case of an error
		if (jhoveData == null)		Logger.getLogger().debug("...NOK!");
		else						Logger.getLogger().debug("...OK");

		alreadyTriedToRetrieveJhove = true;
	}


	/**
	 * @return The property jHoveInfo, if it is already initialized. If not, initialize the property jHoveInfo and return it.
	 * It can however happen that it can't be initialized; in this case, return null.
	 */
	private IdentificationResult getDroidMetadata()
	{
		Logger.getLogger().debug("Getting droid Metadata...");

		if (DROIDData == null)		initializeDroidMetadata();
		return DROIDData;
	}


	private void initializeDroidMetadata()
	{
		//	If I already tried to retrieve DROID, don't try again:
		if (alreadyTriedToRetrieveDROID)
		{
			Logger.getLogger().debug("NOT retrying to retrieve droid Metadata for file:" + getPath());
			return;
		}

		String filePath;
		try
		{
			filePath = getCanonicalPath();
		}
		catch (IOException e)
		{
			Logger.getLogger().warn("Could not get canonical path for file", e);
			return;
		}

		Logger.getLogger().debug("Retrieving droid Metadata for file:" + filePath + "...");

		try
		{
			DROIDData = MetadataProviderDROID.getIdentificationResult(filePath);

			Logger.getLogger().debug("...OK");
		}
		catch (Exception ex)
		{
			Logger.getLogger().debug("...NOK!");
		}

		alreadyTriedToRetrieveDROID = true;
	}



	private int getWellFormed_jhove()
	{
		if (getJhoveMetadata() == null)		return -1;

		return jhoveData.getWellFormed();
	}


	private int getValid_jhove()
	{
		if (getJhoveMetadata() == null)		return -1;

		return jhoveData.getValid();
	}


	private boolean isConsistent_jhove()
	{
		if (getJhoveMetadata() == null)		return false;

		return jhoveData.isConsistent();
	}


	private String getMimeType_jhove()
	{
		if (getJhoveMetadata() == null)		return Unknown;

		return jhoveData.getMimeType();
	}


	private String getFormat_jhove()
	{
		if (getJhoveMetadata() == null)		return Unknown;

		return jhoveData.getFormat();
	}


	private long getSize_jhove()
	{
		if (getJhoveMetadata() == null)		return -1;

		return jhoveData.getSize();
	}


	private Module getModule_jhove()
	{
		if (getJhoveMetadata() == null)		return null;

		return jhoveData.getModule();
	}


	private List<String> getProfile_jhove()
	{
		if (getJhoveMetadata() == null)		return new ArrayList<String>();

		return jhoveData.getProfile();
	}


	private String getMimeType_droid()
	{
		if (getDroidMetadata() == null)		return Unknown;

		return DROIDData.getMimeType();
	}


	private String getFileFormatName_droid()
	{
		if (getDroidMetadata() == null)		return Unknown;

		return DROIDData.getName();
	}


	private String getFileFormatPUID_droid()
	{
		if (getDroidMetadata() == null)		return Unknown;

		return DROIDData.getPuid();
	}


	public static enum Provider
	{
		DROID, JHOVE;
	}

}
