/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;

import ch.docuteam.tools.file.exception.DROIDCouldNotInitializeException;
import ch.docuteam.tools.file.exception.DROIDMultipleIdentificationsFoundException;
import ch.docuteam.tools.file.exception.DROIDNoIdentificationFoundException;
import ch.docuteam.tools.out.Logger;
import uk.gov.nationalarchives.droid.command.action.CommandExecutionException;
import uk.gov.nationalarchives.droid.command.container.Ole2ContainerContentIdentifier;
import uk.gov.nationalarchives.droid.command.container.ZipContainerContentIdentifier;
import uk.gov.nationalarchives.droid.container.ContainerFileIdentificationRequestFactory;
import uk.gov.nationalarchives.droid.container.ContainerSignatureDefinitions;
import uk.gov.nationalarchives.droid.container.ContainerSignatureSaxParser;
import uk.gov.nationalarchives.droid.container.TriggerPuid;
import uk.gov.nationalarchives.droid.container.ole2.Ole2IdentifierEngine;
import uk.gov.nationalarchives.droid.container.zip.ZipIdentifierEngine;
import uk.gov.nationalarchives.droid.core.BinarySignatureIdentifier_Extended;
import uk.gov.nationalarchives.droid.core.interfaces.IdentificationRequest;
import uk.gov.nationalarchives.droid.core.interfaces.IdentificationResult;
import uk.gov.nationalarchives.droid.core.interfaces.IdentificationResultCollection;
import uk.gov.nationalarchives.droid.core.interfaces.RequestIdentifier;
import uk.gov.nationalarchives.droid.core.interfaces.archive.IdentificationRequestFactory;
import uk.gov.nationalarchives.droid.core.interfaces.resource.FileSystemIdentificationRequest;
import uk.gov.nationalarchives.droid.core.interfaces.resource.RequestMetaData;
import uk.gov.nationalarchives.droid.core.signature.FileFormat;

/**
 * This is an abstract class for getting file format information using DROID.
 * <br>
 * The DROID subsystem requires the signature file
 * "config/DROID_SignatureFile_V66.xml" to be in the folder "config" in the
 * working directory.
 *
 * @author denis
 *
 */
public abstract class MetadataProviderDROID {

	private static final String DEFAULT_SIGNATURE_FILE = "config/DROID_SignatureFile_V84.xml";
	private static final String DEFAULT_CONTAINER_SIGNATURE_FILE = "config/container-signature-20160121.xml";

	private static final int DEFAULT_EXTENSION_USAGE = 1;

	private static String signatureFile = DEFAULT_SIGNATURE_FILE;
	private static String containerSignatureFile = DEFAULT_CONTAINER_SIGNATURE_FILE;

	private static int extensionUsage = DEFAULT_EXTENSION_USAGE;

	private static BinarySignatureIdentifier_Extended signatureIdentificator = null;
	private static ContainerSignatureDefinitions containerSignatureDefs = null;
	private static List<TriggerPuid> containerSignatureTriggerPuids = null;

	private static Boolean isInitialized = false;

	public static void setSignatureFile(String newSignatureFile) {
		signatureFile = newSignatureFile;

		isInitialized = false;
		// I will (re-)initialize myself the next time I am used.
	}

	public static void setContainerSignatureFile(String newContainerSignatureFile) {
		containerSignatureFile = newContainerSignatureFile;

		isInitialized = false;
		// I will (re-)initialize myself the next time I am used.
	}

	/*
	 * set the usage of file extension matching: 1 = use tentative extension
	 * matching, i.e. only rely on extensions that do not have a defined
	 * signature 2 = use full extension matching, i.e. use extension even for
	 * file formats that have defined signatures anything else = disable
	 * extension matching altogether
	 */
	public static void setExtensionUsage(int levelOfExtensionUsage) {
		extensionUsage = levelOfExtensionUsage;
	}

	public static IdentificationResult getIdentificationResult(String filePath) throws DROIDCouldNotInitializeException,
			DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException {
		List<IdentificationResult> resultList = getIdentificationResults(filePath);

		if (resultList == null || resultList.isEmpty())
			throw new DROIDNoIdentificationFoundException(filePath);
		if (resultList.size() != 1)
			throw new DROIDMultipleIdentificationsFoundException(filePath, resultList);

		return resultList.get(0);
	}

	// The following are convenience methods (shortcuts for retrieving specific
	// metadata directly):

	public static String getFileFormatPUID(String fileName) throws DROIDCouldNotInitializeException,
			DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException {
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null) ? null : result.getPuid();
	}

	public static String getMimeType(String fileName) throws DROIDCouldNotInitializeException,
			DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException {
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null) ? null : result.getMimeType();
	}

	public static String getFileFormatName(String fileName) throws DROIDCouldNotInitializeException,
			DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException {
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null) ? null : result.getName();
	}

	public static String getFileFormatVersion(String fileName) throws DROIDCouldNotInitializeException,
			DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException {
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null) ? null : result.getVersion();
	}

	public static String getFileFormatMethod(String fileName) throws DROIDCouldNotInitializeException,
			DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileNotFoundException {
		IdentificationResult result = getIdentificationResult(fileName);
		return (result == null) ? null : result.getMethod().getMethod();
	}

	private static void initializeIfNecessary() throws DROIDCouldNotInitializeException {
		if (isInitialized) {
			return;
		}

		try {
			Logger.getLogger().debug("Initializing DROID...");

			if (!new File(signatureFile).exists()) {
				throw new FileNotFoundException(signatureFile);
			}
			if (!new File(containerSignatureFile).exists()) {
				throw new FileNotFoundException(containerSignatureFile);
			}

			signatureIdentificator = new BinarySignatureIdentifier_Extended();
			signatureIdentificator.setSignatureFile(signatureFile);
			signatureIdentificator.init();

			containerSignatureDefs = new ContainerSignatureSaxParser()
					.parse(new FileInputStream(containerSignatureFile));
			containerSignatureTriggerPuids = containerSignatureDefs.getTiggerPuids();

			isInitialized = true;

			Logger.getLogger().debug("...OK");
		} catch (java.lang.Exception ex) {
			Logger.getLogger().debug("...NOK!");

			throw new DROIDCouldNotInitializeException(ex);
		}
	}

	private static List<IdentificationResult> getIdentificationResults(String filePath)
			throws DROIDCouldNotInitializeException, FileNotFoundException {
		initializeIfNecessary();

		if (!new File(filePath).exists()) {
			throw new FileNotFoundException(filePath);
		}

		File file = new File(filePath);
		RequestMetaData metadata = new RequestMetaData(file.length(), file.lastModified(), file.getName());
		RequestIdentifier id = new RequestIdentifier(file.toURI());
		IdentificationRequest request = new FileSystemIdentificationRequest(metadata, id);

		FileInputStream fis = null;
		try {
			fis = new FileInputStream(file);
			request.open(fis);

			// Identify the file. Try 3 different systems: first container, then
			// binary (= signature), and finally file extension:
			// (NOTE: To get the container identifications, I need the binary
			// identifications)
			IdentificationResultCollection signatureResultCollection = signatureIdentificator
					.matchBinarySignatures(request);
			IdentificationResultCollection containerResultCollection = getContainerResults(request,
					signatureResultCollection);

			IdentificationResultCollection finalResultCollection;
			if (containerResultCollection.getResults().size() > 0) {
				finalResultCollection = containerResultCollection;
			} else if (signatureResultCollection.getResults().size() > 0) {
				finalResultCollection = signatureResultCollection;
			} else {
				switch (extensionUsage) {
				case 1:
					finalResultCollection = signatureIdentificator.matchExtensions(request, false);
					break;
				case 2:
					finalResultCollection = signatureIdentificator.matchExtensions(request, true);
					break;
				default:
					finalResultCollection = containerResultCollection;
				}
			}

			signatureIdentificator.removeLowerPriorityHits(finalResultCollection);

			return finalResultCollection.getResults();
		} catch (Exception e) {
			Logger.getLogger().warn("Exception occured while trying to identify file: " + filePath, e);
			return null;
		} finally {
			try {
				request.close();
				if (fis != null)
					fis.close();
			} catch (IOException ex) {
				// TODO why is there nothing here ?
			}
		}
	}

	private static IdentificationResultCollection getContainerResults(IdentificationRequest request,
			IdentificationResultCollection results) throws CommandExecutionException {
		IdentificationResultCollection containerResults = new IdentificationResultCollection(request);

		for (IdentificationResult identResult : results.getResults()) {
			String filePuid = identResult.getPuid();
			if (filePuid == null) {
				continue;
			}

			TriggerPuid containerPuid = null;
			for (TriggerPuid tp : containerSignatureTriggerPuids) {
				if (tp.getPuid().equals(filePuid)) {
					containerPuid = tp;
					break;
				}
			}
			if (containerPuid == null) {
				continue;
			}

			IdentificationRequestFactory requestFactory = new ContainerFileIdentificationRequestFactory();
			String containerType = containerPuid.getContainerType();

			if ("OLE2".equals(containerType)) {
				try {
					Ole2ContainerContentIdentifier ole2Identifier = new Ole2ContainerContentIdentifier();
					ole2Identifier.init(containerSignatureDefs, containerType);
					Ole2IdentifierEngine ole2IdentifierEngine = new Ole2IdentifierEngine();
					ole2IdentifierEngine.setRequestFactory(requestFactory);
					ole2Identifier.setIdentifierEngine(ole2IdentifierEngine);
					ole2Identifier.process(request.getSourceInputStream(), containerResults);
				} catch (IOException e) {
					Logger.getLogger().warn("Couldn't identify OLE2 container in more detail", e);
				}
			} else if ("ZIP".equals(containerType)) {
				try {
					ZipContainerContentIdentifier zipIdentifier = new ZipContainerContentIdentifier();
					zipIdentifier.init(containerSignatureDefs, containerType);
					ZipIdentifierEngine zipIdentifierEngine = new ZipIdentifierEngine();
					zipIdentifierEngine.setRequestFactory(requestFactory);
					zipIdentifier.setIdentifierEngine(zipIdentifierEngine);
					zipIdentifier.process(request.getSourceInputStream(), containerResults);
				} catch (IOException e) {
					Logger.getLogger().warn("Couldn't identify ZIP container in more detail", e);
				}
			} else {
				throw new CommandExecutionException("Unknown container type: " + containerPuid);
			}
		}

		IdentificationResultCollection finalContainerResults = new IdentificationResultCollection(request);
		for (IdentificationResult r : containerResults.getResults()) {
			FileFormat ff = signatureIdentificator.getFileFormatForPuid(r.getPuid());
			finalContainerResults.addResult(new MetadataProviderDROID_IdentificationResult(r, ff));
		}

		return finalContainerResults;
	}

}
