/**
 *	Copyright (C) 2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file.ssh;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import com.jcraft.jsch.Channel;
import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import com.jcraft.jsch.SftpException;

import ch.docuteam.tools.out.Logger;

/**
 * Copies a file to a remote host via SFTP. Don't forget to call the {@link #closeConnection() channelSftpExit} method and {@link SSHSession#disconnectSession()
 * SSHSessionProvider.disconnectSession()} on the SSHSessionProvider object after finishing sending files.
 * 
 * @author iliya
 *
 */
public class SFTPChannel {

	protected String user;
	protected String host;
	protected int port;
	protected String privateKeyFile;
	protected String knownHostsFile;

	private SSHSession sshSession;
	private ChannelSftp channelSftp;

	public SFTPChannel(String user, String host, int port, String privateKeyFile, String knownHostsFile) {
		this.user = user;
		this.host = host;
		this.port = port;
		this.privateKeyFile = privateKeyFile;
		this.knownHostsFile = knownHostsFile;
	}

	/**
	 * Copies a file to the location that comes with the {@link SSHSession} given by the initialization of this class.
	 * 
	 * @throws SftpException
	 * @throws JSchException
	 * @throws IOException
	 */
	public void transferFileLeaveSessionOpen(String sourceFile, String targetFile, String targetDir) throws SftpException, JSchException, IOException {
		File f = new File(sourceFile);
		
		//	make sure we have an active channel
		getChannelSftp();

		if (targetDir.startsWith("/")) {
			Logger.debug("change directory to root");
			channelSftp.cd("/");
			targetDir = targetDir.substring(1);
		}

		try (InputStream inputStream = new FileInputStream(f)) {
			// initialize the channel and go to the target dir on the remote end
			for (String dir : targetDir.split("/")) {
				try {
					Logger.debug("change directory to '" + dir + "'");
					channelSftp.cd(dir);
				} catch (SftpException e) {
					Logger.debug("make and change to directory '" + dir + "'");
					channelSftp.mkdir(dir);
					channelSftp.cd(dir);
				}
			}
			Logger.debug("transferring to '" + targetFile + "'");
			channelSftp.put(inputStream, targetFile);
			Logger.getLogger().info("File transfered successfully using SFTP.");
		}
	}

	/**
	 * Copies a file to the location that comes with the {@link SSHSession} and closes the {@link SFTPChannel} and {@link SSHSession}.
	 * 
	 * @throws SftpException
	 * @throws JSchException
	 * @throws IOException
	 */
	public void transferFileCloseSession(String sourceFile, String targetFile, String targetDir) throws SftpException, JSchException, IOException {
		this.transferFileLeaveSessionOpen(sourceFile, targetFile, targetDir);
		closeConnection();
	}

	/**
	 * Buffered reader to read a file on a remote host, don't forget to close the reader after use.
	 * 
	 * @param targetFile
	 * @param targetDir
	 * @return
	 * @throws FileNotFoundException
	 * @throws SftpException
	 * @throws JSchException
	 */
	public BufferedReader readFile(String targetFile, String targetDir) throws FileNotFoundException, SftpException, JSchException {
		InputStream is = null;
		InputStreamReader isr = null;

		getChannelSftp().cd(targetDir);
		is = getChannelSftp().get(targetFile);
		isr = new InputStreamReader(is);
		BufferedReader br = new BufferedReader(isr);
		return br;
	}

	public void removeFile(String targetFile, String targetDir) throws FileNotFoundException, SftpException, JSchException {
		getChannelSftp().cd(targetDir);
		getChannelSftp().rm(targetFile);
	}

	/**
	 * Open and connect the SFTP channel
	 * 
	 * @return
	 * @throws FileNotFoundException
	 * @throws JSchException
	 * @throws SftpException 
	 */
	public ChannelSftp getChannelSftp() throws FileNotFoundException, JSchException, SftpException {
		if (channelSftp == null) {
			Session session = getSSHConnection();
			Channel channel = session.openChannel("sftp");
			channelSftp = (ChannelSftp) channel;
			Logger.getLogger().info("SFTP channel opened");
		}
		if (!channelSftp.isConnected()) {
			channelSftp.connect();
			Logger.getLogger().debug("SFTP channel connected");
		}
		Logger.debug("SFTP working directory: '" + channelSftp.pwd() + "'");
		return channelSftp;
	}

	/**
	 * Initialize the underlying secure shell connection
	 * @return
	 * @throws FileNotFoundException
	 * @throws JSchException
	 */
	public Session getSSHConnection() throws FileNotFoundException, JSchException {
		if (sshSession == null) {
			sshSession = new SSHSession(user, host, port, privateKeyFile, knownHostsFile);
		}
		return sshSession.connect();
	}

	/**
	 * Closes the opened SFTP channel and SSH Session.
	 */
	public void closeConnection() {
		if (channelSftp != null) {
			channelSftp.disconnect();
			Logger.getLogger().debug("SFTP channel disconnected");
		}
		if (sshSession != null) {
			sshSession.disconnect();
			Logger.getLogger().debug("SSH session disconnected");
		}
	}

}
