/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.id;

import java.io.File;
import java.io.IOException;

import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.out.Logger;

/**
 * This abstract class returns unique Integers, in ascending order.
 *
 * @author denis
 *
 */
public abstract class NextNumber
{

	private static final String LockFilePostfix = ".lock";

	private static final String DefaultFilePath = "./config/NextNumber";
	private static final String DefaultStartValue = "0";

	/**
	 * The file path. It can be changed using initialize():
	 */
	private static String				FilePath = DefaultFilePath;

	/**
	 * The initial content of the file. The start value can be changed using initialize():
	 */
	private static String				StartValue = DefaultStartValue;


	public static void initialize(String filePath)
	{
		initialize(filePath, 0l);
	}

	public static void initialize(String filePath, Long startValue)
	{
		FilePath = filePath;
		StartValue = startValue.toString();
	}


	public static Long get() throws Exception
	{
		Logger.getLogger().debug("Getting...");

		initializeIfNecessary();

		if (!waitUntilUnlocked())		throw new Exception("Timeout!");

		Long num = null;
		try
		{
			lock();
			num = new Long(FileUtil.getFileContentAsString(FilePath).trim());		//	trim() is to remove any possible trailing /n
			increment(num);
		}
		finally
		{
			unlock();
		}

		Logger.getLogger().info("Got: " + num);

		return num;
	}


	public static String getString() throws Exception
	{
		return get().toString();
	}


	private static void initializeIfNecessary() throws IOException
	{
		if (!new File(FilePath).exists())
		{
			Logger.getLogger().debug("Initalizing: " + FilePath);

			FileUtil.createFileWithContent(FilePath, StartValue);
		}
	}


	private static void increment(Long num) throws IOException
	{
		Logger.getLogger().debug("Incrementing: " + num);

		FileUtil.createFileWithContent(FilePath, (++num).toString());

		Logger.getLogger().debug("Incremented: " + num);
	}


	private static boolean waitUntilUnlocked()
	{
		Logger.getLogger().debug("Locked?");

		File lockFile = new File(FilePath + LockFilePostfix);

		boolean unlocked = false;
		for (int i = 20; i > 0; i--)
		{
			if (!lockFile.exists())
			{
				unlocked = true;
				break;
			}

			Logger.getLogger().debug("Retrying: " + i);

			//	Wait 1/2 sec and then retry:
			try { Thread.sleep(500); } catch (InterruptedException x){}
		}

		return unlocked;
	}


	private static void lock() throws IOException
	{
		Logger.getLogger().debug("Locking...");

		FileUtil.createFileWithContent(FilePath + LockFilePostfix, null);
	}


	private static void unlock() throws FileUtilExceptionListException
	{
		Logger.getLogger().debug("Unlocking...");

		FileUtil.delete(FilePath + LockFilePostfix);
	}


}
