/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.os;

import java.io.File;
import java.util.Map;

import ch.docuteam.tools.out.Logger;

public abstract class EnvironmentVariable
{

	private static final String HOME_DIR = ".";

	private static final String EnvVarStartString = "${";
	private static final String EnvVarEndString = "}";

	/**
	 * Returns an unmodifiable string map view of the current system environment.
	 * The environment is a system-dependent mapping from names to
	 * values which is passed from parent to child processes.
	 *
	 * <p>If the system does not support environment variables, an
	 * empty map is returned.
	 */
	public static Map<String, String> getAll()
	{
		return System.getenv();
	}


	/**
	 * Return the value of the environment variable 'name', or null if undefined.
	 */
	public static String get(String name)
	{
		return get(name, null);
	}


	/**
	 * Return the value of the environment variable 'name', or the 'ifUndefined' String if undefined.
	 */
	public static String get(String name, String ifUndefined)
	{
		String value = System.getenv(name);
		if (value == null)
		{
			if (ifUndefined == null){
				Logger.getLogger().warn("Undefined Environment Variable: " + name);
			}
			value = ifUndefined;
		}

		Logger.getLogger().debug("Retrieving Environment Variable: '" + name + "' with default: '" + ifUndefined + "' returning: '" + value + "'");

		return value;
	}


	/**
	 * Return the value of the environment variable 'name', or the home path "." if undefined.
	 */
	public static String getAsFolder(String name)
	{
		return get(name, HOME_DIR);
	}

	/**
	 * Return the file on the respective path or null if file does not exist.
	 * 
	 * @param environmentVariableName
	 * @param relativeFileName
	 * @return the file or null if file does not exist
	 */
	public static File getFileTryCurrentContextFirst(String environmentVariableName, String relativeFileName)
	{
		File file = new File("." + relativeFileName);
		if (file.isFile()) return file;

		file = new File(getAsFolder(environmentVariableName) + relativeFileName);
		return file.isFile() ? file : null;
	}

	/**
	 * The input string may contain one or more environment variables (embedded between '${' and '}'). If this is the case, resolve them.
	 * @param string
	 * @return the string itself with possible environment variables replaced by their content. The content may be empty.
	 */
	public static String resolveIn(String string)
	{
		if ((string == null) || !(string.contains(EnvVarStartString) && string.contains(EnvVarEndString))) {
			return string;
		}
		
		int startIndex = 0;
		do
		{
			int envVarStartIndex = string.indexOf(EnvVarStartString, startIndex);
			if (envVarStartIndex == -1)		break;
			int envVarEndIndex = string.indexOf(EnvVarEndString, envVarStartIndex);
			if (envVarEndIndex == -1)		break;

			String envVarName = string.substring(envVarStartIndex + EnvVarStartString.length(), envVarEndIndex);
			String envVarContent = get(envVarName);
			if (envVarContent == null)		envVarContent = "";
			string = string.replace(string.substring(envVarStartIndex, envVarEndIndex + EnvVarEndString.length()), envVarContent);

			startIndex = envVarStartIndex + envVarContent.length();
		}
		while (true);

		return string;
	}


}
