/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.translations;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

/**
 * @author denis
 *
 */
public abstract class I18N
{

	public static final String[]			AvailableLanguages = new String[] {"de", "en", "fr"};


	private static List<Translator>			Translators = new ArrayList<Translator>();


	public static void initialize(String language, String translationResourceBasePath)
	{
		Translators.add(new Translator(language, translationResourceBasePath));
	}

	public static void clear()
	{
		Translators.clear();
	}


	/**
	 * Return the translated word, or if something goes wrong, the word itself.
	 * Assume that the class has already been initialized.
	 */
	public static String translate(String word)
	{
		if (Translators.isEmpty())		throw new RuntimeException("I18N was not initialized");

		return doTranslate(word);
	}


	/**
	 * Return the translated word, or if something goes wrong, the word itself.
	 * Don't throw an exception when the class has not yet been initialized, but just return the word itself.
	 */
	public static String translate_NoCheck(String word)
	{
		if (Translators.isEmpty())		return word;

		return doTranslate(word);
	}



	/**
	 * Return the translated word, or if something goes wrong, the word itself. Format any placeholder in the string using the object list.
	 * Assume that the class has already been initialized.
	 */
	public static String translate(String word, Object... objects)
	{
		if (Translators.isEmpty())		throw new RuntimeException("I18N was not initialized");

		return doTranslate(word, objects);
	}


	/**
	 * Return the translated word, or if something goes wrong, the word itself. Format any placeholder in the string using the object list.
	 * Don't throw an exception when the class has not yet been initialized, but just return the word itself.
	 */
	public static String translate_NoCheck(String word, Object... objects)
	{
		if (Translators.isEmpty())		return word;

		return doTranslate(word, objects);
	}


	private static String doTranslate(String word)
	{
		String value = null;
		for (Translator translator : Translators)
		{
			value = translator.translate(word);
			if (!value.equals(word))
			{
				return value;
			}
		}
		return value;
	}
	
	private static String doTranslate(String word, Object...objects)
	{
		String value = null;
		for (Translator translator : Translators)
		{
			value = translator.translate(word, objects);
			if (!value.equals(word))
			{
				return value;
			}
		}
		return value;
	}
	

	/**
	 * Instances of this class actually do the translation.
	 */
	public static class Translator
	{
		protected ResourceBundle				resources = null;


		public Translator(String language, String translationResourceBasePath)
		{
			this.resources = ResourceBundle.getBundle(translationResourceBasePath, new Locale(language));
		}


		/**
		 * Return the translated word, or if something goes wrong, the word itself.
		 */
		public String translate(String word)
		{
			if (this.resources == null)		throw new RuntimeException("This translator was not initialized correctly");

			try
			{
				return this.resources.getString(word);
			}
			catch(Exception x)
			{
				return word;
			}
		}

		/**
		 * Return the translated word, or if something goes wrong, the word itself. Format any placeholder in the string using the object list.
		 */
		public String translate(String word, Object... objects)
		{
			try
			{
				return String.format(this.translate(word), objects);
			}
			catch(Exception x)
			{
				return word;
			}
		}
	}

}
