/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.io.File;
import java.io.Serializable;
import java.util.*;

import javax.swing.ImageIcon;

import org.dom4j.*;
import org.dom4j.io.SAXReader;

import ch.docuteam.darc.ead.C;
import ch.docuteam.darc.exceptions.MetadataElementIsNotAllowedException;
import ch.docuteam.darc.exceptions.MetadataElementIsNotDefinedException;
import ch.docuteam.tools.out.Logger;


/**
 * This class represents a level of description in a METS file.
 * The levels are initialized through the file "./config/levels.xml".
 * <p>
 * A level has a name, an order number, and optionally an icon.
 * <p>
 * A level has a list of levels that are allowed as its sub-levels, and one level that is the default sub-level.
 * <p>
 * A level has a list of <a href="./LevelMetadataElement.html">LevelMetadataElements</a>
 * (actually two lists: a Map for quick access and a List&lt;LevelOfDescription&gt; for displaying them in a GUI in their correct order).
 *
 * @author denis
 *
 */
public class LevelOfDescription implements Serializable
{
	//	The default file for initializing levels.
	private static final String	DEFAULT_LEVELS_FILE_NAME = "./config/levels.xml";

	//	The values separator. Default is ';', it can be changed using the method setAllowedValuesSeparator().
	private static String DefaultAllowedValuesSeparator = ";";

	/**
	 * The currently used file for initializing levels.
	 */
	private static String InitializationFilePath = DEFAULT_LEVELS_FILE_NAME;

	/**
	 * The values separator. Default is ';', it can be changed using the method setAllowedValuesSeparator().
	 */
	private static String AllowedValuesSeparator = DefaultAllowedValuesSeparator;

	/**
	 * A Map containing all LevelOfDescriptions. Access key is the level name.
	 */
	private static Map<String, LevelOfDescription> All;

	/**
	 * A List containing all LevelOfDescriptions in their correct order.
	 */
	private static List<LevelOfDescription> AllOrdered;

	// TODO This is rather just the default level than that it necessarily has
	// to be the "undefined" level.
	/**
	 * The undefined Level = the first level being initialized.
	 */
	static private LevelOfDescription Undefined;

	/**
	 * Level used to indicate that the file is the master (preservation) copy of
	 * the record
	 */
	static private LevelOfDescription masterFileLevel = null;

	/**
	 * Level used to indicate that the file is the master (preservation) copy of
	 * the record
	 */
	static private LevelOfDescription derivedFileLevel = null;

	private String name;
	private Integer order;

	/** The icon used for this Level in a GUI (might be null). */
	private ImageIcon icon;

	/** Set containing this level's allowed sublevel names (might be empty). */
	private Set<String> allowedSubLevelNames = new HashSet<String>();

	/** The default level name when inserting a new subelement (might be null). */
	private String defaultSubLevelName;

	/**
	 * Map containing this level's LevelMetadataElements. Key is the
	 * LevelMetadataElement's accessorName, value is the LevelMetadataElement
	 * itself.
	 */
	private Map<String, LevelMetadataElement> levelMetadataElements = new HashMap<String, LevelMetadataElement>();

	/**
	 * Vector containing this level's LevelMetadataElements in their correct order.
	 */
	private List<LevelMetadataElement> levelMetadataElementsOrdered = new Vector<LevelMetadataElement>();

	/**
	 * Constructor used by LevelOfDescriptionSet, to generate Levels that don't belong to the standard set of Levels specified in levels.xml.
	 * @param name
	 */
	LevelOfDescription(String name, int order)
	{
		this.name = name;
		this.order = order;

		Logger.getLogger().debug("Created: " + this);
	}

	/**
	 * Private Constructor to generate Levels that belong to the standard set of Levels specified in levels.xml.
	 * @param name
	 */
	private LevelOfDescription(String name, String iconFileName, String allowedSubLevelNames)
	{
		this.name = name;
		this.order = All.size();

		All.put(name, this);
		AllOrdered.add(this);

		//	The first level created is the Undefined one:
		if (this.order == 0)		Undefined = this;

		if (iconFileName != null && new File(iconFileName).exists())
			this.icon = new ImageIcon(iconFileName);

		if (allowedSubLevelNames != null && !allowedSubLevelNames.trim().isEmpty())
		{
			String[] allAllowedSubLevelNames = allowedSubLevelNames.split("\\s+");
			this.defaultSubLevelName = allAllowedSubLevelNames[0];
			for (String subLevelName: allAllowedSubLevelNames)		this.allowedSubLevelNames.add(subLevelName);
		}

		Logger.getLogger().debug("Created standard Level: " + this);
	}


	static public void setInitializationFilePath(String initializationFilePath)
	{
		InitializationFilePath = initializationFilePath;
		All = null;
		AllOrdered = null;
		MetadataElement.clear();
		AllowedValuesSeparator = DefaultAllowedValuesSeparator;
		//	LevelsOfDescription and MetadataElements will be (re-)initialized the next time they are accessed.
	}

	/**
	 * @return the path to the currently used levels configuration file
	 */
	public static String getInitializationFilePath() {
		return InitializationFilePath;
	}

	static public void setAllowedValuesSeparator(String sep)
	{
		AllowedValuesSeparator = sep;
	}

	static public void setMasterFileLevel(LevelOfDescription master)
	{
		masterFileLevel = master;
	}

	static public void setDerivedFileLevel(LevelOfDescription derived)
	{
		derivedFileLevel = derived;
	}

	//	--------		Accessing			-------------------------------------------------------

	static public LevelOfDescription getUndefined()
	{
		initializeIfNecessary();
		return Undefined;
	}

	static public String getAllowedValuesSeparator()
	{
		return AllowedValuesSeparator;
	}

	static public LevelOfDescription getMasterFileLevel()
	{
		return masterFileLevel;
	}

	static public LevelOfDescription getDerivedFileLevel()
	{
		return derivedFileLevel;
	}

	/**
	 * IMPORTANT NOTE: during initialization of the levels, references to non-existing metadata elements, and bad validators
	 * are found, but no bad default expressions. These are found only when they are used. That makes sense because
	 * using a default expression requires a Node (ch.docuteam.darc.mets.structmap.NodeAbstract) to apply this default expression to;
	 * but during initialization of the levels, there is no Node around.
	 */
	private static void initialize()
	{
		Logger.getLogger().debug("Initializing LevelOfDescriptions from file: " + InitializationFilePath);

		MetadataElement.clear();

		All = new HashMap<String, LevelOfDescription>();
		AllOrdered = new Vector<LevelOfDescription>();

		org.dom4j.Document levelsConfigFile;
		try
		{
			levelsConfigFile = new SAXReader().read(new File(InitializationFilePath));

			Element rootElement = levelsConfigFile.getRootElement();

			Logger.getLogger().debug("Reading optional AllowedValuesSeparator:");
			Node n = rootElement.selectSingleNode("./LEVELS:MetadataElements/LEVELS:AllowedValuesSeparator");
			if (n != null)		setAllowedValuesSeparator(n.getText());

//			ExceptionCollector.clear();		//	ToDo: Nested Exception Collectors?

			Logger.getLogger().debug("Reading MetadataElements:");
			for (Object e: rootElement.selectNodes("./LEVELS:MetadataElements/LEVELS:MetadataElement"))
			{
				Element mde = (Element)e;
				try {
					new MetadataElement(C.class, mde.attributeValue("accessorNameID"),
							mde.attributeValue("validatorClassName"), mde.attributeValue("postActionClassName"),
							mde.attributeValue("defaultExpression"), mde.attributeValue("allowedValues"));
				}
				catch (Exception ex)
				{
					ch.docuteam.tools.exception.Exception.remember(ex);
				}
			}
			Logger.getLogger().debug("Reading MetadataElements finished.");

			Logger.getLogger().debug("Reading Levels:");
			for (Object l: rootElement.selectNodes("./LEVELS:Levels/LEVELS:Level"))
			{
				Element le = (Element)l;
				LevelOfDescription level = new LevelOfDescription(le.attributeValue("nameID"), le.attributeValue("iconFileName"), le.attributeValue("allowedSublevelNameRefs"));

				for (Object o: le.selectNodes("./LEVELS:LevelMetadataElement"))
				{
					Element lme = (Element)o;
					LevelMetadataElement levelMetadataElement;
					try
					{
						levelMetadataElement = new LevelMetadataElement(level, lme.attributeValue("accessorNameRef"), lme.attributeValue("isMandatory"), lme.attributeValue("isAlwaysDisplayed"), lme.attributeValue("isRepeatable"), lme.attributeValue("isReadOnly"), lme.attributeValue("keepInTemplate"), lme.attributeValue("displayRows"));
					}
					catch (MetadataElementIsNotDefinedException ex)
					{
						ch.docuteam.tools.exception.Exception.remember(ex);
						continue;		//	Ignore bad Metadata Elements
					}
					level.levelMetadataElements.put(levelMetadataElement.getId(), levelMetadataElement);
					level.levelMetadataElementsOrdered.add(levelMetadataElement);
				}
			}
			Logger.getLogger().debug("Reading Levels finished.");

//			if (!ExceptionCollector.isEmpty())		throw new ExceptionCollectorException();		//	ToDo: Nested Exception Collectors?
		}
		catch (DocumentException ex)
		{
			ex.printStackTrace();
		}
//		catch (ExceptionCollectorException ex)
//		{
//			ex.printStackTrace();
//		}
	}


	/**
	 * Return the Levels list in the order the levels were inserted.
	 * Package visibility because this method should be used only by LevelOfDescriptionSet.
	 */
	static List<LevelOfDescription> getAll()
	{
		initializeIfNecessary();
		return AllOrdered;
	}


	/**
	 * This method has package visibility because it can be called by MetadataElement.
	 */
	static void initializeIfNecessary()
	{
		if (All == null){
			initialize();
		}
	}


	public String getName()
	{
		return this.name;
	}

	public Integer getOrder()
	{
		return this.order;
	}

	/**
	 * The ImageIcon returned may be null
	 * @return
	 */
	public ImageIcon getIcon()
	{
		return this.icon;
	}


	public Set<String> getAllowedSubLevelNames()
	{
		return this.allowedSubLevelNames;
	}

	public LevelOfDescription getDefaultSubLevel()
	{
		return (this.defaultSubLevelName != null)? All.get(this.defaultSubLevelName): getUndefined();
	}

	public Map<String, LevelMetadataElement> getDynamicMetadataElements()
	{
		return this.levelMetadataElements;
	}


	public LevelMetadataElement getDynamicMetadataElement(String accessorName)
			throws MetadataElementIsNotAllowedException, MetadataElementIsNotDefinedException {
		if (!this.levelMetadataElements.containsKey(accessorName)) {
			if (MetadataElement.get(accessorName) == null) {
				throw new MetadataElementIsNotDefinedException(this, accessorName);
			} else {
				throw new MetadataElementIsNotAllowedException(this, accessorName);
			}
		}

		return this.levelMetadataElements.get(accessorName);
	}

	public List<LevelMetadataElement> getDynamicMetadataElementsOrdered()
	{
		return this.levelMetadataElementsOrdered;
	}

	public boolean allowsSubLevel(String name)
	{
		return this.allowedSubLevelNames.contains(name);
	}

	public boolean allowsSubLevel(LevelOfDescription level)
	{
		return this.allowedSubLevelNames.contains(level.name);
	}

	public boolean isUndefined()
	{
		return this == Undefined;
	}

	@Override
	public String toString()
	{
		return this.order + ":" + this.name;
	}

}

