/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.text.*;
import java.util.regex.Pattern;

import ch.docuteam.darc.exceptions.MetadataElementValidatorException;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.tools.translations.I18N;

/**
 * A MetadataElementValidator that only accepts a string in the standard date
 * format "dd.MM.yyyy".
 *
 * @author denis
 *
 */
public class MetadataElementValidatorDateCH implements MetadataElementValidator {
	
	static private final DateFormat[] StandardDateFormat = { new SimpleDateFormat("yyyy"),
			new SimpleDateFormat("M.yyyy"), new SimpleDateFormat("d.M.yyyy") };
	
	static private final Pattern[] StandardDatePattern = { Pattern.compile("\\d{4}"),
			Pattern.compile("\\d{1,2}\\.\\d{4}"), Pattern.compile("\\d{1,2}\\.\\d{1,2}\\.\\d{4}") };

	/*
	 * (non-Javadoc)
	 * 
	 * @see ch.docuteam.darc.ead.DataElementValidator#check(java.lang.String,
	 * java.lang.Object)
	 */
	@Override
	public void check(String value, NodeAbstract context, MetadataElement metadataElement)
			throws MetadataElementValidatorException {
		if (value == null || value.isEmpty())
			return;

		// Check for the locale's value of "no date"
		if (value.equals(I18N.translate_NoCheck("NoDate")))
			return;

		// Check pattern without semantic first to identify bad inputs like
		// "24.12.2010xxxx" which are actually parsed correctly by DateFormat
		for (int i = 0; i < StandardDatePattern.length; i++) {
			if (StandardDatePattern[i].matcher(value).matches()) {
				try {
					DateFormat df = StandardDateFormat[i];
					df.setLenient(false);
					df.parse(value);
					return;
				} catch (ParseException ex) {
					throw new MetadataElementValidatorException(
							I18N.translate_NoCheck("MessageValidatorDateCHCantConvert", value));
				}
			}
		}

		// If the code reaches this point, no pattern was matching
		throw new MetadataElementValidatorException(I18N.translate_NoCheck("MessageValidatorDateCHCantConvert", value));

	}

}
