/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Pattern;

import ch.docuteam.darc.exceptions.MetadataElementValidatorException;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.tools.translations.I18N;

/**
 * A MetadataElementValidator that only accepts a string in the standard date
 * format "yyyyMMdd".
 *
 * @author iliya
 *
 */
public class MetadataElementValidatorDateYYYYMMDD implements MetadataElementValidator {
	// takes 4 digits for year, 00 to 12 for month and 00 to 31 for day
	private String dateRegex = "^\\d{4}(1[0-2]|0[0-9])([0-2]\\d{1}|3[0-1])$";
	// similar to above but doesn't parse dates with 00 month and some day
	// different than 00 (20150012 is not allowed)
	private String dateNotKnownRegex = "^\\d{4}((1[0-2]|0[0-9])00)";
	private String dateFormat = "yyyyMMdd";

	@Override
	public void check(String value, NodeAbstract context, MetadataElement metadataElement)
			throws MetadataElementValidatorException {
		Date date = null;
		Pattern p = Pattern.compile(dateRegex);
		if (p.matcher(value).matches()) {
			try {
				SimpleDateFormat sdf = new SimpleDateFormat(dateFormat);
				date = sdf.parse(value);
				// this is needed to check for date inputs like 20160231
				if (!value.equals(sdf.format(date))) {
					// here the case with the 00 for day or day and month is
					// regarded
					Pattern unfullDatePattern = Pattern.compile(dateNotKnownRegex);
					if (!unfullDatePattern.matcher(value).matches()) {
						throw new MetadataElementValidatorException(
								I18N.translate_NoCheck("MessageValidatorDateYYYYMMDDCantConvert", value));
					}
				}
				// return true;
			} catch (ParseException ex) {
				throw new MetadataElementValidatorException(
						I18N.translate_NoCheck("MessageValidatorDateYYYYMMDDCantConvert", value));
			}
		} else {
			throw new MetadataElementValidatorException(
					I18N.translate_NoCheck("MessageValidatorDateYYYYMMDDCantConvert", value));
		}

	}

}
