package ch.docuteam.converter;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.net.SocketException;

import com.pdftools.converter.Converter;
import com.pdftools.converter.Job;

import ch.docuteam.converter.exceptions.PDFToolsAppendDocException;
import ch.docuteam.converter.exceptions.PDFToolsCantCloseJobException;
import ch.docuteam.converter.exceptions.PDFToolsCantCreateOutputException;
import ch.docuteam.converter.exceptions.PDFToolsConverterBadVMArgumentException;
import ch.docuteam.tools.out.Logger;

/**
 * This class is a wrapper around the PDFTools Converter. NOTE: PDFTools
 * Converter works only under Windows, NOT under OSX or Linux! NOTE: PDFTools
 * requires the following preconditions to run correctly: 1: The dll files
 * "O2PProxyAPI.dll" and "O2PProxyNET.dll" (in the correct 32Bit/64Bit version
 * for the client computer) have to be present. 2: The dll "O2PProxyNET.dll" has
 * to be registered in the "Global Assembly Cache" . Refer to:
 * <http://msdn.microsoft.com/en-us/library/ex0ss12c%28v=vs.110%29.aspx>. 3: The
 * Java application has to be started with the VM Parameter:
 * -Djava.library.path=[dllPath], where [dllPath] is the relative or absulute
 * path to the folder containing O2PProxyAPI.dll (without trailing slash). 4: If
 * not initialized explicitly with an URL pointing to the (remote) PDFTools
 * service, assume a local installation. This class is used by the FileConverter
 * reflectively or directly via the main() method, hence it seems to the
 * compiler as if it is unused. The only method used from the outside is static
 * public void main().
 */
public class PDFToolsConverter {

	static private Converter ConverterInstance;
	static private Job JobInstance;

	/**
	 * This main method is the gateway to run the converter. It's NOT just for
	 * tests! Don't delete it!
	 * 
	 * @throws SocketException
	 * @throws PDFToolsConverterBadVMArgumentException
	 * @throws PDFToolsCantCloseJobException
	 * @throws PDFToolsAppendDocException
	 * @throws PDFToolsCantCreateOutputException
	 */
	static public void main(String... args) throws SocketException, PDFToolsConverterBadVMArgumentException,
			PDFToolsCantCreateOutputException, PDFToolsAppendDocException, PDFToolsCantCloseJobException {
		if (args.length < 2 && args.length > 4) {
			System.err.println("ERROR: Wrong number of arguments.");
			System.err.println("");
			System.err.println(
					"Usage: PDFToolsConverter [path/to/]sourceFile [path/to/]destinationFile [document options [job options]]");
			return;
		}

		if (args.length == 2) {
			convert2PDF(args[0], args[1]);
		} else if (args.length == 3) {
			convert2PDF(args[0], args[1], args[2]);
		} else if (args.length == 4) {
			convert2PDF(args[0], args[1], args[2], args[3]);
		}
	}

	/**
	 * Initialize local PDF-Tools converter.
	 * 
	 * @throws SocketException
	 * @throws PDFToolsConverterBadVMArgumentException
	 */
	static public void initialize() throws SocketException, PDFToolsConverterBadVMArgumentException {
		initialize(null);
	}

	/**
	 * Initialize remote PDF-Tools converter.
	 * 
	 * @param url
	 *            The service URL for connecting to the document converter
	 *            service. When null is specified, the default
	 *            "tcp://localhost:7981/O2PService" is used.
	 * @throws SocketException
	 * @throws PDFToolsConverterBadVMArgumentException
	 */
	static public void initialize(String url) throws SocketException, PDFToolsConverterBadVMArgumentException {
		checkRequiredVMArgument();

		ConverterInstance = Converter.createConverter(url);
		JobInstance = ConverterInstance.createJob();
	}

	/**
	 * Disconnect PDF-Tools converter and release all resources.
	 */
	static public void disconnect() {
		if (JobInstance != null)
			JobInstance.destroyObject();
		if (ConverterInstance != null)
			ConverterInstance.destroyObject();
	}

	static public void convert2PDF(String sourceFilePath, String destinationFilePath)
			throws SocketException, PDFToolsConverterBadVMArgumentException, PDFToolsCantCreateOutputException,
			PDFToolsAppendDocException, PDFToolsCantCloseJobException {
		convert2PDF(sourceFilePath, destinationFilePath, "");
	}

	static public void convert2PDF(String sourceFilePath, String destinationFilePath, String documentOptions)
			throws SocketException, PDFToolsConverterBadVMArgumentException, PDFToolsCantCreateOutputException,
			PDFToolsAppendDocException, PDFToolsCantCloseJobException {
		convert2PDF(sourceFilePath, destinationFilePath, documentOptions, "");

	}

	public static void convert2PDF(String sourceFilePath, String destinationFilePath, String documentOptions,
			String jobOptions) throws SocketException, PDFToolsConverterBadVMArgumentException,
			PDFToolsCantCreateOutputException, PDFToolsAppendDocException, PDFToolsCantCloseJobException {
		initializeIfNecessary();
		if (JobInstance == null) {
			Logger.getLogger().error("PDF-Tools Converter can not be initialized");
			return;
		}

		Logger.getLogger().info("Converting: '" + sourceFilePath + "' to: '" + destinationFilePath + "'");

		boolean ok = false;
		try {
			ok = JobInstance.createOutput(destinationFilePath);
			if (!ok) {
				String message = "Returned error Code: " + JobInstance.getLastError().Code + ", error text: "
						+ JobInstance.getLastError().Text + ", when attempted to create file: " + destinationFilePath;
				throw new PDFToolsCantCreateOutputException(message);
			}

			JobInstance.setOptions(jobOptions);

			ok = JobInstance.appendDoc(sourceFilePath, documentOptions);
			if (!ok) {
				String message = "Append Doc for file: " + sourceFilePath + " failed with code: "
						+ JobInstance.getLastError().Code + " and with text: " + JobInstance.getLastError().Text;
				throw new PDFToolsAppendDocException(message);
			}

		} finally {
			// need this separate handling of return value to make sure we're not overwriting an Exception from the try block
			boolean okClose = JobInstance.close();
			if (!okClose && ok) {
				String message = "Can't close pdf tools job instance, code: " + JobInstance.getLastError().Code
						+ " text: " + JobInstance.getLastError().Text;
				throw new PDFToolsCantCloseJobException(message);
			}
		}
	}

	static private void initializeIfNecessary() throws SocketException, PDFToolsConverterBadVMArgumentException {
		if (JobInstance == null)
			initialize();
	}

	static private void checkRequiredVMArgument() throws PDFToolsConverterBadVMArgumentException {
		String javaLibraryPath = null;
		for (String vmArg : ManagementFactory.getRuntimeMXBean().getInputArguments()) {
			if (vmArg.startsWith("-Djava.library.path")) {
				javaLibraryPath = vmArg.substring(vmArg.lastIndexOf("=") + 1).trim();
				break;
			}
		}

		if (javaLibraryPath == null)
			throw new PDFToolsConverterBadVMArgumentException(
					"The Java VM Argument '-Djava.library.path' is not defined. It has to point to the folder containing the file 'O2PProxyAPI.dll'");

		if (!new File(javaLibraryPath + "/O2PProxyAPI.dll").exists())
			throw new PDFToolsConverterBadVMArgumentException(
					"The Java VM Argument '-Djava.library.path' doesn't point to the folder containing the file 'O2PProxyAPI.dll'");
	}

}
