/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.converter;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;

import ch.docuteam.converter.exceptions.BadPronomIdException;
import ch.docuteam.converter.exceptions.FileConversionException;
import ch.docuteam.tools.file.FileUtil;
import ch.docuteam.tools.file.FileWithMetadata;
import ch.docuteam.tools.file.MetadataProviderDROID;
import ch.docuteam.tools.file.exception.DROIDCouldNotInitializeException;
import ch.docuteam.tools.file.exception.DROIDMultipleIdentificationsFoundException;
import ch.docuteam.tools.file.exception.DROIDNoIdentificationFoundException;
import ch.docuteam.tools.file.exception.FileIsNotADirectoryException;
import ch.docuteam.tools.file.exception.FileUtilExceptionListException;
import ch.docuteam.tools.id.UniqueID;
import ch.docuteam.tools.os.OperatingSystem;
import ch.docuteam.tools.os.SystemProcess;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.DateFormatter;

/**
 * @author denis
 *
 */
public abstract class FileConverter {

	static private final String DefaultMigrationConfigFile = "config/migration-config.xml";
	static private final String DefaultMigrationHomeFolder = ".";
	static private String DefaultCommandlineSeparator = "#";
	static private String[] AlternativeCommandlineSeparators = new String[] { ";", "&", "@", ",", ":" };

	static private String MigrationConfigFile = DefaultMigrationConfigFile;
	static private String MigrationHomeFolder = DefaultMigrationHomeFolder;

	static private org.dom4j.Document MigrationConfigFileDocument;

	static private String RecentlyUsedFileConverterName = "";

	static public File convertFile(File file)
			throws DocumentException, IllegalArgumentException, SecurityException, IOException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException,
			IndexOutOfBoundsException, FileIsNotADirectoryException, BadPronomIdException,
			FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException,
			DROIDMultipleIdentificationsFoundException, FileConversionException {
		return convertFile(file.getPath());
	}

	static public File convertFile(File sourceFile, File destinationFolder)
			throws DocumentException, IllegalArgumentException, SecurityException, IOException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException,
			IndexOutOfBoundsException, FileIsNotADirectoryException, BadPronomIdException,
			FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException,
			DROIDMultipleIdentificationsFoundException, FileConversionException {
		return convertFile(sourceFile.getPath(), destinationFolder.getPath());
	}

	static public File convertFile(String fileName)
			throws DocumentException, IllegalArgumentException, SecurityException, IOException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException,
			IndexOutOfBoundsException, FileIsNotADirectoryException, BadPronomIdException,
			FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException,
			DROIDMultipleIdentificationsFoundException, FileConversionException {
		return convertFile(fileName, null);
	}

	static public File convertFile(String filePath, String destinationFolderPath)
			throws DocumentException, IllegalArgumentException, SecurityException, IOException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException,
			IndexOutOfBoundsException, FileIsNotADirectoryException, BadPronomIdException,
			FileUtilExceptionListException, DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException,
			DROIDMultipleIdentificationsFoundException, FileConversionException {
		// Whatever comes in, make the fileNames absolute:
		filePath = new File(filePath).getAbsolutePath();
		if (destinationFolderPath != null)
			destinationFolderPath = new File(destinationFolderPath).getAbsolutePath();

		FileWithMetadata file = new FileWithMetadata(filePath);
		if (!file.exists()) {
			Logger.getLogger().error("File not found: '" + filePath + "'");
			throw new FileNotFoundException(filePath);
		}

		Logger.getLogger().info("Converting file: " + filePath);

		RecentlyUsedFileConverterName = "";

		List<?> migrationInstructions = getMigrationInstructions(file);
		if (migrationInstructions.isEmpty()) {
			Logger.getLogger().info("... no conversion instruction found.");
			return null;
		}
		if (migrationInstructions.size() >= 2) {
			Logger.getLogger().error("Can't handle multiple conversion instructions for file '" + file.getName() + "'");
			// TODO Create own exception instead of using a rather generic
			// runtime exception which might occur for other reasons as well
			throw new IndexOutOfBoundsException(
					"Can't handle multiple conversion instructions for file '" + file.getName() + "'");
		}

		Element migrationInstruction = (Element) migrationInstructions.get(0);
		List<Element> migrationSteps = new ArrayList<Element>();
		List<?> steps = migrationInstruction.selectNodes("step");
		if (steps.isEmpty()) {
			// If there is no attribute applicationId, don't convert:
			if (migrationInstruction.attribute("applicationID") == null)
				return null;
		} else {
			for (Object step : steps) {
				if (step instanceof Element && ((Element) step).attribute("applicationID") != null)
					migrationSteps.add((Element) step);
			}
			if (migrationSteps.isEmpty())
				return null;
		}

		// Note: file.getFormatPronomID() NEVER returns null,
		// migrationInstruction.attributeValue("targetPronom") MIGHT return
		// null:
		if (destinationFolderPath != null)
			FileUtil.createFolderMerging(destinationFolderPath);

		Logger.getLogger().info("Treating file as " + migrationInstruction.attributeValue("name") + "...");

		// This is for the case that the file paths are too long under Windows:
		boolean usingTempFolder = false;
		File tempFolder = null;
		String originalFilePath = null;
		String originalDestinationFolderPath = null;
		File convertedFile = null;
		try {
			if (OperatingSystem.isWindows() && filePath.length() >= 255
					|| (destinationFolderPath != null && destinationFolderPath.length() >= 255)) {
				// Paths are too long - use temporary folder for conversion:
				usingTempFolder = true;
				originalFilePath = filePath;
				originalDestinationFolderPath = destinationFolderPath;

				tempFolder = new File(FileUtil.getTempFolder() + "/FileConverter" + File.separator
						+ Long.valueOf(Thread.currentThread().getId()).toString());
				tempFolder.mkdirs();
				filePath = tempFolder.getAbsolutePath() + File.separator + "Temp_" + UniqueID.getString() + "."
						+ FileUtil.asFileNameExtension(filePath);
				FileUtil.copyToOverwriting(originalFilePath, filePath);
				file = new FileWithMetadata(filePath);
				destinationFolderPath = FileUtil.getTempFolder() + "/FileConverter";

				Logger.getLogger().debug("Too long paths, using temp file: " + filePath);
			}

			convertedFile = (migrationSteps.isEmpty()) ? convertFile(file, destinationFolderPath, migrationInstruction)
					: convertFile(file, destinationFolderPath, migrationSteps);
		} finally {
			if (usingTempFolder) {
				if (convertedFile != null) {
					// Move converted file to original destination:
					String destinationFilePath = originalDestinationFolderPath + "/"
							+ FileUtil.asFileNameWithoutExtension(originalFilePath) + "."
							+ FileUtil.asFileNameExtension(convertedFile.getName());
					// The returned file is of course the file in the original
					// destination folder:
					convertedFile = FileUtil.moveTo(convertedFile.getPath(), destinationFilePath);
				}

				// Cleanup temporary files:
				FileUtil.deleteOnExit(tempFolder);
			}
		}

		return convertedFile;
	}

	public static String getRecentlyUsedFileConverterName() {
		return RecentlyUsedFileConverterName;
	}

	public static void setMigrationConfigFile(String newMigrationConfigFile) throws DocumentException {
		MigrationConfigFile = newMigrationConfigFile;
		initialize();
	}

	public static void setMigrationHomeFolder(String newMigrationHomeFolder) {
		MigrationHomeFolder = newMigrationHomeFolder;
	}

	static private File convertFile(FileWithMetadata file, String destinationFolderName, Element migrationInstruction)
			throws IOException, IllegalArgumentException, SecurityException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException,
			FileIsNotADirectoryException, BadPronomIdException, DROIDCouldNotInitializeException,
			DROIDNoIdentificationFoundException, DROIDMultipleIdentificationsFoundException, FileConversionException {
		if (destinationFolderName == null)
			destinationFolderName = file.getParentFile().getAbsolutePath();

		// this is the filename that the method will return as the result
		String destinationPath = calculateDestinationFilePath(file, destinationFolderName,
				migrationInstruction.attributeValue("targetExtension"));
		Logger.getLogger().info("Destination file: " + destinationPath);

		// this is the intermediary filename for the actual conversion that will
		// not pose any problem with special characters that the original
		// filename may contain
		String conversionTargetPath = destinationFolderName + File.separator + UniqueID.getString();
		if (migrationInstruction.attributeValue("targetExtension").length() > 0) {
			conversionTargetPath += "." + migrationInstruction.attributeValue("targetExtension");
		}
		Logger.getLogger().info("Conversion target file: " + conversionTargetPath);

		String[] commandLine = constructCommandLine(migrationInstruction,
				new String[] { file.getAbsolutePath(), conversionTargetPath });
		executeCommandLine(commandLine);

		FileUtil.renameTo(conversionTargetPath, destinationPath);

		String expectedPronomIds = migrationInstruction.attributeValue("targetPronom");
		if (expectedPronomIds != null && !expectedPronomIds.isEmpty()) {
			// collect output files
			Path destination = Paths.get(destinationPath);
			final List<Path> files = new ArrayList<>();
			Files.walkFileTree(destination, new SimpleFileVisitor<Path>() {
				@Override
				// the check for the PUID is not done in visitFile, because
				// visitFile throws IOException and getFileFormatPUID throws
				// incompatible exceptions
				public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
					if (attrs.isRegularFile()) {
						files.add(file);
					}
					return FileVisitResult.CONTINUE;
				}
			});
			
			boolean isValid = true;
			List<String> expectedPronomIdsList = Arrays.asList(expectedPronomIds.split(","));
			String wrongPronomId = null;
			// loop through files and check their formats
			for (Path filePath : files) {
				String currentPronomId = MetadataProviderDROID.getFileFormatPUID(filePath.toString());
				if (!expectedPronomIdsList.contains(currentPronomId)) {
					wrongPronomId = currentPronomId;
					isValid = false;
					break;
				}
			}
			if (!isValid) {
				throw new BadPronomIdException(expectedPronomIds, wrongPronomId);
			}
		}

		return new File(destinationPath);
	}

	static private File convertFile(FileWithMetadata file, String destinationFolderName, List<?> migrationInstructions)
			throws IOException, IllegalArgumentException, SecurityException, IllegalAccessException,
			InvocationTargetException, NoSuchMethodException, ClassNotFoundException, InterruptedException,
			FileIsNotADirectoryException, BadPronomIdException, FileUtilExceptionListException,
			DROIDCouldNotInitializeException, DROIDNoIdentificationFoundException,
			DROIDMultipleIdentificationsFoundException, FileConversionException {
		List<String> tempFilePathsToDelete = new ArrayList<String>();
		String sourcePathString = file.getAbsolutePath();
		String destinationPathString = "";
		int stepNumber = migrationInstructions.size();

		try {
			// Loop thru the steps:
			int stepCounter = 1;
			for (Object i : migrationInstructions) {
				Element migrationInstruction = (Element) i;

				// Note: file.getFormatPronomID() NEVER returns null,
				// migrationInstruction.attributeValue("targetPronom") MIGHT
				// return null:
				String[] expectedPronomIds = null;
				String expected = migrationInstruction.attributeValue("targetPronom");

				if (destinationFolderName == null)
					destinationFolderName = file.getParentFile().getAbsolutePath();

				// this is the filename that the method will return as the
				// result
				destinationPathString = calculateDestinationFilePath(file, destinationFolderName,
						migrationInstruction.attributeValue("targetExtension"));
				Logger.getLogger().info(
						"Step " + stepCounter + "/" + stepNumber + ": Destination file: " + destinationPathString);

				// Remember this intermediary file for later deletion (excluding
				// the last step):
				if (stepCounter != stepNumber)
					tempFilePathsToDelete.add(destinationPathString);

				// this is the intermediary filename for the actual conversion
				// that will not pose any problem with special characters that
				// the original filename may contain
				String conversionTargetPath = destinationFolderName + File.separator + UniqueID.getString() + "."
						+ migrationInstruction.attributeValue("targetExtension");
				Logger.getLogger().info("Conversion target file: " + conversionTargetPath);

				// Perform the conversion:
				String[] commandLine = constructCommandLine(migrationInstruction,
						new String[] { sourcePathString, conversionTargetPath });
				executeCommandLine(commandLine);

				FileUtil.renameTo(conversionTargetPath, destinationPathString);

				String currentPronomId = MetadataProviderDROID.getFileFormatPUID(destinationPathString);
				expectedPronomIds = null;
				if (expected != null && !expected.isEmpty()) {
					expectedPronomIds = expected.split(",");
					boolean found = false;
					for (String expectedPronomId : expectedPronomIds) {
						if (currentPronomId.equalsIgnoreCase(expectedPronomId)) {
							found = true;
						}
					}
					if (!found) {
						throw new BadPronomIdException(expectedPronomIds.toString(), currentPronomId);
					}
				}

				// Set the input of the next step to be the output of this step:
				sourcePathString = destinationPathString;
				++stepCounter;
			}
		} finally {
			// Delete all intermediary files:
			for (String path : tempFilePathsToDelete)
				FileUtil.delete(path);
		}

		return new File(destinationPathString);
	}

	static private void initializeIfNecessary() throws DocumentException {
		if (MigrationConfigFileDocument == null)
			initialize();
	}

	static private void initialize() throws DocumentException {
		Logger.getLogger().info("Initializing File Migration, from " + MigrationConfigFile + "...");

		// Parse MigrationConfig-File:
		MigrationConfigFileDocument = new SAXReader().read(new File(MigrationConfigFile));
	}

	static public String calculateDestinationFilePath(File file, String destinationFolderName, String newExtension) {
		String destinationFilePath = destinationFolderName + "/" + FileUtil.asFileNameWithoutExtension(file.getName());
		String suffix = "";
		if (newExtension.length() > 0) {
			suffix = "." + newExtension;
		}

		// Check if the destination file already exists. If yes, append the
		// current timestamp to the filename:
		if (new File(destinationFilePath + suffix).exists())
			destinationFilePath += "_" + DateFormatter.getCurrentDateTimeString(DateFormatter.NumericalMSecs);
		destinationFilePath += suffix;

		return destinationFilePath;
	}

	static public List<?> getMigrationInstructions(FileWithMetadata file) throws DocumentException {
		initializeIfNecessary();

		Logger.getLogger().info("Looking for PUID " + file.getFormatPronomID() + "...");
		List<?> instructions = MigrationConfigFileDocument
				.selectNodes("//puid[@name='" + file.getFormatPronomID() + "']");
		if (!instructions.isEmpty())
			return instructions;

		Logger.getLogger().info("Looking for mimetype " + file.getMimeType() + "...");
		instructions = MigrationConfigFileDocument.selectNodes("//mimeType[@name='" + file.getMimeType() + "']");
		if (!instructions.isEmpty())
			return instructions;

		String fileNameExtension = file.getName().substring(file.getName().lastIndexOf(".") + 1);
		Logger.getLogger().info("Looking for extension " + fileNameExtension + "...");
		return MigrationConfigFileDocument.selectNodes("//extension[@name='" + fileNameExtension + "']");
	}

	static private String[] constructCommandLine(Element migrationInstruction, String[] args) {
		Element applicationElement = (Element) MigrationConfigFileDocument
				.selectSingleNode("//application[@id='" + migrationInstruction.attributeValue("applicationID") + "']");
		String commandLine = applicationElement.attributeValue("executable");
		RecentlyUsedFileConverterName = applicationElement.attributeValue("name");

		String applicationParameter = applicationElement.attributeValue("parameter");
		String instructionParameter = migrationInstruction.attributeValue("parameter");

		// instructionParameter take precedence over applicationParameter:
		if (instructionParameter != null && !instructionParameter.trim().isEmpty())
			commandLine += DefaultCommandlineSeparator + instructionParameter;
		else if (applicationParameter != null && !applicationParameter.trim().isEmpty())
			commandLine += DefaultCommandlineSeparator + applicationParameter;

		// Check for unwanted occurences of separator in commandline and replace
		String separator = determineApplicableSeparator(commandLine, args[0]);
		if (!separator.equals(DefaultCommandlineSeparator)) {
			commandLine = commandLine.replace(DefaultCommandlineSeparator, separator);
		}

		int argIndex = 0;
		while (true) {
			String argPlaceholderString = "{[arg" + (argIndex + 1) + "]}";
			if (commandLine.indexOf(argPlaceholderString) == -1)
				break;

			commandLine = commandLine.replace(argPlaceholderString, args[argIndex]);
			argIndex = argIndex + 1;
		}

		return commandLine.trim().split(separator);
	}

	static private String determineApplicableSeparator(String parameters, String filepath) {
		if (!filepath.contains(DefaultCommandlineSeparator)) {
			return DefaultCommandlineSeparator;
		}
		for (String separator : AlternativeCommandlineSeparators) {
			if (!filepath.contains(separator) && !parameters.contains(separator)) {
				return separator;
			}
		}
		return DefaultCommandlineSeparator;
	}

	/**
	 * @throws ClassNotFoundException
	 * @throws SecurityException
	 * @throws NoSuchMethodException
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 *             The call to the java class converters goes via reflection,
	 *             that's why the respective exceptions are not directly visible
	 *             and should be handled by the calling application.
	 * @throws FileIsNotADirectoryException
	 * @throws InterruptedException
	 * @throws IOException
	 * @throws FileNotFoundException
	 * @throws FileConversionException
	 */
	static private void executeCommandLine(String[] commandLine)
			throws IllegalAccessException, IllegalArgumentException, InvocationTargetException, NoSuchMethodException,
			SecurityException, ClassNotFoundException, FileNotFoundException, IOException, InterruptedException,
			FileIsNotADirectoryException, FileConversionException {
		if (commandLine[0].startsWith("Class:")) {
			// Cut away leading "Class:" from class name:
			String className = commandLine[0].substring(6);
			Logger.getLogger().debug("Calling java class: " + className);

			// Cut away 1st element of commandLine (= class name):
			commandLine = Arrays.copyOfRange(commandLine, 1, commandLine.length);

			Logger.getLogger().debug("Using the following parameters: " + Arrays.toString(commandLine));
			Object result = Class.forName(className).getMethod("main", commandLine.getClass()).invoke(null,
					new Object[] { commandLine });
			Logger.getLogger().info("Conversion-process finished" + ((result != null) ? ": " + result : ""));
		} else {
			Logger.getLogger().info("Executing command line: " + Arrays.toString(commandLine));

			int errorCode = SystemProcess.execute(MigrationHomeFolder, commandLine);
			Logger.getLogger().info("Conversion-process finished: " + errorCode);

			if (errorCode != 0)
				throw new FileConversionException(errorCode);
		}
	}

	/**
	 * This class is a dummy conversion class that simply creates a copy of a
	 * file. This class is used by the FileConverter reflectively, hence it
	 * seems to the compiler as if it is unused. The only method used from the
	 * outside is static public void main().
	 */
	@SuppressWarnings("unused")
	static private class Copy {
		/**
		 * This main method is the gateway to run the converter. It's NOT just
		 * for tests! Don't delete it!
		 */
		static public void main(String... args) throws IOException, FileUtilExceptionListException {
			if (args.length != 2) {
				System.err.println("ERROR: Wrong number of arguments.");
				System.err.println("");
				System.err.println("Usage: FileConverter.Copy [path/to/]sourceFile [path/to/]destinationFile");
				return;
			}

			FileUtil.copyToOverwriting(args[0], args[1]);
		}
	}

}
