/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.common;

import java.util.Observer;

import org.dom4j.tree.DefaultDocument;

import ch.docuteam.darc.mdconfig.LevelOfDescriptionSet;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.out.MessageDistributor;

/**
 *
 * DocumentAbstract is the abstract superclass of a METS or EAD Document.
 *
 * @author denis
 *
 */
public abstract class DocumentAbstract extends DefaultDocument {
	/**
	 * The file path to this METS.xml document. It is always absolute.
	 */
	protected String filePath;

	/**
	 * From Docudarc version 2.11.2 onwards, all Documents have their own set of
	 * LevelOfDescription:
	 */
	protected LevelOfDescriptionSet levels = new LevelOfDescriptionSet();

	/**
	 * true if any modifications were applied to this document, false otherwise.
	 */
	protected boolean isModified = false;

	/**
	 * true if at least one file has the 'canRead' attribute set to false (=
	 * current user has no read access on that file).
	 */
	protected boolean isAtLeastOneFileNotReadable = false;

	/**
	 * true if at least one file has the 'canWrite' attribute set to false (=
	 * current user has no write access on that file).
	 */
	protected boolean isAtLeastOneFileNotWritable = false;

	/**
	 * This is that every document can distribute messages to whoever is
	 * interested in.
	 */
	// "false" means: don't send message to System.out
	protected MessageDistributor observable = new MessageDistributor(false);

	public String getFilePath() {
		return this.filePath;
	}

	public void setFilePath(String newFilePath) {
		this.filePath = newFilePath;
	}

	public LevelOfDescriptionSet getLevels() {
		return this.levels;
	}

	public boolean isModified() {
		return this.isModified;
	}

	public void setIsModified() {
		this.isModified = true;

		StackTraceElement caller = Thread.currentThread().getStackTrace()[2];
		Logger.getLogger().debug("Modified by: " + caller.getClassName() + "." + caller.getMethodName());
	}

	public void setIsNotModified() {
		this.isModified = false;
	}

	public boolean isAtLeastOneFileNotReadable() {
		return this.isAtLeastOneFileNotReadable;
	}

	public void setIsAtLeastOneFileNotReadable() {
		this.setIsAtLeastOneFileNotReadable(true);
	}

	public void setIsAtLeastOneFileNotReadable(boolean isAtLeastOneFileNotReadable) {
		this.isAtLeastOneFileNotReadable = isAtLeastOneFileNotReadable;
	}

	public boolean isAtLeastOneFileNotWritable() {
		return this.isAtLeastOneFileNotWritable;
	}

	public void setIsAtLeastOneFileNotWritable() {
		this.setIsAtLeastOneFileNotWritable(true);
	}

	public void setIsAtLeastOneFileNotWritable(boolean isAtLeastOneFileNotWritable) {
		this.isAtLeastOneFileNotWritable = isAtLeastOneFileNotWritable;
	}

	/**
	 * Return the type of this document: METS, EAD, ...
	 */
	public abstract Type getDocumentType();

	public void distributeMessage(String message) {
		this.observable.distribute(message);
	}

	public void addObserver(Observer observer) {
		if (observer != null)
			this.observable.addObserver(observer);
	}

	public void deleteObserver(Observer observer) {
		if (observer != null)
			this.observable.deleteObserver(observer);
	}

	public void deleteObservers() {
		this.observable.deleteObservers();
	}

	/**
	 * Types used for the method getDocumentType():
	 */
	public enum Type {
		METS, EAD, BAR, OTHER
	}

}
