/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.ead;

import java.io.*;
import java.util.Map;
import java.util.TreeMap;

import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.*;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.tools.exception.ExceptionCollector;
import ch.docuteam.tools.exception.ExceptionCollectorException;

/**
 * An instance of this class is the java-equivalent of an EAD file. It contains
 * <ul>
 * <li>{@link Header}</li>
 * <li>{@link ArchDesc}</li>
 * </ul>
 *
 * @author denis
 *
 */
public class Document extends DocumentAbstract {

	static protected final String DefaultEADTemplateFile = "resources/templates/ead.xml";

	static protected String EADTemplateFile = DefaultEADTemplateFile;

	protected Header eadHeader;
	protected ArchDesc archDesc;

	static public Document createEmpty() throws FileNotFoundException, DocumentException, ExceptionCollectorException {
		return read(EADTemplateFile);
	}

	static public Document read(String filePath)
			throws FileNotFoundException, DocumentException, ExceptionCollectorException {
		if (!new File(filePath).exists()) {
			throw new FileNotFoundException(filePath);
		}

		Document ead = Document.read(new BufferedReader(new FileReader(filePath)));
		ead.filePath = filePath;

		return ead;
	}

	static public Document read(BufferedReader br) throws DocumentException, ExceptionCollectorException {
		SAXReader reader = new SAXReader();
		reader.setDocumentFactory(new DocumentFactory());
		Document ead = (Document) reader.read(br);

		ExceptionCollector.clear();
		ead.parse();
		if (!ExceptionCollector.isEmpty()) {
			throw new ExceptionCollectorException();
		}

		return ead;
	}

	static public void setEADTemplateFile(String newEADTemplateFile) {
		EADTemplateFile = newEADTemplateFile;
	}

	public Header getEADHeader() {
		return eadHeader;
	}

	public ArchDesc getArchDesc() {
		return archDesc;
	}

	public void setId(String id) {
		eadHeader.setEadId(id);
	}

	public String getId() {
		return eadHeader.getEadId();
	}

	/**
	 * Return the type of this document: METS, EAD, ...
	 */
	@Override
	public Type getDocumentType() {
		return Type.EAD;
	}

	public ComponentAbstract searchUnitTitle(String title) {
		return archDesc.searchUnitTitle(title);
	}

	public void saveAs(String filePath) throws IOException {
		this.filePath = filePath;
		new File(this.filePath).getParentFile().mkdirs();

		save();
	}

	public void save() throws IOException {
		Writer oswriter = new java.io.OutputStreamWriter(new FileOutputStream(this.filePath), "utf-8");
		OutputFormat.createPrettyPrint();
		OutputFormat outformat = new OutputFormat("	", false, "utf-8");
		XMLWriter writer = new XMLWriter(oswriter, outformat);
		writer.write(this);
		writer.flush();
		writer.close();

		isModified = false;
	}

	@Override
	public String toString() {
		return new StringBuilder("\n====\tEAD-Document on '").append(this.filePath).append("'")
				.append(this.isModified ? "*" : "").append("\t====\n\t").append(this.eadHeader).append("\n")
				.append(this.archDesc).toString();
	}

	private Document parse() {
		Element root = this.getRootElement();
		eadHeader = Header.parse(this, root);
		archDesc = ArchDesc.parse(this, root);

		return this;
	}

	/**
	 * This class is needed within the class
	 * <a href="./Document.html">Document</a> to make the SAXReader generate an
	 * instance of the Document class.
	 *
	 * @author denis
	 */
	static private class DocumentFactory extends org.dom4j.DocumentFactory {
		// Factory methods
		@Override
		public Document createDocument() {
			Map<String, String> namespaces = new TreeMap<String, String>();
			namespaces.put("EAD", "urn:isbn:1-931666-22-9");
			namespaces.put("xlink", "http://www.w3.org/1999/xlink");
			this.setXPathNamespaceURIs(namespaces);

			ch.docuteam.darc.ead.Document ead = new ch.docuteam.darc.ead.Document();
			ead.setDocumentFactory(this);

			return ead;
		}

	}

}
