/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.lang.reflect.InvocationTargetException;

import bsh.EvalError;
import ch.docuteam.darc.exceptions.*;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.tools.string.ToolTipText;

/**
 * This class represents concrete instances of
 * <a href="./LevelMetadataElement.html">LevelMetadataElements</a>. Since an
 * instance of LevelMetadataElement can be "repeating", instances of this class
 * are the repetitions, together with their corresponding values.
 *
 * @author denis
 *
 */
public class MetadataElementInstance implements ToolTipText {

	static final private String KeyStartString = "${";
	static final private String KeyEndString = "}";

	private LevelMetadataElement levelMetadataElement;
	private NodeAbstract node;
	private int i;
	private String value;

	public MetadataElementInstance(LevelMetadataElement lme, NodeAbstract node, String value, int i) {
		this.levelMetadataElement = lme;
		this.node = node;
		this.i = i;
		this.value = value;
	}

	public String getName() {
		return this.levelMetadataElement.getId();
	}

	public String getValue() {
		return this.value;
	}

	/**
	 * The value may contain a key in form of ${key}. This key must be hidden
	 * when displaying me in the GUI.
	 * 
	 * @return
	 */
	public String getDisplayValue() {
		if ((this.value == null) || !(this.value.contains(KeyStartString) && this.value.contains(KeyEndString)))
			return this.value;

		int keyStartIndex = this.value.indexOf(KeyStartString);
		int keyEndIndex = this.value.indexOf(KeyEndString, keyStartIndex);
		if (keyEndIndex == -1)
			return this.value;

		return this.value.substring(0, keyStartIndex) + this.value.substring(keyEndIndex + KeyEndString.length());
	}

	public void setValue(String value) throws IllegalArgumentException, IllegalAccessException,
			InvocationTargetException, MetadataElementValidatorException, LevelMetadataElementIsReadOnly,
			MetadataElementSetterPostActionException, MetadataElementAllowedValuesException {
		// Avoid overwriting a value with the same value:
		if (this.value == null && value == null)
			return;
		if (this.value == null && value.isEmpty())
			return;
		if (value != null && value.equals(this.value))
			return;

		this.levelMetadataElement.setValueInNode(this.i, value, this.node);
		this.value = this.levelMetadataElement.getValueFromNode(this.i, this.node);
	}

	public void setValue_force(String value) throws IllegalArgumentException, IllegalAccessException,
			InvocationTargetException, MetadataElementValidatorException, MetadataElementSetterPostActionException,
			MetadataElementAllowedValuesException {
		// Avoid overwriting a value with the same value:
		if (this.value == null && value == null)
			return;
		if (this.value == null && value.isEmpty())
			return;
		if (value != null && value.equals(this.value))
			return;

		this.levelMetadataElement.setValueInNode_NoCheck(this.i, value, this.node);
		this.value = this.levelMetadataElement.getValueFromNode(this.i, this.node);
	}

	/**
	 * Set my value to the default value as defined in my LevelMetadataElement,
	 * but only if my value is not yet set (i.e. is null or empty). If it is
	 * already set, leave it as it is.
	 * 
	 * @throws IllegalArgumentException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws MetadataElementValidatorException
	 * @throws EvalError
	 * @throws MetadataElementSetterPostActionException
	 * @throws MetadataElementAllowedValuesException
	 */
	public void initializeValueIfNecessary() throws IllegalArgumentException, IllegalAccessException,
			InvocationTargetException, EvalError, MetadataElementValidatorException,
			MetadataElementSetterPostActionException, MetadataElementAllowedValuesException {
		if (this.value == null || this.value.isEmpty()) {
			String defaultValue = this.getLevelMetadataElement().getDefaultValueFromNode(this.node);
			if (defaultValue != null)
				this.setValue_force(defaultValue);
		}
	}

	public int getIndex() {
		return this.i;
	}

	public LevelMetadataElement getLevelMetadataElement() {
		return this.levelMetadataElement;
	}

	public String getAttributesString() {
		StringBuilder s = new StringBuilder();
		s.append(this.isMandatory() ? (this.value == null || this.value.isEmpty() ? "!" : "*")
				: (this.isAlwaysDisplayed() ? "O" : " "));
		s.append(this.isRepeatable() ? "+" : " ");
		s.append(this.isReadOnly() ? "X" : " ");

		return s.toString();
	}

	public boolean isMandatory() {
		return this.levelMetadataElement.isMandatory();
	}

	public boolean isAlwaysDisplayed() {
		return this.levelMetadataElement.isAlwaysDisplayed();
	}

	public boolean isRepeatable() {
		return this.levelMetadataElement.isRepeatable();
	}

	public boolean isReadOnly() {
		return this.levelMetadataElement.isReadOnly();
	}

	public boolean keepInTemplate() {
		return this.levelMetadataElement.keepInTemplate();
	}

	/**
	 * This implementation is awkward! I have to omit "!this.value.isEmpty()",
	 * otherwise I can't insert new empty MetadataElements.
	 * 
	 * @return
	 */
	public boolean isToBeDisplayed() {
		return (this.isMandatoryOrSet() || this.levelMetadataElement.isAlwaysDisplayed());
	}

	/**
	 * This implementation is awkward! I have to omit "!this.value.isEmpty()",
	 * otherwise I can't insert new empty MetadataElements.
	 * 
	 * @return
	 */
	public boolean isMandatoryOrSet() {
		return (this.levelMetadataElement.isMandatory() || this.value != null);
	}

	/**
	 * This implementation is awkward! I have to insert "this.value.isEmpty()",
	 * otherwise I can't display empty mandatory elements.
	 * 
	 * @return
	 */
	public boolean isMandatoryButNotSet() {
		return (this.levelMetadataElement.isMandatory() && (this.value == null || this.value.isEmpty()));
	}

	/**
	 * Return those elements which are mandatory or empty, but not undefined.
	 * 
	 * @return
	 */
	public boolean isMandatoryOrEmpty() {
		return (this.levelMetadataElement.isMandatory() || (this.value != null && this.value.isEmpty()));
	}

	public boolean canBeDeleted() throws IllegalArgumentException, IllegalAccessException, InvocationTargetException {
		return this.levelMetadataElement.canDeleteOneInstanceFromNode(this.node);
	}

	@Override
	public String getToolTipText() {
		return this.levelMetadataElement.getMetadataElement().getToolTipText();
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("MetadataElement [").append(this.getAttributesString())
				.append(this.levelMetadataElement.isRepeatable() ? this.i : " ").append(":")
				.append(this.levelMetadataElement.getId()).append("='").append(this.value).append("']");
		return builder.toString();
	}

}
