/**
 *	Copyright (C) 2014-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.lang.reflect.InvocationTargetException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import ch.docuteam.darc.exceptions.*;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.mets.structmap.NodeFolder;
import ch.docuteam.tools.translations.I18N;


/**
 * A MetadataElementValidator that accepts a string in the standard date format
 * "dd.MM.yyyy" (with or without days/months) and also accepts a date range of
 * two dates separated by a dash (both dates need to be in the same format), 
 * verifying whether values of parent and children nodes fit the respective values.
 * 
 * @author Andreas Nef, Docuteam GmbH
 * 
 */
public class MetadataElementValidatorDateHierarchyYear implements MetadataElementValidator {

	@Override
	public void check(String value, NodeAbstract context, MetadataElement metadataElement) throws MetadataElementValidatorException {
		//	There is no date range set, so return
		if (value == null || value.isEmpty())	return;

		//	Check for the locale's value of "no date"
		if (value.equals(I18N.translate_NoCheck("NoDate")))	return;

		//	TODO process also creationPeriod
		if (!(metadataElement.getAccessorName().startsWith("from") || metadataElement.getAccessorName().startsWith("to")))
			throw new MetadataElementValidatorException("cannot form a logic for this metadata element");

		try {
			//	check against the parent's value
			if (!context.isRoot()) {
				String parentGenericValue = ((NodeFolder) context.getParent()).getDynamicMetadataValueForName(metadataElement.getAccessorName());
				if (parentGenericValue == null || parentGenericValue.isEmpty())		return;
				
				Date parentDateValue = new SimpleDateFormat("yyyy").parse(parentGenericValue);
				if (metadataElement.getAccessorName().startsWith("from"))
				{
					if (parentDateValue.after(new SimpleDateFormat("yyyy").parse(value)))
						throw new MetadataElementValidatorException("the value is chronologically before the parent's value");
				}
				else if (metadataElement.getAccessorName().startsWith("to")) {
					if (parentDateValue.before(new SimpleDateFormat("yyyy").parse(value)))
						throw new MetadataElementValidatorException("the value is chronologically after the parent's value");
				}
			}
			
			//	check against the children's values
			if (context.isFolder()) {
				for (NodeAbstract node: ((NodeFolder)context).getDescendants()) {
					String childGenericValue = node.getDynamicMetadataValueForName(metadataElement.getAccessorName());
					if (childGenericValue != null && !childGenericValue.isEmpty()) {
						Date childDateValue = new SimpleDateFormat("yyyy").parse(childGenericValue);
						if (metadataElement.getAccessorName().startsWith("from"))
						{
							if (childDateValue.before(new SimpleDateFormat("yyyy").parse(value)))
								throw new MetadataElementValidatorException("the value is chronologically after the child's value");
						}
						else if (metadataElement.getAccessorName().startsWith("to")) {
							if (childDateValue.after(new SimpleDateFormat("yyyy").parse(value)))
								throw new MetadataElementValidatorException("the value is chronologically before the child's value");
						}
					}
				}
			}
		} catch (ParseException e) {
			throw new MetadataElementValidatorException("parse error: could not parse the value '" + value + "' as a year ('yyyy')");
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
			throw new MetadataElementValidatorException("error accessing the parent's metadata");
		} catch (IllegalAccessException e) {
			e.printStackTrace();
			throw new MetadataElementValidatorException("error accessing the parent's metadata");
		} catch (InvocationTargetException e) {
			e.printStackTrace();
			throw new MetadataElementValidatorException("error accessing the parent's metadata");
		} catch (MetadataElementIsNotAllowedException e) {
			e.printStackTrace();
			throw new MetadataElementValidatorException("error accessing the parent's metadata (element is not allowed)");
		} catch (MetadataElementIsNotDefinedException e) {
			e.printStackTrace();
			throw new MetadataElementValidatorException("error accessing the parent's metadata (element is not defined)");
		}
	}

}
