/**
 *	Copyright (C) 2014-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mdconfig;

import java.text.*;

import ch.docuteam.darc.exceptions.MetadataElementValidatorException;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.tools.translations.I18N;

/**
 * A MetadataElementValidator that accepts a string in the standard date format
 * "dd.MM.yyyy" (with or without days/months) and also accepts a date range of
 * two dates separated by a dash (both dates need to be in the same format).
 * 
 * @author Andreas Nef, Docuteam GmbH
 * 
 */
public class MetadataElementValidatorDateRangeCH implements MetadataElementValidator {

	@Override
	public void check(String value, NodeAbstract context, MetadataElement metadataElement)
			throws MetadataElementValidatorException {
		// There is no value set, so return
		if (value == null || value.isEmpty())
			return;

		// Check for the locale's value of "no date"
		if (value.equals(I18N.translate_NoCheck("NoDate")))
			return;

		String[] dates = value.split(" *- *");

		// only allow either a single date or a single date range
		if (dates.length > 2) {
			throw new MetadataElementValidatorException(
					I18N.translate_NoCheck("MessageValidatorDateRangeCHInvalidNumberOfValues", value));
		}

		// check the date values as such, using the validator for European date
		// format "[[d.]M.]yyyy"
		for (String date : dates) {
			new MetadataElementValidatorDateCH().check(date, context, metadataElement);
		}

		if (dates.length == 2) {
			// check that both dates of the range are of the same format
			if (dates[0].split("\\.").length != dates[1].split("\\.").length) {
				throw new MetadataElementValidatorException(
						I18N.translate_NoCheck("MessageValidatorDateRangeCHDifferentFormats", value));
			}

			DateFormat df;
			switch (dates[0].split("\\.").length) {
			case 1:
				df = new SimpleDateFormat("yyyy");
				break;
			case 2:
				df = new SimpleDateFormat("M.yyyy");
				break;
			case 3:
				df = new SimpleDateFormat("d.M.yyyy");
				break;
			default:
				// given the use of MetadataElementValidatorDateCH above, the
				// code should never reach this point
				throw new MetadataElementValidatorException(
						I18N.translate_NoCheck("MessageValidatorDateCHCantConvert", value));
			}
			df.setLenient(false);

			try {
				if (df.parse(dates[0]).after(df.parse(dates[1])))
					throw new MetadataElementValidatorException(
							I18N.translate_NoCheck("MessageValidatorDateRangeCHNotValid", value));
			} catch (ParseException e) {
				// given the use of MetadataElementValidatorDateCH above, the
				// code should never reach this point
				throw new MetadataElementValidatorException(
						I18N.translate_NoCheck("MessageValidatorDateCHCantConvert", value));
			}
		}
	}

}
