/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.darc.mets.dmdsec;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import ch.docuteam.darc.common.DocumentAbstract;
import ch.docuteam.darc.common.NodeAbstract;
import ch.docuteam.darc.mets.Document;
import ch.docuteam.tools.id.UniqueID;

/**
 * This abstract class, used by the class
 * <a href="../Document.html">Document</a>, represents the METS Descriptive
 * Metadata Section.
 * <p>
 * This class inserts Exceptions into the
 * <a href="../../../docutools/exception/ExceptionCollector.html">
 * ExceptionCollector</a> (instead of throwing them) in the following cases:
 * <ul>
 * <li>Parsing a METS file, when the number of DMDSections found in the METS
 * file is not exactly 1</li>
 * </ul>
 * 
 * @author denis
 */
public abstract class DMDSectionAbstract extends NodeAbstract {

	/**
	 * This method is used only when a DMDSection is created programmatically.
	 */
	@SuppressWarnings("unchecked")
	public DMDSectionAbstract(DocumentAbstract document) {
		this.document = document;

		// Place the new DMDSection before the AMDSection:
		this.element = DocumentHelper.createElement("METS:dmdSec");
		int amdSectionIndex = this.document.getRootElement().content()
				.indexOf(((Document) this.document).getAMDSection().getElement());
		this.document.getRootElement().content().add(amdSectionIndex, this.element);

		this.setId(UniqueID.getXML());

		((Document) this.document).addDMDSection(this);
	}

	/**
	 * This constructor is used only when a METS-File is being read. Create the
	 * DMDSection out of the given document.
	 */
	protected DMDSectionAbstract(Document document, Element element) {
		this.document = document;
		this.element = element;
		this.id = this.element.attributeValue("ID");
	}

	/**
	 * This method is used only when a METS-File is being read. Check if there
	 * are any DMDSections around, and if yes, decide to which subclass they
	 * belong and create instances.
	 */
	static public Map<String, DMDSectionAbstract> parse(Document document) {
		Map<String, DMDSectionAbstract> dmdSecs = new HashMap<String, DMDSectionAbstract>();

		@SuppressWarnings("rawtypes")
		List dmdSecNodes = document.selectNodes("./METS:mets/METS:dmdSec");

		// Check if a DMDSection exists at all - if not, return an empty List:
		if (dmdSecNodes.size() == 0)
			return dmdSecs;

		for (Object o : dmdSecNodes) {
			Element element = (Element) o;

			if (element.selectSingleNode("./METS:mdWrap[@MDTYPE = 'EAD']") != null) {
				DMDSectionWithEAD dmdSection = new DMDSectionWithEAD(document, element);
				dmdSecs.put(dmdSection.id, dmdSection);
			} else if (element.selectSingleNode("./METS:mdWrap[@MDTYPE = 'DC']") != null) {
				DMDSectionWithOAI_DC dmdSection = new DMDSectionWithOAI_DC(document, (Element) o);
				dmdSecs.put(dmdSection.id, dmdSection);
			} else {
				DMDSectionGeneric dmdSection = new DMDSectionGeneric(document, (Element) o);
				dmdSecs.put(dmdSection.id, dmdSection);
			}
		}

		return dmdSecs;
	}

	/**
	 * Remove this dmdSection from the document's dmds:
	 */
	public void delete() {
		((Document) this.document).deleteDMDSection(this);
		this.document.setIsModified();
		this.element.detach();
	}

	@Override
	public String toString() {
		return new StringBuilder("\n[DMDSection:").append(this.id).append("\n]").toString();
	}

}
