/**
 *	Copyright (C) 2011-2016 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;

import java.io.File;
import java.util.*;

import ch.docuteam.tools.os.EnvironmentVariable;
import ch.docuteam.tools.out.Logger;
import edu.harvard.hul.ois.jhove.*;

/**
 * This is an abstract class for getting file format information using JHOVE (JSTOR/Harvard Object Validation Environment).
 * <br>
 * The JHOVE subsystem requires the configuration file "jhove.conf" to be in the folder "config" in the DOCUTOOLS_HOME folder.
 * @author denis
 *
 */
public abstract class MetadataProviderJHOVE
{

	private static final String		JhoveConfigFile = "config/jhove.conf";
	private static final String		SaxClass = "org.apache.xerces.parsers.SAXParser";
	private static final String		CanValidate = "edu.harvard.hul.ois.jhove.canValidate";


	private static JhoveBase		JhoveBase = null;


	static
	{
		Logger.getLogger().debug("Retrieving jhove...");

		String configFilePath = EnvironmentVariable.getAsFolder("DOCUTOOLS_HOME") + "/" + JhoveConfigFile;

		try
		{
			JhoveBase = new JhoveBase();
			JhoveBase.setChecksumFlag(false);
			JhoveBase.init(configFilePath, SaxClass);
			if (JhoveBase.getAbort())		JhoveBase = null;
		}
		catch (java.lang.Exception e)
		{
			Logger.getLogger().warn("Could not initialize JHOVE", e);
			JhoveBase = null;
		}

		Logger.getLogger().debug(JhoveBase == null? "...NOK!": "...done");
	}


	/*
	 * Returns false if processing should be aborted. Calls itself recursively
	 * for directories.
	 */
	public static RepInfo getRepInfo(String fileName)
	{
		if (JhoveBase == null)					return null;
		if (!new File(fileName).exists())		return null;

		File file = null;
		boolean isTemp = false;
		long lastModified = -1;

		RepInfo info = new RepInfo(fileName);

		file = new File(fileName);

		if (!file.exists()) {
			info.setMessage(new ErrorMessage("file not found"));
			info.setWellFormed(RepInfo.FALSE);
		} else if (!file.isFile() || !file.canRead()) {
			info.setMessage(new ErrorMessage("file cannot be read"));
			info.setWellFormed(RepInfo.FALSE);
		} else {
			info.setSize(file.length());
			if (lastModified < 0) {
				lastModified = file.lastModified();
			}
			info.setLastModified(new Date(lastModified));

			/*
			 * Invoke all modules until one returns well-formed. If a module
			 * doesn't know how to validate, we don't want to throw
			 * arbitrary files at it, so we'll skip it.
			 */
			Iterator<?> iter = JhoveBase.getModuleList().iterator();
			while (iter.hasNext()) {
				Module mod = (Module) iter.next();
				RepInfo infc = (RepInfo) info.clone();

				if (mod.hasFeature(CanValidate)) {
					try {
						if (!JhoveBase.processFile(null, mod, /* moduleParam, */ false, file, infc)) {
							return null;
						}
						if (infc.getWellFormed() == RepInfo.TRUE) {
							info.copy(infc);
							break;
						}
						// We want to know what modules matched the
						// signature, so we force the sigMatch
						// property
						// to be persistent.
						info.setSigMatch(infc.getSigMatch());

					} catch (java.lang.Exception e) {
						/*
						 * The assumption is that in trying to analyze the
						 * wrong type of file, the module may go off its
						 * track and throw an exception, so we just continue
						 * on to the next module.
						 */
						continue;
					}
				}
			}
		}

		if (file != null && isTemp)		file.delete();

		return info;
	}


	//	The following are convenience methods (shortcuts for retrieving specific metadata directly):

	public static int getWellFormed(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.getWellFormed();
	}

	public static int getValid(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.getValid();
	}

	public static boolean isConsistent(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.isConsistent();
	}

	public static String getMimeType(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.getMimeType();
	}

	public static String getFormat(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.getFormat();
	}

	public static String getVersion(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.getVersion();
	}

	public static long getSize(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.getSize();
	}

	public static Module getModule(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.getModule();
	}

	public static List<String> getProfile(String fileName)
	{
		RepInfo result = getRepInfo(fileName);
		return (result == null)? null: result.getProfile();
	}


}
