package ch.docuteam.packer.gui.launcher;

import static java.nio.file.StandardWatchEventKinds.ENTRY_CREATE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_DELETE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_MODIFY;

import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledThreadPoolExecutor;

import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;

import ch.docuteam.darc.mets.Document.Mode;
import ch.docuteam.packer.gui.FileProperty;

public class SIPTable {
	
	private LauncherView launcherView;
	private JTable table;
	private SIPTableModel sipTableModel;
	private FileProperty selectedSip;
	Executor fileChangeWatcherExecutor;

	public JTable getTable() {
		return table;
	}
	
	// TODO Not sure if this field should be exposed
	public SIPTableModel getSipTableModel() {
		return sipTableModel;
	}


	public SIPTable(LauncherView launcherView) {
		this.launcherView = launcherView;
		sipTableModel = new SIPTableModel(launcherView);
		this.table = new JTable(sipTableModel);

		table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			@Override
			public void valueChanged(ListSelectionEvent e) {
				if (!e.getValueIsAdjusting())
					listSelectionChanged(e);
			}
		});
		table.addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(MouseEvent e) {
				listSelectionWasClicked(e);
			}
		});
		table.getColumnModel().getColumn(1).setMinWidth(50);
		table.getColumnModel().getColumn(1).setPreferredWidth(150);
		table.getColumnModel().getColumn(1).setMaxWidth(500);
		table.getColumnModel().getColumn(2).setMinWidth(80);
		table.getColumnModel().getColumn(2).setMaxWidth(80);
		// Right-align numbers in "size" and "last modified" column:
		DefaultTableCellRenderer rightAlignmentRenderer = new DefaultTableCellRenderer();
		rightAlignmentRenderer.setHorizontalAlignment(DefaultTableCellRenderer.RIGHT);
		table.getColumnModel().getColumn(2).setCellRenderer(rightAlignmentRenderer);
		table.getColumnModel().getColumn(3).setMinWidth(150);
		table.getColumnModel().getColumn(3).setMaxWidth(150);
		table.getColumnModel().getColumn(3).setCellRenderer(new DateTableCellRenderer());
		table.getColumnModel().getColumn(4).setMinWidth(40);
		table.getColumnModel().getColumn(4).setMaxWidth(40);
		// Make sipTable sortable:
		table.setAutoCreateRowSorter(true);

		// Table:
		// Use Enter to show underlying sip
		String open = "open.sip.readwrite";
		KeyStroke enter = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
		table.getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(enter, open);
		table.getActionMap().put(open, launcherView.getOpenSIPInWorkspaceAction());

		open = "open.sip.readwrite.no.file.ops.action";
		enter = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.ALT_DOWN_MASK);
		table.getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(enter, open);
		table.getActionMap().put(open, launcherView.getOpenSIPInWorkspaceReadWriteNoFileOpsAction());

		open = "open.sip.readonly";
		enter = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.SHIFT_DOWN_MASK);
		table.getInputMap(JTable.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(enter, open);
		table.getActionMap().put(open, launcherView.getOpenSIPInWorkspaceReadOnlyAction());
		
		fileChangeWatcherExecutor = new ScheduledThreadPoolExecutor(5);
		registerFileChangeWatchService();
	}

	// TODO a cleaner solution might be possible with some kind of signaling to
	// the thread when the workspace changes, or the executor implementation
	// might also be not optimal and the while (true) too
	public void registerFileChangeWatchService() {
		Runnable fileWatcherThread = new Runnable() {
			
			@Override
			public void run() {
				try (WatchService watcher = FileSystems.getDefault().newWatchService()) {
					WatchKey key = FileSystems.getDefault().getPath(launcherView.getSipDirectory()).register(watcher,
							ENTRY_CREATE, ENTRY_DELETE);

					while (true) {
						key = watcher.take();
						for (WatchEvent<?> event : key.pollEvents()) {
							if (event.kind() == ENTRY_MODIFY || event.kind() == ENTRY_CREATE
									|| event.kind() == ENTRY_DELETE) {
								Path changed = (Path) event.context();
								if (!changed.toString().endsWith("lock")) {
//									rereadSIPTable();
								}

							}
						}
						key.reset();
					}
				} catch (IOException | InterruptedException e) {
					e.printStackTrace();
				}
			}
		};
		fileChangeWatcherExecutor.execute(fileWatcherThread);
	}
	
	protected void listSelectionChanged(ListSelectionEvent e) {
		int selectedRow = table.getSelectedRow();
		if (selectedRow == -1) {
			selectedSip = null;
			launcherView.setSelectedSIP(null);
		} else {
			if (selectedRow < sipTableModel.getRowCount()) {
				selectedSip = sipTableModel.getSipAtIndex(table.convertRowIndexToModel(selectedRow));
				launcherView.setSelectedSIP(selectedSip);
			} else {
				selectedSip = null;
				launcherView.setSelectedSIP(null);
			}
		}
		launcherView.getFooterTextField().setText(selectedSip == null ? "" : selectedSip.getFile().getPath());
		launcherView.enableOrDisableActions();
	}
	
	/**
	 * Reread the SIP table. The number of SIPs in the workspace might have
	 * changed. Try to retain the current selection index.
	 */
	public void rereadSIPTable() {
		//	TODO	instead of rereading workspaces on demand, the workspace should be watched and updated if necessary.
		sipTableModel.readDirContent();
		sipTableModel.fireTableDataChanged();
		sipTableModel.updateSizeAndLockedByColumns();
		int selectedSipNewRow = sipTableModel.getRowIndexOfSip(selectedSip);
		if (selectedSipNewRow > -1) {
			try {
				//reselect the row that was selected prior to rereading the table
				table.getSelectionModel().setSelectionInterval(selectedSipNewRow, selectedSipNewRow);
			} catch (Exception ex) {
				//TODO something has to happen here, although it might be not necessary
			}
		}
	}
	
	protected void listSelectionWasClicked(MouseEvent e) {
		if (e.getClickCount() == 2) {
			// Double-Click:
			if ((e.getModifiersEx() & (InputEvent.ALT_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK)) == 0) {
				launcherView.openSelectedSIPInWorkspace(Mode.ReadWrite);
			} else if ((e.getModifiersEx()
					& (InputEvent.ALT_DOWN_MASK | InputEvent.SHIFT_DOWN_MASK)) == InputEvent.ALT_DOWN_MASK) {
				launcherView.openSelectedSIPInWorkspace(Mode.ReadWriteNoFileOps);
			} else if ((e.getModifiersEx()
					& (InputEvent.SHIFT_DOWN_MASK | InputEvent.ALT_DOWN_MASK)) == InputEvent.SHIFT_DOWN_MASK) {
				launcherView.openSelectedSIPInWorkspace(Mode.ReadOnly);
			}
		} else if (e.getButton() == MouseEvent.BUTTON3) {
			// Right-Click:

			// Find out which row was right-clicked:
			int rowNumber = table.rowAtPoint(e.getPoint());

			// Select this row:
			table.getSelectionModel().setSelectionInterval(rowNumber, rowNumber);

			// Show popup menu:
			launcherView.getPopupMenu().show(table, e.getX(), e.getY());
		}
		// else: ignore. Single clicks are handled in listSelectionChanged().
	}
	
	
}
