package ch.docuteam.packer.gui.sipView;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.File;
import java.io.IOException;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JTable;
import javax.swing.TransferHandler;
import javax.swing.tree.TreePath;

import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.treetable.TreeTableNode;

import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.mets.structmap.NodeAbstract.SubmitStatus;
import ch.docuteam.packer.gui.sipView.tableModel.TreeTableModel;
import ch.docuteam.darc.mets.structmap.NodeFolder;
import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.translations.I18N;

/**
 * This is my Transfer Handler for D&D:
 * 
 * @author denis
 */
public class TreeTableTransferHandler extends TransferHandler {

	private SIPView sipView;

	public TreeTableTransferHandler(SIPView sipView) {
		super();
		this.sipView = sipView;
	}

	/**
	 * Allow move only
	 */
	@Override
	public int getSourceActions(JComponent c) {
		// MOVE/COPY/COPY_OR_MOVE/LINK/NONE
		return COPY_OR_MOVE; 
	}

	/**
	 * Create an instance of our own Transferable out of the drag source
	 * component.
	 */
	@Override
	protected Transferable createTransferable(JComponent c) {
		return new TreeNodeListTransferable((JXTreeTable) c);
	}

	/**
	 * Only on drop, not on paste. Allow only our own data flavor and file
	 * lists. Allow dragging multiple items. Check source nodes and target node.
	 */
	@Override
	public boolean canImport(TransferSupport transferSupport) {
		// If this document doesn't allow file operations, disable D&D:
		if (!sipView.getDocument().areFileOperationsAllowed()) {
			return false;
		}

		// Accept only drop:
		if (!transferSupport.isDrop()) {
			return false;
		}

		// Don't allow insert, only "over":
		if (((JTable.DropLocation) transferSupport.getDropLocation()).isInsertRow()) {
			return false;
		}

		// Accept only file lists or our own data flavor:
		if (!(transferSupport.isDataFlavorSupported(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor)
				|| transferSupport.isDataFlavorSupported(DataFlavor.javaFileListFlavor))) {
			return false;
		}

		// Check target to allow DnD:
		int targetIndex = ((JTable.DropLocation) transferSupport.getDropLocation()).getRow();

		// This happens when the drag target is outside of the table:
		if (targetIndex == -1) {
			return false;
		}

		// Target node:
		NodeAbstract targetNode = (NodeAbstract) ((JXTreeTable) transferSupport.getComponent())
				.getPathForRow(targetIndex).getLastPathComponent();

		// Don't allow drop on files:
		if (targetNode.isFile()) {
			return false;
		}

		// Don't allow drop if target or any of its predecessors is not readable
		// or not writable:
		if (!targetNode.fileExists() || !targetNode.canRead() || !targetNode.canWrite()
				|| targetNode.hasPredecessorNotReadableOrWritableByCurrentUser()) {
			return false;
		}

		// Now distinguish the different data flavors:
		if (transferSupport.isDataFlavorSupported(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor)) {
			// Moving nodes within packer:

			String targetSIP = sipView.getDocument().getSIPFolder();
			TreeNodeListTransferable transferData = null;
			String sourceSIP = "";
			try {
				transferData = (TreeNodeListTransferable) transferSupport.getTransferable()
						.getTransferData(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor);
				sourceSIP = transferData.getSipPath();
			} catch (UnsupportedFlavorException ex) {
				// Ignore silently:
				if (sipView.getLauncherView().isInDevelopMode()) {
					ex.printStackTrace();
				}
				return false;
			} catch (IOException ex) {
				// Ignore silently:
				if (sipView.getLauncherView().isInDevelopMode()) {
					ex.printStackTrace();
				}
				return false;
			}

			// Don't allow dragging between different SIPs:
			if (!targetSIP.equals(sourceSIP)) {
				return false;
			}

			// Source parent node:
			TreeTableNode sourceParent = transferData.getDraggedNodes().get(0).getParent();

			// Don't allow dragging onto own parent:
			if (sourceParent == targetNode) {
				return false;
			}

			// Loop through all source nodes and test each of them:
			for (NodeAbstract node : transferData.getDraggedNodes()) {
				// Don't allow root as drag source (root can not be moved):
				if (node.isRoot()) {
					return false;
				}

				// Make sure that all selected source nodes have the same
				// parent. if not, reject the drag:
				if (node.getParent() != sourceParent) {
					return false;
				}

				// Don't allow dragging onto one of the source nodes:
				if (node == targetNode) {
					return false;
				}

				// Don't allow dragging onto a descendant of one of the source
				// nodes:
				if (node.isFolder() && ((NodeFolder) node).isPredecessorOf(targetNode)) {
					return false;
				}

				// Don't allow dragging if the node is being/has been submitted:
				if (node.getSubmitStatus().equals(SubmitStatus.SubmitRequestPending)
						|| node.getSubmitStatus().equals(SubmitStatus.Submitted)) {
					return false;
				}

				// Don't allow dragging if the node itself or any of its
				// predecessors is not readable or not writable:
				if (!node.fileExists() || !node.canRead() || !node.canWrite()
						|| node.hasPredecessorNotReadableOrWritableByCurrentUser()) {
					return false;
				}

				// Don't allow dragging if the node itself or any of its
				// descendants is not readable or not writable:
				if (node.isFolder() && (((NodeFolder) node).hasDescendantNotReadableOrWritableByCurrentUser())) {
					return false;
				}
			}

			return true;
		} else if (transferSupport.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
			// When dragging from the file system, show Drag-Copy cursor (not
			// the default Drag-Move cursor):
			transferSupport.setDropAction(COPY);

			// Inserting files and folders from the file system or any other
			// fileList provider:
			// Always allow file lists:
			return true;
		}

		// Reject any other data flavor:
		return false;
	}

	/**
	 * Actually import or move the data. Distinguish file lists (importing) and
	 * our own data flavor (moving nodes within the tree).
	 */
	@Override
	public boolean importData(TransferSupport transferSupport) {
		if (!this.canImport(transferSupport))
			return false;

		// Import files or move nodes within the tree?
		boolean isOK = false;
		if (transferSupport.isDataFlavorSupported(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor)) {
			// It is our own data flavor:
			isOK = this.moveNodesWithinSIP(transferSupport);
		} else if (transferSupport.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
			// It is a file list:
			isOK = this.importFileList(transferSupport);
		} else
			return false; // Should actually never happen: flavor is neither
							// javaFileListFlavor nor
							// DocuteamPackerTreeNodeListDataFlavor.

		return isOK;
	}

	/**
	 * For cleanup after successful move. Not needed here for now.
	 */
	@Override
	public void exportDone(JComponent component, Transferable t, int action) {
	}

	/**
	 * Move nodes within the tree. This method is only called when the
	 * DataFlavor is our own
	 * (TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor).
	 * 
	 * @param transferSupport
	 * @return
	 */
	private boolean moveNodesWithinSIP(TransferSupport transferSupport) {
		// Target component:
		JXTreeTable treeTable = (JXTreeTable) transferSupport.getComponent();

		// Target tree path:
		TreePath targetPath = treeTable
				.getPathForRow(((JTable.DropLocation) transferSupport.getDropLocation()).getRow());
		Logger.getLogger().debug("D&D TargetPath       = " + targetPath);

		// Target node:
		NodeFolder targetNode = (NodeFolder) targetPath.getLastPathComponent();
		Logger.getLogger().debug("D&D TargetNode       = " + targetNode);

		// Source node's parent path:
		TreePath sourceParentPath;

		// Do the move:
		try {
			TreeNodeListTransferable transferData = (TreeNodeListTransferable) transferSupport.getTransferable()
					.getTransferData(TreeNodeListTransferable.DocuteamPackerTreeNodeListDataFlavor);

			// Source parent path (I know that all source nodes have the same
			// parent):
			sourceParentPath = transferData.getParentPath();
			Logger.getLogger().debug("D&D SourceParentPath = " + sourceParentPath);

			for (NodeAbstract n : transferData.getDraggedNodes())
				n.moveTo(targetNode);
		} catch (java.lang.Exception e) {
			JOptionPane.showMessageDialog(sipView, e.toString(), I18N.translate("TitleCantMoveItem"),
					JOptionPane.WARNING_MESSAGE);
			return false;
		}

		// Refresh the view:
		TreeTableModel model = (TreeTableModel) treeTable.getTreeTableModel();

		model.refreshTreeStructure(sourceParentPath);
		model.refreshTreeStructure(targetPath);

		treeTable.expandPath(sourceParentPath);
		treeTable.expandPath(targetPath);

		return true;
	}

	/**
	 * Import files. This method is only called when the DataFlavor is
	 * DataFlavor.javaFileListFlavor.
	 * 
	 * @param transferSupport
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private boolean importFileList(TransferSupport transferSupport) {
		try {
			List<File> droppedFiles = (List<File>) transferSupport.getTransferable()
					.getTransferData(DataFlavor.javaFileListFlavor);
			Logger.getLogger().debug("D&D files = " + droppedFiles);

			// Do the import:
			sipView.importFilesAndFolders(droppedFiles);
		} catch (Exception ex) {
			ex.printStackTrace();
			return false;
		}

		return true;
	}

}
