package ch.docuteam.packer.gui.sipView.tableModel;

import static ch.docuteam.packer.gui.PackerConstants.*;
import java.util.List;

import javax.swing.DefaultCellEditor;
import javax.swing.JComboBox;
import javax.swing.JOptionPane;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

import ch.docuteam.darc.exceptions.MetadataElementValidatorException;
import ch.docuteam.darc.mdconfig.LevelMetadataElement;
import ch.docuteam.darc.mdconfig.MetadataElementInstance;
import ch.docuteam.darc.mets.structmap.NodeAbstract;
import ch.docuteam.darc.util.KeyAndValue;
import ch.docuteam.packer.gui.sipView.SIPView;
import ch.docuteam.packer.gui.sipView.StyledComboBoxUI;
import ch.docuteam.packer.gui.sipView.TextAreaTableCellEditor;
import ch.docuteam.packer.gui.sipView.cellRenderer.UnfocusableCellRenderer;
import ch.docuteam.tools.gui.TableModelWithSpecificCellEditorPerRow;
import ch.docuteam.tools.translations.I18N;

/**
 * Metadata table model containing the method "getToolTipText(int i)" to return
 * the specific tooltip text of the corresponding metadataElementInstance.
 * 
 * @author Docuteam
 *
 */
public class MetadataTableModel extends AbstractTableModel implements TableModelWithSpecificCellEditorPerRow {

	private SIPView sipView;

	private UnfocusableCellRenderer unfocusableCellRenderer = new UnfocusableCellRenderer();

	private TextAreaTableCellEditor textAreaRenderer = new TextAreaTableCellEditor();
	private TextAreaTableCellEditor textAreaEditor = new TextAreaTableCellEditor();
	private DefaultCellEditor comboBoxEditor = new DefaultCellEditor(new StyledComboBox());

	public MetadataTableModel(SIPView sipView) {
		// Initialize the ComboBoxEditor:
		comboBoxEditor.setClickCountToStart(CLICK_COUNT_TO_START);
		this.sipView = sipView;
	}

	private NodeAbstract fileStructureNode;
	private List<MetadataElementInstance> metadataElementInstances;

	/**
	 * Return the specific tooltip text of this MetadataElementInstance
	 * 
	 * @param i
	 * @return
	 */
	public String getToolTipText(int i) {
		return metadataElementInstances.get(i).getToolTipText();
	}

	/**
	 * Return the MetadataElement at the list position i.
	 * 
	 * @param i
	 * @return
	 */
	public MetadataElementInstance get(int i) {
		return metadataElementInstances.get(i);
	}

	public void setFileStructureNode(NodeAbstract fileStructureNode) {
		if (fileStructureNode == null) {
			fileStructureNode = null;
			metadataElementInstances = null;
		} else {
			this.fileStructureNode = fileStructureNode;
			metadataElementInstances = fileStructureNode.getDynamicMetadataElementInstancesToBeDisplayed();
		}

		fireTableDataChanged();
	}

	@Override
	public int getRowCount() {
		if (metadataElementInstances == null)
			return 0;

		return metadataElementInstances.size();
	}

	@Override
	public int getColumnCount() {
		return 3;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getColumnName(int column) {
		switch (column) {
		case 0:
			return I18N.translate("HeaderMetadataAttributes");
		case 1:
			return I18N.translate("HeaderMetadataName");
		case 2:
			return I18N.translate("HeaderMetadataValue");
		}

		return null;
	}

	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		if (metadataElementInstances == null)
			return null;

		MetadataElementInstance mdei = metadataElementInstances.get(rowIndex);

		switch (columnIndex) {
		case 0:
			return mdei.getAttributesString();
		case 1:
			return I18N.translate(mdei.getName());
		case 2:
			return mdei.getDisplayValue();
		}

		return null;
	}

	@Override
	public boolean isCellEditable(int rowIndex, int columnIndex) {
		if (columnIndex != EDITABLE_COLUMN_INDEX)
			return false;

		if (!sipView.getDocument().canWrite())
			return false;

		if (!fileStructureNode.fileExists() || !fileStructureNode.canRead() || !fileStructureNode.canWrite())
			return false;

		if (!fileStructureNode.doesSubmitStatusAllowEditing())
			return false;

		if (metadataElementInstances == null)
			return false;

		if (!metadataElementInstances.get(rowIndex).isReadOnly())
			return true;

		return false;
	}

	@Override
	public void setValueAt(Object value, int rowIndex, int columnIndex) {
		if (metadataElementInstances == null)
			return;
		if (columnIndex != EDITABLE_COLUMN_INDEX)
			return;

		try {
			if (value == null)
				return;

			// There are only two different types possible for value: String or
			// KeyAndValue:
			if (value.getClass() == KeyAndValue.class)
				metadataElementInstances.get(rowIndex).setValue(((KeyAndValue) value).getOriginalString());
			else
				metadataElementInstances.get(rowIndex).setValue(value.toString());
		} catch (MetadataElementValidatorException ex) {
			JOptionPane.showMessageDialog(sipView, ex.getMessage());
		} catch (Exception ex) {
			ex.printStackTrace();
		}

		fireTableDataChanged();
	}

	/**
	 * This method returns a specific cell editor, depending on the row and
	 * column: If the row contains a MDElement whose allowedValues attribute is
	 * not null, return a ComboBox containing the supplied values as the cell
	 * editor. Otherwise return null, meaning the default cell editor.
	 * 
	 * @param rowIndex
	 * @return
	 */
	@Override
	public TableCellEditor getCellEditor(int rowIndex, int columnIndex) {
		LevelMetadataElement lme = metadataElementInstances.get(rowIndex).getLevelMetadataElement();

		// If the levelMetadataElement has several allowedValues, the editor is
		// the ComboBoxEditor
		// (takes precedence over the TextAreaEditor):
		List<KeyAndValue> allowedValues = lme.getMetadataElement().getAllowedValues();
		if (allowedValues != null && !allowedValues.isEmpty()) {
			JComboBox comboBox = (JComboBox) comboBoxEditor.getComponent();

			comboBox.removeAllItems();
			for (KeyAndValue item : allowedValues)
				comboBox.addItem(item);

			// If the first item of the allowedValues list is a "*", make this
			// ComboBox editable:
			if (allowedValues.get(0).getOriginalString().equals("*")) {
				comboBox.removeItemAt(0);
				comboBox.setEditable(true);
			} else {
				comboBox.setEditable(false);
			}
			return comboBoxEditor;
		}

		// If the levelMetadataElement has a displayRows value >= 2, the editor
		// is a TextAreaTableCellEditor:
		int displayRows = lme.getDisplayRows();
		if (displayRows >= 2)
			return textAreaEditor;

		// Otherwise return null - this causes JTable to return the default cell
		// editor.
		return null;
	}

	/**
	 * This method returns a specific cell renderer, depending on the row and
	 * column: If the row contains a MDElement whose allowedValues attribute is
	 * not null, return a ComboBox containing the supplied values as the cell
	 * editor. Otherwise return null, meaning the default cell editor.
	 * 
	 * @param rowIndex
	 * @return
	 */
	@Override
	public TableCellRenderer getCellRenderer(int rowIndex, int columnIndex) {
		// If the levelMetadataElement has a displayRows value >= 2, the
		// renderer is a TextAreaTableCellEditor:
		int displayRows = metadataElementInstances.get(rowIndex).getLevelMetadataElement().getDisplayRows();
		if (displayRows < 2) {
			return unfocusableCellRenderer;
		} else {
			return textAreaRenderer;
		}
	}

	class StyledComboBox extends JComboBox {
		public StyledComboBox() {
			setUI(new StyledComboBoxUI());
		}
	}
}
