/*
 * Decompiled with CFR 0.152.
 */
package de.waldheinz.fs.fat;

import de.waldheinz.fs.BlockDevice;
import de.waldheinz.fs.fat.BootSector;
import de.waldheinz.fs.fat.FatType;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;

final class Fat {
    public static final int FIRST_CLUSTER = 2;
    private final long[] entries;
    private final FatType fatType;
    private final int sectorCount;
    private final int sectorSize;
    private final BlockDevice device;
    private final BootSector bs;
    private final long offset;
    private final int lastClusterIndex;
    private int lastAllocatedCluster;

    public static Fat read(BootSector bs, int fatNr) throws IOException, IllegalArgumentException {
        if (fatNr > bs.getNrFats()) {
            throw new IllegalArgumentException("boot sector says there are only " + bs.getNrFats() + " FATs when reading FAT #" + fatNr);
        }
        long fatOffset = bs.getFatOffset(fatNr);
        Fat result = new Fat(bs, fatOffset);
        result.read();
        return result;
    }

    public static Fat create(BootSector bs, int fatNr) throws IOException, IllegalArgumentException {
        if (fatNr > bs.getNrFats()) {
            throw new IllegalArgumentException("boot sector says there are only " + bs.getNrFats() + " FATs when creating FAT #" + fatNr);
        }
        long fatOffset = bs.getFatOffset(fatNr);
        Fat result = new Fat(bs, fatOffset);
        if (bs.getDataClusterCount() > (long)result.entries.length) {
            throw new IOException("FAT too small for device");
        }
        result.init(bs.getMediumDescriptor());
        result.write();
        return result;
    }

    private Fat(BootSector bs, long offset) throws IOException {
        this.bs = bs;
        this.fatType = bs.getFatType();
        if (bs.getSectorsPerFat() > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("FAT too large");
        }
        if (bs.getSectorsPerFat() <= 0L) {
            throw new IOException("boot sector says there are " + bs.getSectorsPerFat() + " sectors per FAT");
        }
        if (bs.getBytesPerSector() <= 0) {
            throw new IOException("boot sector says there are " + bs.getBytesPerSector() + " bytes per sector");
        }
        this.sectorCount = (int)bs.getSectorsPerFat();
        this.sectorSize = bs.getBytesPerSector();
        this.device = bs.getDevice();
        this.offset = offset;
        this.lastAllocatedCluster = 2;
        if (bs.getDataClusterCount() > Integer.MAX_VALUE) {
            throw new IOException("too many data clusters");
        }
        if (bs.getDataClusterCount() == 0L) {
            throw new IOException("no data clusters");
        }
        this.lastClusterIndex = (int)bs.getDataClusterCount() + 2;
        this.entries = new long[(int)((float)(this.sectorCount * this.sectorSize) / this.fatType.getEntrySize())];
        if (this.lastClusterIndex > this.entries.length) {
            throw new IOException("file system has " + this.lastClusterIndex + "clusters but only " + this.entries.length + " FAT entries");
        }
    }

    public FatType getFatType() {
        return this.fatType;
    }

    public BootSector getBootSector() {
        return this.bs;
    }

    public BlockDevice getDevice() {
        return this.device;
    }

    private void init(int mediumDescriptor) {
        this.entries[0] = (long)(mediumDescriptor & 0xFF) | 0xFFFFF00L & this.fatType.getBitMask();
        this.entries[1] = this.fatType.getEofMarker();
    }

    private void read() throws IOException {
        byte[] data = new byte[this.sectorCount * this.sectorSize];
        this.device.read(this.offset, ByteBuffer.wrap(data));
        for (int i = 0; i < this.entries.length; ++i) {
            this.entries[i] = this.fatType.readEntry(data, i);
        }
    }

    public void write() throws IOException {
        this.writeCopy(this.offset);
    }

    public void writeCopy(long offset) throws IOException {
        byte[] data = new byte[this.sectorCount * this.sectorSize];
        for (int index = 0; index < this.entries.length; ++index) {
            this.fatType.writeEntry(data, index, this.entries[index]);
        }
        this.device.write(offset, ByteBuffer.wrap(data));
    }

    public int getMediumDescriptor() {
        return (int)(this.entries[0] & 0xFFL);
    }

    public long getEntry(int index) {
        return this.entries[index];
    }

    public int getLastFreeCluster() {
        return this.lastAllocatedCluster;
    }

    public long[] getChain(long startCluster) {
        this.testCluster(startCluster);
        int count = 1;
        long cluster = startCluster;
        while (!this.isEofCluster(this.entries[(int)cluster])) {
            ++count;
            cluster = this.entries[(int)cluster];
        }
        long[] chain = new long[count];
        chain[0] = startCluster;
        cluster = startCluster;
        int i = 0;
        while (!this.isEofCluster(this.entries[(int)cluster])) {
            cluster = this.entries[(int)cluster];
            chain[++i] = cluster;
        }
        return chain;
    }

    public long getNextCluster(long cluster) {
        this.testCluster(cluster);
        long entry = this.entries[(int)cluster];
        if (this.isEofCluster(entry)) {
            return -1L;
        }
        return entry;
    }

    public long allocNew() throws IOException {
        int i;
        int entryIndex = -1;
        for (i = this.lastAllocatedCluster; i < this.lastClusterIndex; ++i) {
            if (!this.isFreeCluster(i)) continue;
            entryIndex = i;
            break;
        }
        if (entryIndex < 0) {
            for (i = 2; i < this.lastAllocatedCluster; ++i) {
                if (!this.isFreeCluster(i)) continue;
                entryIndex = i;
                break;
            }
        }
        if (entryIndex < 0) {
            throw new IOException("FAT Full (" + (this.lastClusterIndex - 2) + ", " + i + ")");
        }
        this.entries[entryIndex] = this.fatType.getEofMarker();
        this.lastAllocatedCluster = entryIndex % this.lastClusterIndex;
        if (this.lastAllocatedCluster < 2) {
            this.lastAllocatedCluster = 2;
        }
        return entryIndex;
    }

    public int getFreeClusterCount() {
        int result = 0;
        for (int i = 2; i < this.lastClusterIndex; ++i) {
            if (!this.isFreeCluster(i)) continue;
            ++result;
        }
        return result;
    }

    public int getLastAllocatedCluster() {
        return this.lastAllocatedCluster;
    }

    public long[] allocNew(int nrClusters) throws IOException {
        long[] rc = new long[nrClusters];
        rc[0] = this.allocNew();
        for (int i = 1; i < nrClusters; ++i) {
            rc[i] = this.allocAppend(rc[i - 1]);
        }
        return rc;
    }

    public long allocAppend(long cluster) throws IOException {
        long newCluster;
        this.testCluster(cluster);
        while (!this.isEofCluster(this.entries[(int)cluster])) {
            cluster = this.entries[(int)cluster];
        }
        this.entries[(int)cluster] = newCluster = this.allocNew();
        return newCluster;
    }

    public void setEof(long cluster) {
        this.testCluster(cluster);
        this.entries[(int)cluster] = this.fatType.getEofMarker();
    }

    public void setFree(long cluster) {
        this.testCluster(cluster);
        this.entries[(int)cluster] = 0L;
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof Fat)) {
            return false;
        }
        Fat other = (Fat)obj;
        if (this.fatType != other.fatType) {
            return false;
        }
        if (this.sectorCount != other.sectorCount) {
            return false;
        }
        if (this.sectorSize != other.sectorSize) {
            return false;
        }
        if (this.lastClusterIndex != other.lastClusterIndex) {
            return false;
        }
        if (!Arrays.equals(this.entries, other.entries)) {
            return false;
        }
        return this.getMediumDescriptor() == other.getMediumDescriptor();
    }

    public int hashCode() {
        int hash = 7;
        hash = 23 * hash + Arrays.hashCode(this.entries);
        hash = 23 * hash + this.fatType.hashCode();
        hash = 23 * hash + this.sectorCount;
        hash = 23 * hash + this.sectorSize;
        hash = 23 * hash + this.lastClusterIndex;
        return hash;
    }

    protected boolean isFreeCluster(long entry) {
        if (entry > Integer.MAX_VALUE) {
            throw new IllegalArgumentException();
        }
        return this.entries[(int)entry] == 0L;
    }

    protected boolean isReservedCluster(long entry) {
        return this.fatType.isReservedCluster(entry);
    }

    protected boolean isEofCluster(long entry) {
        return this.fatType.isEofCluster(entry);
    }

    protected void testCluster(long cluster) throws IllegalArgumentException {
        if (cluster < 2L || cluster >= (long)this.entries.length) {
            throw new IllegalArgumentException("invalid cluster value " + cluster);
        }
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.getClass().getSimpleName());
        sb.append("[type=");
        sb.append((Object)this.fatType);
        sb.append(", mediumDescriptor=0x");
        sb.append(Integer.toHexString(this.getMediumDescriptor()));
        sb.append(", sectorCount=");
        sb.append(this.sectorCount);
        sb.append(", sectorSize=");
        sb.append(this.sectorSize);
        sb.append(", freeClusters=");
        sb.append(this.getFreeClusterCount());
        sb.append("]");
        return sb.toString();
    }
}

