/*
 * Decompiled with CFR 0.152.
 */
package gov.loc.repository.bagit.reader;

import gov.loc.repository.bagit.domain.Version;
import gov.loc.repository.bagit.exceptions.InvalidBagMetadataException;
import gov.loc.repository.bagit.exceptions.InvalidBagitFileFormatException;
import gov.loc.repository.bagit.exceptions.UnparsableVersionException;
import gov.loc.repository.bagit.reader.KeyValueReader;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.AbstractMap;
import java.util.Arrays;
import java.util.List;
import java.util.ResourceBundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.helpers.MessageFormatter;

public final class BagitTextFileReader {
    private static final Logger logger = LoggerFactory.getLogger(BagitTextFileReader.class);
    private static final byte[] BOM = new byte[]{-17, -69, -65};
    private static final ResourceBundle messages = ResourceBundle.getBundle("MessageBundle");
    private static final Version VERSION_1_0 = new Version(1, 0);
    private static final String LINE1_REGEX = "(BagIt-Version: )\\d*\\.\\d*";
    private static final String LINE2_REGEX = "(Tag-File-Character-Encoding: )\\S*";

    private BagitTextFileReader() {
    }

    public static AbstractMap.SimpleImmutableEntry<Version, Charset> readBagitTextFile(Path bagitFile) throws IOException, UnparsableVersionException, InvalidBagMetadataException, InvalidBagitFileFormatException {
        logger.debug(messages.getString("reading_version_and_encoding"), (Object)bagitFile);
        BagitTextFileReader.throwErrorIfByteOrderMarkIsPresent(bagitFile);
        List<AbstractMap.SimpleImmutableEntry<String, String>> pairs = KeyValueReader.readKeyValuesFromFile(bagitFile, ":", StandardCharsets.UTF_8);
        String version = null;
        Charset encoding = null;
        for (AbstractMap.SimpleImmutableEntry<String, String> pair : pairs) {
            if ("BagIt-Version".equals(pair.getKey())) {
                version = pair.getValue();
                logger.debug(messages.getString("bagit_version"), (Object)version);
            }
            if (!"Tag-File-Character-Encoding".equals(pair.getKey())) continue;
            encoding = Charset.forName(pair.getValue());
            logger.debug(messages.getString("tag_file_encoding"), (Object)encoding);
        }
        if (version == null || encoding == null) {
            throw new InvalidBagitFileFormatException(messages.getString("invalid_bagit_text_file_error"));
        }
        Version parsedVersion = BagitTextFileReader.parseVersion(version);
        if (parsedVersion.isSameOrNewer(VERSION_1_0)) {
            List<String> lines = Files.readAllLines(bagitFile, StandardCharsets.UTF_8);
            BagitTextFileReader.throwErrorIfLinesDoNotMatchStrict(lines);
        }
        return new AbstractMap.SimpleImmutableEntry<Version, Object>(parsedVersion, encoding);
    }

    private static void throwErrorIfByteOrderMarkIsPresent(Path bagitFile) throws IOException, InvalidBagitFileFormatException {
        byte[] firstFewBytesInFile = Arrays.copyOfRange(Files.readAllBytes(bagitFile), 0, BOM.length);
        if (Arrays.equals(BOM, firstFewBytesInFile)) {
            String formattedMessage = messages.getString("bom_present_error");
            throw new InvalidBagitFileFormatException(MessageFormatter.format((String)formattedMessage, (Object)bagitFile).getMessage());
        }
    }

    static void throwErrorIfLinesDoNotMatchStrict(List<String> lines) throws InvalidBagitFileFormatException {
        if (lines.size() > 2) {
            List<String> offendingLines = lines.subList(2, lines.size() - 1);
            throw new InvalidBagitFileFormatException(MessageFormatter.format((String)messages.getString("strict_only_two_lines_error"), offendingLines).getMessage());
        }
        if (!lines.get(0).matches(LINE1_REGEX)) {
            throw new InvalidBagitFileFormatException(MessageFormatter.format((String)messages.getString("strict_first_line_error"), (Object)lines.get(0)).getMessage());
        }
        if (!lines.get(1).matches(LINE2_REGEX)) {
            throw new InvalidBagitFileFormatException(MessageFormatter.format((String)messages.getString("strict_second_line_error"), (Object)lines.get(0)).getMessage());
        }
    }

    public static Version parseVersion(String version) throws UnparsableVersionException {
        if (!version.contains(".")) {
            throw new UnparsableVersionException(messages.getString("unparsable_version_error"), version);
        }
        String[] parts = version.trim().split("\\.");
        if (parts.length != 2 || parts[0].isEmpty() || parts[1].isEmpty()) {
            throw new UnparsableVersionException(messages.getString("unparsable_version_error"), version);
        }
        int major = Integer.parseInt(parts[0]);
        int minor = Integer.parseInt(parts[1]);
        return new Version(major, minor);
    }
}

