package com.pdftools.converter;

/**
 * The Converter class provides the functionality for an application to
 * connect to the document converter service and use its API.
 * The application should create only a single object of this class, and
 * share this among multiple parallel activities. Upon termination, the application
 * should call the destroyObject() method to release also server side resources.
 */
public class Converter
{
    /**
     * Create a converter object.
     * @param serviceUrl  The service URL for connecting to the document converter service.
     *                    When null is specified, the default "tcp://localhost:7981/O2PService" is used.
     * @return the Converter object (null, if a connection to the server cannot be established)
     */
    public static Converter createConverter(String serviceUrl)
    {
        long handle = createHandle(serviceUrl);
        if(handle == 0)
            return null;
        Converter instance = new Converter();
        instance._handle = handle;
        return instance;
    }

    /**
     * Release the resources allocated for the converter object.
     * The object is no longer usable after this.
     */
    public void destroyObject()
    {
        if(_handle != 0)
            destroyHandle(_handle);
        _handle = 0;
    }

    /**
     * Stores result information returned from document conversion processing.
     */
    public static class ConversionResult
    {
        public int ErrorCode;
        public String ErrorText;
        public String Title;

        /**
         * @param code	error code
         * @param text	error text
         * @param title	error title
         */
        public void set(int code, String text, String title)
        {
            ErrorCode = code;
            ErrorText = text;
            Title = title;
        }
    }

    /**
     * Process the conversion of a document.
     * @param documentPath   The file system path of the document. The document's path is passed to the server,
     *                       so it must be valid for and accessible by the conversion server.
     * @param options        The processing options
     * @param outputPath     The file system path for storing the result document
     * @return  a ConversionResult object containing error information
     */
    public ConversionResult processConversion(String documentPath, String options, String outputPath)
    {
        ConversionResult result = new ConversionResult();
        process(_handle, documentPath, options, outputPath, result);
        return result;
    }

    /**
     * Create a converter job that can be used to assemble multiple input files.
     * A job is also useful to copy documents to the server for processing and
     * result files back to the caller when a common file system is not available.
     * @return the Job object
     * @throws java.net.SocketException Not connected to a converter service
     */
    public Job createJob() throws java.net.SocketException
    {
        if (_handle == 0)
            throw new java.net.SocketException("Not connected to converter service");
        return new Job(_handle);
    }

    public String getKnownFileExtensions() throws java.net.SocketException
    {
        if (_handle == 0)
            throw new java.net.SocketException("Not connected to converter service");
        return getKnownFileExtensions(_handle);
    }

    /**
     * Definition of error codes
     */
    public static class ERROR
    {
        /** unknown file format */
        public static int O2P_E_UNKFORMAT      =   0x82410C01;
        /** invalid state for requested operation */
        public static int O2P_E_INVALIDOP      =   0x82410C02;
        /** document does not contain any pages */
        public static int O2P_E_NOPAGES        =   0x82410C03;
        /** no 3-Heights(TM) PDF Printer found */
        public static int O2P_E_NOPDFPRINTER   =   0x82410C04;
        /** no screen session available */
        public static int O2P_E_NOSCREENS      =   0x82410C05;
        /** printing timeout experienced */
        public static int O2P_E_PRINTTIMEOUT   =   0x82410C06;
        /** service unavailable */
        public static int O2P_E_SVCUNAVAIL     =   0x82410C07;
        /** some parts of the document could not be processed */
        public static int O2P_W_PARTSMISSING   =   0x02410C08;
        /** PDF/A conversion failed */
        public static int O2P_E_PDFACONVFAIL   =   0x82410C09;
        /** generic error */
        public static int O2P_E_UNKNOWN        =   0x82410C0A;
        /** print application specific error */
        public static int O2P_E_APPERROR       =   0x82410C0B;
        /** PDF/A conversion completed with warnings */
        public static int O2P_W_PDFACONVWARN   =   0x02410C0C;
        /** source document inconsistencies detected */
        public static int O2P_W_SOURCEQUALITY  =   0x02410C0D;
        /** password required */
        public static int O2P_E_PASSWORD       =   0x82410C0E;
        /** content decryption error (mail) */
        public static int O2P_W_DECRYPTERROR   =   0x02410C0E;
        /** OCR error (engine not available) */
        public static int O2P_E_OCR            =   0x82410C0F;
        /** PDF/A conversion failed to reach required compliance level */
        public static int O2P_E_PDFACOMPLIANCE =   0x82410C10;
        /** conversion was successful, result is empty */
        public static int O2P_I_VOIDRESULT     =   0x02410C11;
        /** an unhandled error occurred in a plugin */
        public static int O2P_E_PLUGINERROR    =   0x82410C12;
        /** VBA error in document */
        public static int O2P_E_VBAERROR       =   0x82410C13;
        /** office application was blocked or failed */
        public static int O2P_E_APPBLOCKED     =   0x82410C14;

        /** connection failure */
        public static int REMOTING_CONNECTION_FAILURE = 0x80004005;
        public static int REQUESTED_SERVICE_NOT_FOUND = 0x8013150B;
    }


    /**
     * finalize calls destroyObject in case the application forgot to do so.
     */
    protected void finalize() throws Throwable
    {
        destroyObject();
    }

    private native static long createHandle(String serviceUrl);
    private native static void process(long handle, String path, String options, String outputPath, ConversionResult result);
    private native static String getKnownFileExtensions(long handle);
    private native static void destroyHandle(long handle);
    private long _handle;
    private native static void initialize(String version);
    public final static String VERSION = "2.0.1.0";

    static
    {
        try
        {
            initialize(VERSION);
        }
        catch (UnsatisfiedLinkError ue)
        {
            System.loadLibrary("O2PProxyAPI");
            initialize(VERSION);
        }
    }
}
