/*
 * Decompiled with CFR 0.152.
 */
package net.byteseek.utils;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import net.byteseek.utils.ArgUtils;

public final class ByteUtils {
    private static final String CHAR_BYTE_FORMAT = "'%c'";
    private static final String HEX_BYTE_FORMAT = "%02x";
    private static final byte ASCII_CASE_DIFFERENCE = 32;
    private static final int QUOTE_CHARACTER_VALUE = 39;
    private static final int START_PRINTABLE_ASCII = 32;
    private static final int END_PRINTABLE_ASCII = 126;
    private static final int[] VALID_ALL_BITMASK_SET_SIZES = new int[]{1, 2, 4, 8, 16, 32, 64, 128, 256};
    private static final int[] VALID_ANY_BITMASK_SET_SIZES = new int[]{0, 128, 192, 224, 240, 248, 252, 254, 255};
    private static final Charset ISO_8859_1 = Charset.forName("ISO-8859-1");

    private ByteUtils() {
    }

    public static int countSetBits(byte b) {
        int bits = b & 0xFF;
        int result = bits - (bits >>> 1 & 0x55);
        result = (result >>> 2 & 0x33) + (result & 0x33);
        result = (result >>> 4) + result & 0xF;
        return result;
    }

    public static int countUnsetBits(byte b) {
        return 8 - ByteUtils.countSetBits(b);
    }

    public static int countBytesMatchingAllBits(byte bitmask) {
        return 1 << ByteUtils.countUnsetBits(bitmask);
    }

    public static int countBytesMatchingAnyBit(byte bitmask) {
        return 256 - ByteUtils.countBytesMatchingAllBits(bitmask);
    }

    public static byte[] getBytesMatchingAllBitMask(byte bitMask) {
        int numberOfBytes = ByteUtils.countBytesMatchingAllBits(bitMask);
        byte[] bytes = new byte[numberOfBytes];
        int arrayCount = 0;
        for (int byteIndex = 0; byteIndex < 256; ++byteIndex) {
            if (((byte)byteIndex & bitMask) != bitMask) continue;
            bytes[arrayCount++] = (byte)byteIndex;
        }
        return bytes;
    }

    public static byte[] getBytesNotMatchingAllBitMask(byte bitMask) {
        int numberOfBytes = 256 - ByteUtils.countBytesMatchingAllBits(bitMask);
        byte[] bytes = new byte[numberOfBytes];
        int arrayCount = 0;
        for (int byteIndex = 0; byteIndex < 256; ++byteIndex) {
            if (((byte)byteIndex & bitMask) == bitMask) continue;
            bytes[arrayCount++] = (byte)byteIndex;
        }
        return bytes;
    }

    public static byte[] getBytesMatchingAnyBitMask(byte bitMask) {
        int numberOfBytes = ByteUtils.countBytesMatchingAnyBit(bitMask);
        byte[] bytes = new byte[numberOfBytes];
        int arrayCount = 0;
        for (int byteIndex = 0; byteIndex < 256; ++byteIndex) {
            if (((byte)byteIndex & bitMask) == 0) continue;
            bytes[arrayCount++] = (byte)byteIndex;
        }
        return bytes;
    }

    public static byte[] getBytesNotMatchingAnyBitMask(byte bitMask) {
        int numberOfBytes = 256 - ByteUtils.countBytesMatchingAnyBit(bitMask);
        byte[] bytes = new byte[numberOfBytes];
        int arrayCount = 0;
        for (int byteIndex = 0; byteIndex < 256; ++byteIndex) {
            if (((byte)byteIndex & bitMask) != 0) continue;
            bytes[arrayCount++] = (byte)byteIndex;
        }
        return bytes;
    }

    public static void addBytesMatchingAllBitMask(byte bitMask, Collection<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        for (int byteIndex = 0; byteIndex < 256; ++byteIndex) {
            byte byteValue = (byte)byteIndex;
            if (((byte)byteIndex & bitMask) != bitMask) continue;
            bytes.add(byteValue);
        }
    }

    public static void addAllBytes(Collection<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        for (int i = 0; i < 256; ++i) {
            bytes.add((byte)i);
        }
    }

    public static void addBytesNotMatchingAllBitMask(byte bitMask, Collection<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        for (int byteIndex = 0; byteIndex < 256; ++byteIndex) {
            byte byteValue = (byte)byteIndex;
            if (((byte)byteIndex & bitMask) == bitMask) continue;
            bytes.add(byteValue);
        }
    }

    public static Byte getAllBitMaskForBytes(byte[] bytes) {
        return ByteUtils.getAllBitMaskForBytes(ByteUtils.toSet(bytes));
    }

    public static Set<Byte> toSet(byte[] bytes) {
        ArgUtils.checkNullByteArray(bytes);
        HashSet<Byte> setOfBytes = new HashSet<Byte>((int)((double)bytes.length / 0.75));
        ByteUtils.addAll(bytes, setOfBytes);
        return setOfBytes;
    }

    public static List<Byte> toList(byte[] bytes) {
        ArgUtils.checkNullByteArray(bytes);
        ArrayList<Byte> listOfBytes = new ArrayList<Byte>(bytes.length);
        for (byte b : bytes) {
            listOfBytes.add(b);
        }
        return listOfBytes;
    }

    public static void addAll(byte[] bytes, Collection<Byte> toCollection) {
        ArgUtils.checkNullByteArray(bytes);
        ArgUtils.checkNullCollection(toCollection);
        int size = bytes.length;
        for (int count = 0; count < size; ++count) {
            toCollection.add(bytes[count]);
        }
    }

    public static void addBytes(Collection<Byte> toCollection, byte ... values) {
        ArgUtils.checkNullCollection(toCollection);
        for (byte value : values) {
            toCollection.add(value);
        }
    }

    public static void addStringBytes(String string, Collection<Byte> toCollection) {
        ByteUtils.addAll(ByteUtils.getBytes(string), toCollection);
    }

    public static byte[] getBytes(String string) {
        ArgUtils.checkNullString(string);
        return string.getBytes(ISO_8859_1);
    }

    public static void addCaseInsensitiveStringBytes(String string, Collection<Byte> toCollection) {
        ArgUtils.checkNullCollection(toCollection);
        byte[] byteValues = ByteUtils.getBytes(string);
        for (int charIndex = 0; charIndex < byteValues.length; ++charIndex) {
            byte charAt = byteValues[charIndex];
            if (charAt >= 97 && charAt <= 122) {
                toCollection.add((byte)(charAt - 32));
            } else if (charAt >= 65 && charAt <= 90) {
                toCollection.add((byte)(charAt + 32));
            }
            toCollection.add(charAt);
        }
    }

    public static byte[] toArray(Collection<Byte> collection) {
        ArgUtils.checkNullCollection(collection);
        byte[] result = new byte[collection.size()];
        int position = 0;
        for (Byte b : collection) {
            result[position++] = b;
        }
        return result;
    }

    public static byte[] toArray(byte ... values) {
        return values;
    }

    public static byte[] reverseArray(byte[] array) {
        ArgUtils.checkNullByteArray(array);
        int lastpos = array.length - 1;
        byte[] reversed = new byte[array.length];
        for (int i = 0; i <= lastpos; ++i) {
            reversed[i] = array[lastpos - i];
        }
        return reversed;
    }

    public static byte[] reverseArraySubsequence(byte[] array, int startIndex, int endIndex) {
        ByteUtils.checkBounds(array, startIndex, endIndex);
        int length = endIndex - startIndex;
        int endPos = endIndex - 1;
        byte[] reversed = new byte[length];
        for (int i = 0; i < length; ++i) {
            reversed[i] = array[endPos - i];
        }
        return reversed;
    }

    public static byte[] repeat(int numberOfRepeats, byte[] array) {
        ArgUtils.checkNullByteArray(array);
        ByteUtils.checkNegativeRepeats(numberOfRepeats);
        int repeatLength = array.length;
        int size = repeatLength * numberOfRepeats;
        byte[] repeated = new byte[size];
        for (int repeat = 0; repeat < numberOfRepeats; ++repeat) {
            System.arraycopy(array, 0, repeated, repeat * repeatLength, repeatLength);
        }
        return repeated;
    }

    public static byte[] repeat(int numberOfRepeats, byte[] array, int startIndex, int endIndex) {
        ByteUtils.checkBounds(array, startIndex, endIndex);
        ByteUtils.checkNegativeRepeats(numberOfRepeats);
        int repeatLength = endIndex - startIndex;
        int size = repeatLength * numberOfRepeats;
        byte[] repeated = new byte[size];
        for (int repeat = 0; repeat < numberOfRepeats; ++repeat) {
            System.arraycopy(array, startIndex, repeated, repeat * repeatLength, repeatLength);
        }
        return repeated;
    }

    public static byte[] repeat(byte value, int numberOfRepeats) {
        ByteUtils.checkNegativeRepeats(numberOfRepeats);
        byte[] repeats = new byte[numberOfRepeats];
        Arrays.fill(repeats, value);
        return repeats;
    }

    public static int[] toIntArray(byte[] bytes) {
        ArgUtils.checkNullByteArray(bytes);
        int length = bytes.length;
        int[] integers = new int[length];
        for (int index = 0; index < length; ++index) {
            integers[index] = bytes[index] & 0xFF;
        }
        return integers;
    }

    public static byte[] getAllByteValues() {
        return ByteUtils.getBytesInRange(0, 255);
    }

    public static byte[] getBytesInRange(int from, int to) {
        ByteUtils.checkIntToByteRange(from, to);
        int start = from < to ? from : to;
        int end = from < to ? to : from;
        byte[] range = new byte[end - start + 1];
        int position = 0;
        for (int value = start; value <= end; ++value) {
            range[position++] = (byte)value;
        }
        return range;
    }

    public static void addBytesInRange(int from, int to, Collection<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        ByteUtils.checkIntToByteRange(from, to);
        int start = from < to ? from : to;
        int end = from < to ? to : from;
        for (int value = start; value <= end; ++value) {
            bytes.add((byte)value);
        }
    }

    public static void addInvertedByteValues(byte value, Collection<Byte> bytes) {
        int intValue = value & 0xFF;
        if (intValue > 0) {
            ByteUtils.addBytesInRange(0, intValue - 1, bytes);
        }
        if (intValue < 255) {
            ByteUtils.addBytesInRange(intValue + 1, 255, bytes);
        }
    }

    public static void addBytesNotInRange(int from, int to, Collection<Byte> bytes) {
        int value;
        ArgUtils.checkNullCollection(bytes);
        ByteUtils.checkIntToByteRange(from, to);
        int start = from < to ? from : to;
        int end = from < to ? to : from;
        for (value = 0; value < start; ++value) {
            bytes.add((byte)value);
        }
        for (value = end + 1; value < 256; ++value) {
            bytes.add((byte)value);
        }
    }

    public static Set<Byte> invertedSet(Set<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        int capacity = (int)((double)bytes.size() / 0.75);
        HashSet<Byte> inverted = new HashSet<Byte>(capacity);
        ByteUtils.buildInvertedSet(bytes, inverted);
        return inverted;
    }

    public static boolean inverseOf(Set<Byte> set, Set<Byte> inverseSet) {
        ArgUtils.checkNullCollection(set, "parameter:set");
        ArgUtils.checkNullCollection(inverseSet, "parameter:inverseSet");
        if (set.size() == 256 - inverseSet.size()) {
            boolean setIsSmaller = set.size() < inverseSet.size();
            Set<Byte> needles = setIsSmaller ? set : inverseSet;
            Set<Byte> haystack = setIsSmaller ? inverseSet : set;
            for (Byte needle : needles) {
                if (!haystack.contains(needle)) continue;
                return false;
            }
            return true;
        }
        return false;
    }

    public static Set<Byte> invertedSet(byte value) {
        HashSet<Byte> inverted = new HashSet<Byte>(342);
        for (int i = 0; i < 256; ++i) {
            inverted.add((byte)i);
        }
        inverted.remove(value);
        return inverted;
    }

    public static void buildInvertedSet(Set<Byte> bytes, Set<Byte> invertedSet) {
        ArgUtils.checkNullCollection(bytes, "parameter:bytes");
        ArgUtils.checkNullCollection(invertedSet, "parameter:invertedSet");
        for (int value = 0; value < 256; ++value) {
            if (bytes.contains((byte)value)) continue;
            invertedSet.add((byte)value);
        }
    }

    public static List<Byte> removeIntersection(Set<Byte> firstSet, Set<Byte> secondSet) {
        ArrayList<Byte> bytesRemoved = new ArrayList<Byte>();
        ByteUtils.removeIntersection(firstSet, secondSet, bytesRemoved);
        return bytesRemoved;
    }

    public static void removeIntersection(Set<Byte> firstSet, Set<Byte> secondSet, Collection<Byte> bytesRemoved) {
        ArgUtils.checkNullCollection(firstSet, "parameter:firstSet");
        ArgUtils.checkNullCollection(secondSet, "parameter:secondSet");
        ArgUtils.checkNullCollection(bytesRemoved, "parameter:bytesRemoved");
        Iterator<Byte> byteIterator = firstSet.iterator();
        while (byteIterator.hasNext()) {
            Byte theByte = byteIterator.next();
            if (!secondSet.remove(theByte)) continue;
            bytesRemoved.add(theByte);
            byteIterator.remove();
        }
    }

    public static int floorLogBaseTwo(int i) {
        ByteUtils.checkPositiveInteger(i);
        return 31 - Integer.numberOfLeadingZeros(i);
    }

    public static int ceilLogBaseTwo(int i) {
        ByteUtils.checkPositiveInteger(i);
        return 32 - Integer.numberOfLeadingZeros(i - 1);
    }

    public static boolean isPowerOfTwo(int i) {
        return i > 0 ? (i & i - 1) == 0 : false;
    }

    public static int ceilPowerOfTwo(int i) {
        return 1 << ByteUtils.ceilLogBaseTwo(i);
    }

    public static int nextHighestPowerOfTwo(int i) {
        return Integer.highestOneBit(i) << 1;
    }

    public static Byte getAllBitMaskForBytes(Set<Byte> bytes) {
        byte mask;
        int bitsInCommon;
        ArgUtils.checkNullCollection(bytes);
        int setSize = bytes.size();
        if (setSize == 256) {
            return (byte)0;
        }
        if (Arrays.binarySearch(VALID_ALL_BITMASK_SET_SIZES, setSize) >= 0 && (bitsInCommon = ByteUtils.getBitsInCommon(bytes)) > 0 && setSize == ByteUtils.countBytesMatchingAllBits(mask = (byte)bitsInCommon)) {
            return mask;
        }
        return null;
    }

    public static Byte getAnyBitMaskForBytes(Set<Byte> bytes) {
        byte mask;
        int possibleAnyMask;
        ArgUtils.checkNullCollection(bytes);
        int setSize = bytes.size();
        if (setSize == 0) {
            return (byte)0;
        }
        if (Arrays.binarySearch(VALID_ANY_BITMASK_SET_SIZES, setSize) >= 0 && (possibleAnyMask = ByteUtils.getBitsSetForAllPossibleBytes(bytes)) > 0 && setSize == ByteUtils.countBytesMatchingAnyBit(mask = (byte)possibleAnyMask)) {
            return mask;
        }
        return null;
    }

    public static Byte getAnyBitMaskForBytes(byte[] bytes) {
        return ByteUtils.getAnyBitMaskForBytes(ByteUtils.toSet(bytes));
    }

    public static int getBitsInCommon(Collection<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        if (bytes.isEmpty()) {
            return 0;
        }
        int bitsinCommon = 255;
        for (Byte b : bytes) {
            bitsinCommon &= b.byteValue();
        }
        return bitsinCommon;
    }

    public static int getBitsSetForAllPossibleBytes(Set<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        int bit1 = 0;
        int bit2 = 0;
        int bit3 = 0;
        int bit4 = 0;
        int bit5 = 0;
        int bit6 = 0;
        int bit7 = 0;
        int bit8 = 0;
        for (Byte b : bytes) {
            int value = b & 0xFF;
            bit1 += value & 1;
            bit2 += (value & 2) >> 1;
            bit3 += (value & 4) >> 2;
            bit4 += (value & 8) >> 3;
            bit5 += (value & 0x10) >> 4;
            bit6 += (value & 0x20) >> 5;
            bit7 += (value & 0x40) >> 6;
            bit8 += (value & 0x80) >> 7;
        }
        int anyBitMask = 0;
        if (bit1 == 128) {
            anyBitMask = 1;
        }
        if (bit2 == 128) {
            anyBitMask |= 2;
        }
        if (bit3 == 128) {
            anyBitMask |= 4;
        }
        if (bit4 == 128) {
            anyBitMask |= 8;
        }
        if (bit5 == 128) {
            anyBitMask |= 0x10;
        }
        if (bit6 == 128) {
            anyBitMask |= 0x20;
        }
        if (bit7 == 128) {
            anyBitMask |= 0x40;
        }
        if (bit8 == 128) {
            anyBitMask |= 0x80;
        }
        return anyBitMask;
    }

    public static void addBytesMatchingAnyBitMask(byte bitMask, Collection<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        for (int byteIndex = 1; byteIndex < 256; ++byteIndex) {
            byte byteValue = (byte)byteIndex;
            if ((byteValue & bitMask) == 0) continue;
            bytes.add((byte)byteIndex);
        }
    }

    public static void addBytesNotMatchingAnyBitMask(byte bitMask, Collection<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        for (int byteIndex = 0; byteIndex < 256; ++byteIndex) {
            byte byteValue = (byte)byteIndex;
            if ((byteValue & bitMask) != 0) continue;
            bytes.add((byte)byteIndex);
        }
    }

    public static byte byteFromHex(String hexByte) {
        if (hexByte != null && hexByte.length() == 2) {
            try {
                return (byte)Integer.parseInt(hexByte, 16);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        throw new IllegalArgumentException("Not a valid hex byte [" + hexByte + ']');
    }

    public static String byteToString(boolean prettyPrint, int byteValue) {
        String result = prettyPrint && byteValue >= 32 && byteValue <= 126 && byteValue != 39 ? String.format(CHAR_BYTE_FORMAT, byteValue) : String.format(HEX_BYTE_FORMAT, byteValue);
        return result;
    }

    public static String bytesToString(boolean prettyPrint, byte[] bytes) {
        ArgUtils.checkNullByteArray(bytes);
        return ByteUtils.bytesToString(prettyPrint, bytes, 0, bytes.length);
    }

    public static String bytesToString(boolean prettyPrint, List<Byte> bytes) {
        ArgUtils.checkNullCollection(bytes);
        return ByteUtils.bytesToString(prettyPrint, ByteUtils.toArray(bytes), 0, bytes.size());
    }

    public static String bytesToString(boolean prettyPrint, byte[] bytes, int startIndex, int endIndex) {
        ByteUtils.checkBounds(bytes, startIndex, endIndex);
        int estimatedSize = prettyPrint ? (endIndex - startIndex) * 4 : (endIndex - startIndex) * 2;
        StringBuilder string = new StringBuilder(estimatedSize);
        boolean inString = false;
        boolean firstByte = true;
        for (int byteIndex = startIndex; byteIndex < endIndex; ++byteIndex) {
            int byteValue = 0xFF & bytes[byteIndex];
            if (prettyPrint) {
                if (!firstByte && !inString) {
                    string.append(' ');
                }
                if (byteValue >= 32 && byteValue <= 126 && byteValue != 39) {
                    if (!inString) {
                        string.append('\'');
                    }
                    string.append((char)byteValue);
                    inString = true;
                } else {
                    if (inString) {
                        string.append("' ");
                    }
                    string.append(String.format(HEX_BYTE_FORMAT, byteValue));
                    inString = false;
                }
            } else {
                string.append(String.format(HEX_BYTE_FORMAT, byteValue));
            }
            firstByte = false;
        }
        if (prettyPrint && inString) {
            string.append('\'');
        }
        return string.toString();
    }

    private static void checkNegativeRepeats(int numberOfRepeats) {
        if (numberOfRepeats < 0) {
            throw new IllegalArgumentException("Number of repeats cannot be negative " + numberOfRepeats);
        }
    }

    private static void checkPositiveInteger(int value) {
        if (value <= 0) {
            throw new IllegalArgumentException("The value must be positive " + value);
        }
    }

    private static void checkBounds(byte[] array, int startIndex, int endIndex) {
        ArgUtils.checkNullByteArray(array);
        if (startIndex < 0 || startIndex >= endIndex || endIndex > array.length) {
            throw new IllegalArgumentException("The start index must be between 0 inclusive and the array length exclusive,end index must be greater than the start index and not greater than the length. Array length is " + array.length + " start index is " + startIndex + " end index is " + endIndex);
        }
    }

    private static void checkIntToByteRange(int from, int to) {
        if (from < 0 || from > 255 || to < 0 || to > 255) {
            String message = "The from and to values must be in the range 0 to 255.  Values provided were %d and %d";
            throw new IllegalArgumentException(String.format("The from and to values must be in the range 0 to 255.  Values provided were %d and %d", from, to));
        }
    }
}

