package com.pdftools.converter;

/**
 * Job objects are created by the Converter's createJob method and serve
 * for conversion processing of one or multiple documents.
 * The life time of a job object should be kept short, because the server
 * allocates resource on it't behalf - but only for a limited time span of
 * inactivity.
 */
public class Job
{
    /** Constructor for use by Converter's createJob method
     */
    Job(long convHandle) throws java.net.SocketException
    {
        _handle = createHandle(convHandle);
        if(_handle == 0)
            throw new java.net.SocketException("Job creation failed");
    }

    /**
     * Set job options.
     * @param  options the options string
     */
    public void setOptions(String options)
    {
        setOptions(_handle, options);
    }
    /**
     * set the XMP document metatadata for the output file
     */
    public void setDocMetadata(byte[] xmpPacket)
    {
        setDocMetadata(_handle, xmpPacket);
    }
    /**
     * Initialize the conversion job by specifying the output file path.
     * @param filename  the file system path of the output document
     * @return true on success; use getLastError() to retrieve error information
     */
    public boolean createOutput(String filename)
    {
        return createOutput(_handle, filename);
    }

    /**
     * Convert the specified input document and append it to the job's output
     * @param filename  the file system path of the input document
     * @param options   the document processing options
     * @return true on success; use getLastError() to retrieve error information
     */
    public boolean appendDoc(String filename, String options)
    {
        return appendDoc(_handle, filename, options);
    }

    /**
     * Convert the input document contained in the byte array and append it to the job's output
     * @param docBytes  the byte array containing the input document
     * @param options   the document processing options; consider to specify the ORIGINALNAME setting with
     *                  the appropriate file extension
     * @return true on success
     */
    public boolean appendDoc(byte[] docBytes, String options)
    {
        return appendBytes(_handle, docBytes, options);
    }

    /**
     * Convert the document contained in the docBytes array and append it to the job's output.
     * @param docBytes    the byte array containing the input document
     * @param options     the document processing options (e.g. ORIGINALNAME=filename.docx)
     * @param parameters  a set of String key and byte[] value pairs to be passed along
     *                    (for retrieval by a custom plugin)
     * @return true on success
     */
    public boolean appendDoc(byte[] docBytes, String options, java.util.Dictionary parameters)
    {
        if (parameters == null || parameters.size() == 0)
            return appendDoc(docBytes, options);
        String[] keys = new String[parameters.size()];
        Object[] values = new Object[parameters.size()];
        int n = 0;
        for (java.util.Enumeration e = parameters.keys(); e.hasMoreElements(); )
        {
            String key = (String)e.nextElement();
            Object d = parameters.get(key);
            keys[n] = key;
            values[n] = d;
            n++;
        }
        return appendBytesWithParam(_handle, docBytes, options, keys, values);
    }

    /**
     * Signal the end of the job and close the output document.
     * @return  true on success; use getLastError() to retrieve error information
     */
    public boolean close()
    {
        return close(_handle);
    }

    /** Provides job related error information
     */
    public static class JobError
    {
        public int     Code;
        public String  Text;

        void set(int code, String text)
        {
            Code = code;
            Text = text;
        }
    }

    /**
     * Retrieve error information related to a previous job method call.
     * @return  the JobError object containing a numeric error code and a textual error description
     */
    public JobError getLastError()
    {
        JobError error = new JobError();
        getLastError(_handle, error);
        return error;
    }

    /**
     * Release the resources allocated for the job object.
     */
    public void destroyObject()
    {
        if(_handle != 0)
            destroyHandle(_handle);
        _handle = 0;
    }

    protected void finalize()throws Throwable
    {
        destroyObject();
    }

    private long _handle;
    private native static long createHandle(long convHandle);
    private native static void destroyHandle(long handle);
    private native static void setOptions(long handle, String options);
    private native static void setDocMetadata(long handle, byte[] xmpPacket);
    private native static boolean createOutput(long handle, String filename);
    private native static boolean appendDoc(long handle, String filename, String options);
    private native static boolean appendBytes(long handle, byte[] docBytes, String options);
    private native static boolean appendBytesWithParam(long handle, byte[] docBytes, String options, String[] keys, Object[] values);
    private native static boolean close(long _handle);
    private native static void getLastError(long handle, JobError error);

}
