/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.file;
/**
 *
 */



import java.io.*;
import java.util.*;

import ch.docuteam.tools.out.Logger;
import ch.docuteam.tools.string.StringUtil;

/**
 * @author denis
 *
 */
public class CSVFileReader
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	//	========	Static Final Private	=======================================================

	static protected final Character	DefaultSeparator = ';';
	static protected final Character	QuoteCharacter = '"';

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	========	Instance Public			=======================================================

	//	========	Instance Private		=======================================================

	protected List<List<String>>		list = new ArrayList<List<String>>();
	protected List<String>				header = new ArrayList<String>();

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Constructors Public		=======================================================

	public CSVFileReader(String csvFilePath) throws IOException
	{
		this(csvFilePath, DefaultSeparator);
	}

	public CSVFileReader(String csvFilePath, Character separator) throws IOException
	{
		this(csvFilePath, separator, true);
	}

	public CSVFileReader(String csvFilePath, boolean hasHeaderLine) throws IOException
	{
		this(csvFilePath, DefaultSeparator, true);
	}

	public CSVFileReader(String csvFilePath, Character separator, boolean hasHeaderLine) throws IOException
	{
		Logger.info("Reading file: " + csvFilePath);

		BufferedReader reader = new BufferedReader(new FileReader(csvFilePath));

		try
		{
			boolean is1stLine = true;
			String line;
			do
			{
				line = reader.readLine();

				if (line == null)					break;			//	EOF reached

				line = line.trim();
				if (line.length() == 0)				continue;		//	Skip empty lines
				if (line.startsWith("#"))			continue;		//	Skip comment lines
				if (line.startsWith("//"))			continue;		//	Skip comment lines

				List<String> tokensTrimmed = split(line, separator);

				if (hasHeaderLine && is1stLine)
				{
					is1stLine = false;
					this.header = tokensTrimmed;
				}
				else
				{
					this.list.add(tokensTrimmed);
				}
			}
			while(line != null);
		}
		finally
		{
			reader.close();
		}
	}

	//	========	Constructors Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	static private List<String> split(String line, Character separator)
	{
		//	If the line does NOT contain the quote character, then assume that all occurrences of separator characters are REAL separators:
		if (!line.contains(QuoteCharacter.toString()))
		{
			String[] tokens = line.split(separator.toString());
			List<String> tokensTrimmed = new ArrayList<String>(tokens.length);
			for (String s: tokens)		tokensTrimmed.add(s.trim());
			return tokensTrimmed;
		}

		//	Otherwise: this line might contain embedded separators and masked quotes:
		//	Go character by character through the line:
		boolean isWithinQuotes = false;
		Vector<String> tokens = new Vector<String>(StringUtil.occurrencesOf(line, 1 + separator.toString()));
		StringBuilder tokenBuilder = new StringBuilder();
		for (int i = 0; i < line.length(); i++)
		{
			Character c = line.charAt(i);
			if (c.equals(QuoteCharacter))
			{
				if (isWithinQuotes)
				{
					//	Check for masked quote: is the NEXT character a quote character too?
					if (((Character)line.charAt(i + 1)).equals(QuoteCharacter))
					{
						tokenBuilder.append(c);
						i++;		//	Skip masked quote (NOTE: this is VERY BAD JAVA STYLE!!!)
					}
					else
					{
						//	End of quote:
						tokens.add(tokenBuilder.toString().trim());
						tokenBuilder = new StringBuilder();
						isWithinQuotes = false;
						i++;		//	Skip next separator (NOTE: this is VERY BAD JAVA STYLE!!!)
					}
				}
				else
				{
					//	Begin of quote:
					isWithinQuotes = true;
				}
			}
			else if (c.equals(separator))
			{
				if (isWithinQuotes)
				{
					tokenBuilder.append(c);
				}
				else
				{
					//	End of token:
					tokens.add(tokenBuilder.toString().trim());
					tokenBuilder = new StringBuilder();
				}
			}
			else
			{
				tokenBuilder.append(c);
			}
		}

		//	End of line:
		tokens.add(tokenBuilder.toString().trim());

		return tokens;
	}

	//	========	Instance Public			=======================================================

	//	--------		Accessing			-------------------------------------------------------

	public int size()
	{
		return this.list.size();
	}

	public boolean isEmpty()
	{
		return this.list.isEmpty();
	}


	public List<String> get(int i)
	{
		return this.list.get(i);
	}

	public String get(int row, int column)
	{
		return this.list.get(row).get(column);
	}

	public String get(int row, String columnName)
	{
		return this.get(row, this.indexOfColumn(columnName));
	}


	public List<List<String>> getAll()
	{
		return this.list;
	}

	public List<String> getHeader()
	{
		return this.header;
	}


	public List<String> findFirstRow(int column, String find)
	{
		for(List<String> tokens: this.list)		if (tokens.get(column).equals(find))		return tokens;
		return null;
	}

	public List<String> findFirstRowIgnoreCase(int column, String find)
	{
		for(List<String> tokens: this.list)		if (tokens.get(column).equalsIgnoreCase(find))		return tokens;
		return null;
	}

	public List<String> findFirstRowContaining(int column, String find)
	{
		for(List<String> tokens: this.list)		if (tokens.get(column).contains(find))		return tokens;
		return null;
	}


	public List<String> findFirstRow(String columnName, String find)
	{
		return this.findFirstRow(this.indexOfColumn(columnName), find);
	}

	public List<String> findFirstRowIgnoreCase(String columnName, String find)
	{
		return this.findFirstRowIgnoreCase(this.indexOfColumn(columnName), find);
	}

	public List<String> findFirstRowContaining(String columnName, String find)
	{
		return this.findFirstRowContaining(this.indexOfColumn(columnName), find);
	}



	public List<List<String>> findAllRows(int column, String find)
	{
		List<List<String>> found = new Vector<List<String>>();
		for(List<String> tokens: this.list)		if (tokens.get(column).equals(find))				found.add(tokens);
		return found;
	}

	public List<List<String>> findAllRowsIgnoreCase(int column, String find)
	{
		List<List<String>> found = new Vector<List<String>>();
		for(List<String> tokens: this.list)		if (tokens.get(column).equalsIgnoreCase(find))		found.add(tokens);
		return found;
	}

	public List<List<String>> findAllRowsContaining(int column, String find)
	{
		List<List<String>> found = new Vector<List<String>>();
		for(List<String> tokens: this.list)		if (tokens.get(column).contains(find))				found.add(tokens);
		return found;
	}


	public List<List<String>> findAllRows(String columnName, String find)
	{
		return this.findAllRows(this.indexOfColumn(columnName), find);
	}

	public List<List<String>> findAllRowsIgnoreCase(String columnName, String find)
	{
		return this.findAllRowsIgnoreCase(this.indexOfColumn(columnName), find);
	}

	public List<List<String>> findAllRowsContaining(String columnName, String find)
	{
		return this.findAllRowsContaining(this.indexOfColumn(columnName), find);
	}


	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Interface			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	========	Instance Private		=======================================================

	//	--------		Initializing		-------------------------------------------------------
	//	--------		Accessing			-------------------------------------------------------

	private int indexOfColumn(String columnName)
	{
		int column = this.header.indexOf(columnName);
		if (column == -1)		throw new IndexOutOfBoundsException("ColumnName not found: '" + columnName + "'");
		return column;
	}

	//	--------		Inquiring			-------------------------------------------------------
	//	--------		Business Ops		-------------------------------------------------------
	//	--------		Persistence			-------------------------------------------------------
	//	--------		Support				-------------------------------------------------------
	//	--------		Utilities			-------------------------------------------------------
	//	--------		Debugging			-------------------------------------------------------
	//	---------		Temporary			-------------------------------------------------------

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

}
