/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.os;

import java.awt.Desktop;
import java.io.*;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;

import ch.docuteam.tools.file.exception.FileIsNotADirectoryException;
import ch.docuteam.tools.out.Logger;

/**
 * @author denis
 *
 */
public abstract class SystemProcess
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Final Public		=======================================================

	static public final int		OK = 0;

	//	========	Static Final Private	=======================================================

	//	========	Static Public			=======================================================

	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Main					=======================================================
	//	===========================================================================================

	/**
	 * Just for testing
	 */
	static public void main(String[] args) throws java.lang.Exception
	{
//		//	Mac:
//		openExternally("/Volumes/daten/Ablage/B  Docu/2_Entw/21_Produktentwicklung/507 Bewertungsdatenbank/Versionen/Current/temp/Freaky Wireless Technology.tiff");

		//	Windows:
		ch.docuteam.tools.os.SystemProcess.openExternally("src/ch/docuteam/unittest/files");
		ch.docuteam.tools.os.SystemProcess.openExternally("./src/ch/docuteam/unittest/files/FileFilter/Folder");
		ch.docuteam.tools.os.SystemProcess.openExternally("./src/ch/docuteam/unittest/files/FileFilter/1.txt");
		openExternally("C:/Documents and Settings/user/Desktop/SIPs/DifferentFileTypes/Files/Freaky Wireless Technology.tiff");
		openExternally("C:/Documents and Settings/user/Desktop/SIPs/DifferentFileTypes/Files/Handbuch_Arena.pdf");
		openExternally("http://www.apple.com");
	}

	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	========	Static Initializer		=======================================================

	//	========	Static Public			=======================================================

	static public int execute(String... args) throws FileNotFoundException, IOException, InterruptedException, FileIsNotADirectoryException
	{
		return execute((File)null, args);
	}


	static public int execute(String workingDirectoryPath, String[] args) throws FileNotFoundException, IOException, InterruptedException, FileIsNotADirectoryException
	{
		return execute(new File(workingDirectoryPath), args);
	}


	static public int execute(File workingDirectory, String... args) throws FileNotFoundException, IOException, InterruptedException, FileIsNotADirectoryException
	{
		int error = OK;

		if (args.length == 0)			throw new IndexOutOfBoundsException("No command to execute");

		Logger.debug("Executing command line: " + Arrays.toString(args));

		//	Create process:
		ProcessBuilder processBuilder = new ProcessBuilder(args);
		processBuilder.redirectErrorStream(true);

		if (workingDirectory != null)
		{
			if (!workingDirectory.exists())			throw new FileNotFoundException(workingDirectory.getPath());
			if (!workingDirectory.isDirectory())	throw new FileIsNotADirectoryException(workingDirectory);

			processBuilder.directory(workingDirectory);
		}
		Logger.debug("Working directory: " + processBuilder.directory());

		Process process = null;
		try
		{
			// Start process:
			process = processBuilder.start();
			StreamGobbler outputGobbler = new StreamGobbler(process.getInputStream());
			outputGobbler.start();

			// Wait for the process to finish and check the result:
			error = process.waitFor();
			if (error == 5)		error = OK;			//	Ignore error #5

			// wait for the outputGobbler:
			outputGobbler.join();
		}
		finally
		{
			// close all IO streams:
			try { if (process != null)		process.getInputStream().close(); }
			catch (IOException x){}

			try { if (process != null)		process.getOutputStream().close(); }
			catch (IOException x){}

			try { if (process != null)		process.getErrorStream().close(); }
			catch (IOException x){}

			//	Cleanup all objects thoroughly:
			if (process != null)			process.destroy();
			process = null;
		}

		return error;
	}


	/**
	 * Open files, directories, or URLs with an external program (text editor, file browser, internet browser etc.).
	 * @throws FileIsNotADirectoryException
	 * @throws URISyntaxException
	 * @throws SystemProcessCantLaunchApplicationException
	 */
	static public void openExternally(String fileOrURI) throws IOException, InterruptedException, SystemProcessException, FileIsNotADirectoryException, URISyntaxException, SystemProcessCantLaunchApplicationException
	{
		Logger.debug("Opening externally: " + fileOrURI);

		if (Desktop.isDesktopSupported())
		{
			//	Open the file or URI using Desktop if supported:

			// Distinguish between file and web page: a web page has to start with "http":
			//	Note: I can't use new URI(fileOrURI).getScheme() because if the fileOrURI contains a space character, this will throw an exception!
			if (fileOrURI.startsWith("http"))
			{
				URI uri = new URI(fileOrURI);

				Logger.debug("Opening URI using Desktop: " + uri);

				Desktop.getDesktop().browse(uri);
			}
			else
			{
				File file = new File(fileOrURI);

				Logger.debug("Opening file using Desktop: " + file);

				try
				{
					//	I need this "getCanonicalFile()" otherwise a possibly contained "./" causes windows to not find this file!"
					Desktop.getDesktop().open(file.getCanonicalFile());
				}
				catch (IOException e)
				{
					//	Throw specific exception that can be identified further up in the call stack:
					throw new SystemProcessCantLaunchApplicationException(fileOrURI);
				}
			}
		}
		else if (OperatingSystem.isWindows())
		{
			//	OS = Windows (this should never occur as all modern Windows versions support Desktop):

			//	Open the file or URI using an OS call:
			String[] cmd = new String[]{"cmd", "/c", "\"" + fileOrURI + "\""};
//			String[] cmd = new String[]{"cmd", "/c", "rundll32 url.dll,FileProtocolHandler \"" + filePath + "\""});
													//	This doesn't work any more in Windows7 and Windows8 Server.

			Logger.debug("Executing command line: " + Arrays.toString(cmd));

			int error = execute(cmd);

			Logger.debug("Result: " + error);
			if (error != OK)	throw new SystemProcessException(error);
		}
		else
		{
			//	OS = anything else (this should never occur as all modern UNIX or OS X versions support Desktop):

			//	Open the file or URI using an OS call:
			String[] cmd = new String[]{"open", fileOrURI};

			Logger.debug("Executing command line: " + Arrays.toString(cmd));

			ProcessBuilder processBuilder = new ProcessBuilder(cmd);
			processBuilder.redirectErrorStream(true);
			Process process = processBuilder.start();

			// Wait for the process to finish:
			int error = process.waitFor();

			Logger.debug("Result: " + error);
			if (error == 5)		error = OK;			//	Ignore error #5
			if (error != OK)	throw new SystemProcessException(error);
		}
	}

	//	========	Static Private			=======================================================

	//	===========================================================================================
	//	========	Inner Classes			=======================================================
	//	===========================================================================================

	/**
	 * @author andreas
	 *
	 */
	static public class StreamGobbler extends Thread
	{
		InputStream is;

		StreamGobbler(InputStream is)
		{
			this.is = is;
		}

		@Override
		public void run()
		{
			Logger.debug("Run");

			BufferedReader br = null;

			try
			{
				br = new BufferedReader(new InputStreamReader(this.is));

				String line = null;
				while ((line = br.readLine()) != null)		Logger.info(line);
			}
			catch (IOException x)
			{
				Logger.error(x.getMessage());
			}
			finally
			{
				try { if (br != null)	br.close(); }
				catch (IOException ex){}
			}
		}
	}

}
