/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.out;

import java.awt.Desktop;
import java.io.File;
import java.net.URI;
import java.util.List;
import java.util.Properties;

import javax.activation.DataHandler;
import javax.activation.FileDataSource;
import javax.mail.*;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;


/**
 * @author Andreas Nef, Docuteam GmbH, 2010
 * @author Denis Lemberger, Docuteam GmbH, 2010
 */

public abstract class MailSender
{
	static public boolean sendMessageSMTP(String subject, String body, final Properties props)
	{
		return sendMessageSMTP(props.getProperty("mail.receiver"), subject, body, props);
	}

	static public boolean sendMessageSMTP(String subject, String body, final Properties props, List<File> attachments)
	{
		return sendMessageSMTP(props.getProperty("mail.receiver"), subject, body, props, attachments);
	}

	static public boolean sendMessageSMTP(String receiver, String subject, String body, final Properties props)
	{
		return sendMessageSMTP(receiver, subject, body, props, null);
	}

	static public boolean sendMessageSMTP(String receiver, String subject, String body, final Properties props, List<File> attachments)
	{
		try
		{
			Session session = null;
			if (Boolean.parseBoolean(props.getProperty("mail.smtp.auth"))) {
				session = Session.getDefaultInstance(props,
					new javax.mail.Authenticator()
					{
						@Override
						protected PasswordAuthentication getPasswordAuthentication()
						{
							return new PasswordAuthentication(props.getProperty("mail.smtp.user"), props.getProperty("mail.smtp.pwd"));
						}
					});
			}
			else
			{
				session = Session.getDefaultInstance(props);
			}
			MimeMessage message = new MimeMessage(session);
			message.setSender(new InternetAddress(props.getProperty("mail.smtp.user")));
			for (String recipient: receiver.split(","))
				message.addRecipient(Message.RecipientType.TO, new InternetAddress(recipient));
			message.setSubject(subject);

			if (attachments == null) {
				message.setText(body);
			} else {
				Multipart multipart = new MimeMultipart();
				MimeBodyPart messageBodyPart;
				
				messageBodyPart = new MimeBodyPart();
				messageBodyPart.setText(body);
				multipart.addBodyPart(messageBodyPart);
				
				for (File attachment : attachments) {
					messageBodyPart = new MimeBodyPart();
					messageBodyPart.setDataHandler(new DataHandler(new FileDataSource(attachment)));
					messageBodyPart.setFileName(attachment.getName());
					multipart.addBodyPart(messageBodyPart);
				}
				message.setContent(multipart);
			}

			Transport.send(message, message.getAllRecipients());
		}
		catch (NoSuchProviderException e)
		{
			Logger.warn("SMTP-Transfer did not succeed.", e);
			return false;
		}
		catch (AddressException e)
		{
			Logger.warn("Invalid address, check sender (" + props.getProperty("mail.smtp.user") + ") or receiver (" + receiver + ") information", e);
			return false;
		}
		catch (MessagingException e)
		{
			Logger.warn("SMTP-Server " + props.getProperty("mail.smtp.host") + " ist not available", e);
			return false;
		}

		return true;
	}


	static public boolean sendMessageLocalClient(String subject, String body, Properties props)
	{
		return sendMessageLocalClient(props.getProperty("mail.receiver"), subject, body, props);
	}


	static public boolean sendMessageLocalClient(String receiver, String subject, String body, Properties props)
	{
		Logger.info("Opening local mail editor with mail template...");
		try
		{
			Logger.info(new URI("mailto", receiver + "?SUBJECT=" + subject + "&BODY=" + body, null).toString());
			Desktop.getDesktop().mail(new URI("mailto", receiver + "?SUBJECT=" + subject + "&BODY=" + body, null));
		}
		catch (java.lang.Exception e)
		{
			Logger.warn("Mail could not be sent.", e);
			return false;
		}

		return true;
	}

}