/**
 *	Copyright (C) 2011-2014 Docuteam GmbH
 *
 *	This program is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 3
 *	as published by the Free Software Foundation.
 *	
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package ch.docuteam.tools.string;
/**
 * A convenience wrapper for Date and Calendar formatting
 */


import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 *This abstract class offers static methods for formatting and parsing Dates and Calendars. It is just a wrapper for convenience.
 * <p>
 *The following pattern letters are defined (all other characters from 'A' to 'Z' and from 'a' to 'z' are reserved):
 * <p>
 * <Code>
 * <table>
 *<tr><th>Letter</th><th>Date or Time Component</th><th>Presentation</th><th>Examples</th></tr>
 *<tr><td>	G</td><td>	Era designator				</td><td>Text			</td><td>AD</tr>
 *<tr><td>	y</td><td>	Year						</td><td>Year			</td><td>1996; 96</tr>
 *<tr><td>	M</td><td>	Month in year				</td><td>Month			</td><td>July; Jul; 07</tr>
 *<tr><td>	w</td><td>	Week in year				</td><td>Number			</td><td>27</tr>
 *<tr><td>	W</td><td>	Week in month				</td><td>Number			</td><td>2</tr>
 *<tr><td>	D</td><td>	Day in year					</td><td>Number			</td><td>189</tr>
 *<tr><td>	d</td><td>	Day in month				</td><td>Number			</td><td>10</tr>
 *<tr><td>	F</td><td>	Day of week in month		</td><td>Number			</td><td>2</tr>
 *<tr><td>	E</td><td>	Day in week					</td><td>Text			</td><td>Tuesday; Tue</tr>
 *<tr><td>	a</td><td>	Am/pm marker				</td><td>Text			</td><td>PM</tr>
 *<tr><td>	H</td><td>	Hour in day (0-23)			</td><td>Number			</td><td>0</tr>
 *<tr><td>	k</td><td>	Hour in day (1-24)			</td><td>Number			</td><td>24</tr>
 *<tr><td>	K</td><td>	Hour in am/pm (0-11)		</td><td>Number			</td><td>0</tr>
 *<tr><td>	h</td><td>	Hour in am/pm (1-12)		</td><td>Number			</td><td>12</tr>
 *<tr><td>	m</td><td>	Minute in hour				</td><td>Number			</td><td>30</tr>
 *<tr><td>	s</td><td>	Second in minute			</td><td>Number			</td><td>55</tr>
 *<tr><td>	S</td><td>	Millisecond					</td><td>Number			</td><td>978</tr>
 *<tr><td>	z</td><td>	Time zone					</td><td>General time zone	</td><td>Pacific Standard Time; PST; GMT-08:00</tr>
 *<tr><td>	Z</td><td>	Time zone					</td><td>RFC 822 time zone	</td><td>-0800</tr>
 *</table>
 *</Code>
 *
 * @author denis
 */
public abstract class DateFormatter
{
	//	===========================================================================================
	//	========	Structure				=======================================================
	//	===========================================================================================

	//	========	Static Public			=======================================================

	/**
	 * 	The default pattern, e.g '2010-09-29T12:34:56'
	 */
	static public final String				Default = "yyyy-MM-dd'T'HH:mm:ss";

	/**
	 * 	A purely numerical pattern without milliseconds, can be used for sorting, e.g '20102909123456'
	 */
	static public final String				Numerical = "yyyyMMddHHmmss";
	/**
	 * 	A purely numerical pattern with milliseconds, can be used for sorting, e.g '20102909123456789'
	 */
	static public final String				NumericalMSecs = "yyyyMMddHHmmssSSS";

	/**
	 * 	A short pattern without seconds, e.g '10-09-29 12:34'
	 */
	static public final String				Short = "yy-MM-dd HH:mm";
	/**
	 * 	A medium-sized pattern, e.g '2010-09-29 12:34:56'
	 */
	static public final String				Medium = "yyyy-MM-dd HH:mm:ss";
	/**
	 * 	A long pattern including milliseconds, e.g '2010-09-29 12:34:56.789'
	 */
	static public final String				Long = "yyyy-MM-dd HH:mm:ss.SSS";

	/**
	 * 	A pattern used in german-talking countries, not good for sorting, e.g '29.09.2010 12:34:56'
	 */
	static public final String				German = "dd.MM.yyyy HH:mm:ss";

	//	========	Static Private			=======================================================

	/**
	 *  Date Formatter for the default date format
	 */
	private static DateFormat		DefaultDateFormat = new SimpleDateFormat(Default);

	
	//	===========================================================================================
	//	========	Methods					=======================================================
	//	===========================================================================================

	//	Default Format:
	
	/**
	 * Return the current timestamp in the default date/time format
	 * @return The current Date&Time in default format
	 */
	static public String getCurrentDateTimeString()
	{
		return getDateTimeString(new Date());
	}

	/**
	 * Return the calendar's date-time string in the default date/time format
	 * @return The calendar's Date&Time in default format
	 */
	static public String getDateTimeString(Calendar c)
	{
		return getDateTimeString(c.getTime());
	}

	/**
	 * Return the date's date-time string in the default date/time format
	 * @return The date's Date&Time in default format. Return null if the string can't be parsed.
	 */
	static public String getDateTimeString(Date d)
	{
		return DefaultDateFormat.format(d);
	}

	/**
	 * Return the long's date-time string in the default date/time format
	 * @return The long's Date&Time in default format. Return null if the string can't be parsed.
	 */
	static public String getDateTimeString(long l)
	{
		return DefaultDateFormat.format(new Date(l));
	}

	/**
	 * Parse the string s which is assumed to be in the default date/time format
	 * @return The parsed date. Return null if the string can't be parsed.
	 */
	static public Date parse(String s)
	{
		try
		{
			return DefaultDateFormat.parse(s);
		}
		catch (ParseException e)
		{
			return null;
		}
	}

	
	//	Customized Format:
	
	/**
	 * Return the current timestamp in the given date/time format. The format can also be one of the constants defined in this class.
	 * @return The current Date&Time in the given format
	 */
	static public String getCurrentDateTimeString(String format)
	{
		return getDateTimeString(new Date(), format);
	}

	/**
	 * Return the calendar's date-time string in the given date/time format. The format can also be one of the constants defined in this class.
	 * @return The calendar's Date&Time in the given format
	 */
	static public String getDateTimeString(Calendar c, String format)
	{
		return getDateTimeString(c.getTime(), format);
	}

	/**
	 * Return the date's date-time string in the given date/time format. The format can also be one of the constants defined in this class.
	 * @return The date's Date&Time in the given format
	 */
	static public String getDateTimeString(Date d, String format)
	{
		return new SimpleDateFormat(format).format(d);
	}

	/**
	 * Return the long's date-time string in the given date/time format. The format can also be one of the constants defined in this class.
	 * @return The long's Date&Time in default format. Return null if the string can't be parsed.
	 */
	static public String getDateTimeString(long l, String format)
	{
		return new SimpleDateFormat(format).format(new Date(l));
	}

	/**
	 * Parse the string s which is assumed to be in the given date/time format. The format can also be one of the constants defined in this class.
	 * @return The parsed date. Return null if the string can't be parsed.
	 */
	static public Date parse(String s, String format)
	{
		try
		{
			return new SimpleDateFormat(format).parse(s);
		}
		catch (ParseException e)
		{
			return null;
		}
	}

}
